<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Dispositivos - {{$settings->app_name}}</title>
    <link rel="icon" href="{{ route('minilogo') }}" type="image/x-icon">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.1.1/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="{{ asset('js/tema.js') }}" defer></script>
    <style>
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        .pulse-animation {
            animation: pulse 2s infinite;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .fade-in {
            animation: fadeIn 0.3s ease-out forwards;
        }
        @keyframes slideIn {
            from { transform: translateX(-20px); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        .slide-in > * {
            animation: slideIn 0.3s ease-out forwards;
            opacity: 0;
        }
        .slide-in > *:nth-child(1) { animation-delay: 0.05s; }
        .slide-in > *:nth-child(2) { animation-delay: 0.1s; }
        .slide-in > *:nth-child(3) { animation-delay: 0.15s; }
        .slide-in > *:nth-child(4) { animation-delay: 0.2s; }
        .slide-in > *:nth-child(5) { animation-delay: 0.25s; }
    </style>
</head>
<body class="min-h-screen mb-10 text-white bg-black font-poppins">
    @include('layouts._partials.header')

    <!-- Conteúdo principal -->
    <div class="container px-4 py-20 mx-auto">
        <div class="max-w-4xl mx-auto">
            <!-- Cabeçalho da página -->
            <div class="flex items-center justify-between mb-8">
                <h1 class="text-3xl font-bold">Gerenciar Dispositivos</h1>

                <a href="javascript:history.back()"
                   class="flex items-center px-3 py-1 text-sm transition-colors rounded-full bg-zinc-800 hover:bg-zinc-700">
                    <i class="mr-1 fas fa-arrow-left"></i> Voltar
                </a>
            </div>

            <!-- Alertas -->
            @if(session('success'))
                <div class="p-4 mb-6 text-green-800 bg-green-100 border-l-4 border-green-500 rounded-md fade-in">
                    <div class="flex items-center">
                        <div class="flex-shrink-0">
                            <i class="text-green-600 fas fa-check-circle"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-green-800">{{ session('success') }}</p>
                        </div>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="p-4 mb-6 text-red-800 bg-red-100 border-l-4 border-red-500 rounded-md fade-in">
                    <div class="flex items-center">
                        <div class="flex-shrink-0">
                            <i class="text-red-600 fas fa-exclamation-circle"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-sm font-medium text-red-800">{{ session('error') }}</p>
                        </div>
                    </div>
                </div>
            @endif

            <!-- Informações do limite de dispositivos -->
            <div class="p-6 mb-8 overflow-hidden rounded-lg bg-gradient-to-br from-zinc-800 to-zinc-900">
                <div class="flex items-center gap-4 mb-6">
                    <div class="flex items-center justify-center w-12 h-12 rounded-full bg-primary/20">
                        <i class="text-xl fa-solid fa-tv text-primary"></i>
                    </div>
                    <div>
                        <h2 class="text-xl font-semibold">Limite de Dispositivos</h2>
                        <p class="text-sm text-zinc-400">Seu plano atual permite usar até {{ $deviceLimit ?? 1 }} dispositivo(s)</p>
                    </div>
                </div>

                <div class="mb-6">
                    <div class="flex items-center justify-between mb-1 text-sm">
                        <span>Dispositivos em uso</span>
                        <span>{{ count($devices) }} de {{ $deviceLimit }}</span>
                    </div>
                    <div class="w-full h-3 overflow-hidden rounded-full bg-zinc-700">
                        @php
                            $usedCount = count($devices);
                            $limit = $deviceLimit ?? 1;
                            $usedPercentage = min($usedCount, $limit) / $limit * 100;
                        @endphp
                        <div class="h-3 rounded-full {{ $usedCount >= $limit ? 'bg-red-500' : 'bg-primary' }}"
                             style="width: {{ $usedPercentage }}%; transition: width 0.5s ease-in-out;"></div>
                    </div>
                </div>

                <!-- Status de uso -->
                <div class="p-3 text-sm rounded-lg {{ $usedCount >= $limit ? 'bg-red-500/20 text-red-300' : 'bg-primary/20 text-primary' }}">
                    @if($usedCount >= $limit)
                        <div class="flex items-center">
                            <i class="mr-2 fas fa-exclamation-circle"></i>
                            <span>Limite máximo atingido. Remova um dispositivo para adicionar um novo.</span>
                        </div>
                    @else
                        <div class="flex items-center">
                            <i class="mr-2 fas fa-check-circle"></i>
                            <span>Você ainda pode adicionar {{ $limit - $usedCount }} dispositivo(s).</span>
                        </div>
                    @endif
                </div>
            </div>

            <!-- Lista de dispositivos -->
            <div class="p-6 mb-8 rounded-lg bg-zinc-800">
                <div class="flex items-center justify-between mb-6">
                    <div class="flex items-center">
                        <i class="mr-3 text-xl fa-solid fa-mobile-screen text-primary"></i>
                        <h2 class="text-xl font-semibold">Seus Dispositivos</h2>
                    </div>

                    @if(count($devices) > 1)
                        <form id="form-remove-all" action="{{ route('devices.destroy.all') }}" method="POST">
                            @csrf
                            <button type="button"
                                    onclick="confirmDeleteAll()"
                                    class="flex items-center px-4 py-2 text-sm font-medium transition-colors border rounded-md text-zinc-200 border-zinc-600 hover:bg-red-500 hover:border-red-500 hover:text-white">
                                <i class="mr-2 fa-solid fa-broom"></i>
                                Remover Todos
                            </button>
                        </form>
                    @endif
                </div>

                @if(count($devices) > 0)
                    <div class="space-y-4 slide-in">
                        @foreach($devices as $device)
                            <div class="flex flex-col justify-between p-4 transition-colors rounded-lg md:items-center md:flex-row hover:bg-zinc-700/30 bg-zinc-700/50">
                                <div class="flex items-start md:items-center">
                                    <div class="flex items-center justify-center w-10 h-10 mr-4 rounded-full {{ $device->is_current ? 'bg-primary/0' : 'bg-zinc-600' }}">
                                        @if($device->device_type === 'mobile')
                                            <i class="text-sm fa-solid fa-mobile-screen {{ $device->is_current ? 'text-primary' : '' }}"></i>
                                        @elseif($device->device_type === 'tablet')
                                            <i class="text-sm fa-solid fa-tablet-screen-button {{ $device->is_current ? 'text-primary' : '' }}"></i>
                                        @else
                                            <i class="text-sm fa-solid fa-desktop {{ $device->is_current ? 'text-primary' : '' }}"></i>
                                        @endif
                                    </div>
                                    <div>
                                        <div class="flex items-center gap-2">
                                            <h3 class="text-sm font-semibold">{{ $device->device_name }}</h3>
                                            @if($device->is_current)
                                                <span class="inline-block px-2 py-1 text-[8px] lg:text-sm font-semibold rounded-full text-primary bg-primary/20">
                                                    Este dispositivo
                                                </span>
                                            @endif
                                        </div>
                                        <div class="flex flex-wrap text-xs gap-x-4 gap-y-1 text-zinc-400">
                                            <p>
                                                <i class="mr-1 far fa-clock"></i>
                                                @if($device->last_active)
                                                    {{ $device->last_active->diffForHumans() }}
                                                @else
                                                    Desconhecido
                                                @endif
                                            </p>
                                            @if($device->ip_address)
                                                <p>
                                                    <i class="mr-1 fas fa-network-wired"></i>
                                                    {{ preg_replace('/\.\d+\.\d+$/', '.**.**', $device->ip_address) }}
                                                </p>
                                            @endif
                                            @if($device->created_at)
                                                <p>
                                                    <i class="mr-1 far fa-calendar-plus"></i>
                                                    Registrado em {{ $device->created_at->format('d/m/Y') }}
                                                </p>
                                            @endif
                                        </div>
                                    </div>
                                </div>

                                <div class="flex mt-3 md:mt-0">
                                    @if(!$device->is_current)
                                        <form id="form-remove-{{ $device->id }}" action="{{ route('devices.destroy', $device->id) }}" method="POST">
                                            @csrf
                                            @method('DELETE')
                                            <button type="button"
                                                   onclick="confirmDelete('{{ $device->device_name }}', {{ $device->id }})"
                                                   class="flex items-center px-3 py-2 text-sm font-medium transition-colors bg-transparent border rounded-md hover:bg-red-500 hover:text-white border-zinc-600 hover:border-red-500">
                                                <i class="mr-1 fas fa-trash-alt"></i>
                                                Remover
                                            </button>
                                        </form>
                                    @endif
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="p-8 text-center rounded-lg bg-zinc-700/50">
                        <i class="mb-3 text-3xl fas fa-laptop text-zinc-500"></i>
                        <p class="text-zinc-400">Nenhum dispositivo registrado ainda.</p>
                    </div>
                @endif
            </div>

            <!-- Informações e Dicas -->
            <div class="grid grid-cols-1 gap-6 lg:grid-cols-2">
                <!-- Informações -->
                <div class="p-6 rounded-lg bg-zinc-800">
                    <div class="flex items-center mb-4">
                        <div class="flex items-center justify-center w-8 h-8 mr-3 rounded-full bg-primary/20">
                            <i class="text-sm fa-solid fa-circle-info text-primary"></i>
                        </div>
                        <h2 class="text-lg font-semibold">Como funciona</h2>
                    </div>
                    <ul class="space-y-3 text-sm text-zinc-300">
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-check text-primary"></i>
                            <span>Um dispositivo é contabilizado quando você acessa sua conta em um novo aparelho.</span>
                        </li>
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-check text-primary"></i>
                            <span>O sistema considera um dispositivo como "ativo" se ele foi usado nos últimos 30 dias.</span>
                        </li>
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-check text-primary"></i>
                            <span>Para liberar espaço para um novo dispositivo, remova um dos dispositivos não utilizados.</span>
                        </li>
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-check text-primary"></i>
                            <span>Não é possível remover o dispositivo que você está usando atualmente.</span>
                        </li>
                    </ul>
                </div>

                <!-- Dicas e Recomendações -->
                <div class="p-6 rounded-lg bg-zinc-800">
                    <div class="flex items-center mb-4">
                        <div class="flex items-center justify-center w-8 h-8 mr-3 rounded-full bg-primary/20">
                            <i class="text-sm fa-solid fa-lightbulb text-primary"></i>
                        </div>
                        <h2 class="text-lg font-semibold">Dicas de Segurança</h2>
                    </div>
                    <ul class="space-y-3 text-sm text-zinc-300">
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-shield-alt text-primary"></i>
                            <span>Verifique regularmente quais dispositivos estão acessando sua conta.</span>
                        </li>
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-shield-alt text-primary"></i>
                            <span>Se não reconhecer um dispositivo, remova-o imediatamente e altere sua senha.</span>
                        </li>
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-shield-alt text-primary"></i>
                            <span>Para maior segurança, evite fazer login em dispositivos públicos ou compartilhados.</span>
                        </li>
                        <li class="flex">
                            <i class="w-5 mr-2 fas fa-shield-alt text-primary"></i>
                            <span>Lembre-se sempre de fazer logout após usar a plataforma em dispositivos de terceiros.</span>
                        </li>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal de confirmação -->
    <div id="confirmModal" class="fixed inset-0 z-50 flex items-center justify-center hidden bg-black/80">
        <div class="w-full max-w-md p-6 mx-4 rounded-lg bg-zinc-800 fade-in">
            <div class="mb-4 text-center">
                <i class="text-4xl text-yellow-500 fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmTitle" class="mb-4 text-lg font-medium text-center"></h3>
            <p id="confirmMessage" class="mb-6 text-center text-zinc-300"></p>
            <div class="flex justify-center gap-3">
                <button id="cancelButton" class="px-4 py-2 font-medium transition-colors border rounded-md border-zinc-600 hover:bg-zinc-700">
                    Cancelar
                </button>
                <button id="confirmButton" class="px-4 py-2 font-medium text-white transition-colors bg-red-500 rounded-md hover:bg-red-600">
                    Confirmar
                </button>
            </div>
        </div>
    </div>

    <!-- Fundo com gradiente -->
    <div class="fixed inset-0 -z-10 bg-gradient-to-br from-zinc-900 to-black"></div>

    <script>
        // Sistema de modal de confirmação
        const modal = document.getElementById('confirmModal');
        const confirmTitle = document.getElementById('confirmTitle');
        const confirmMessage = document.getElementById('confirmMessage');
        const confirmButton = document.getElementById('confirmButton');
        const cancelButton = document.getElementById('cancelButton');

        let confirmCallback = null;

        function showModal(title, message, callback) {
            confirmTitle.textContent = title;
            confirmMessage.textContent = message;
            confirmCallback = callback;
            modal.classList.remove('hidden');
            document.body.style.overflow = 'hidden';
        }

        function hideModal() {
            modal.classList.add('hidden');
            document.body.style.overflow = '';
            confirmCallback = null;
        }

        confirmButton.addEventListener('click', () => {
            if (confirmCallback) confirmCallback();
            hideModal();
        });

        cancelButton.addEventListener('click', hideModal);
        modal.addEventListener('click', (e) => {
            if (e.target === modal) hideModal();
        });

        // Funções de confirmação corrigidas
        function confirmDelete(deviceName, deviceId) {
            showModal(
                "Remover dispositivo",
                `Tem certeza que deseja remover "${deviceName}" da sua lista de dispositivos?`,
                () => {
                    // Usar o ID específico do formulário para garantir que o correto seja enviado
                    const form = document.getElementById('form-remove-' + deviceId);
                    if (form) {
                        form.submit();
                    } else {
                        console.error('Formulário não encontrado para o dispositivo:', deviceId);
                    }
                }
            );
        }

        function confirmDeleteAll() {
            showModal(
                "Remover todos os dispositivos",
                "Tem certeza que deseja remover todos os outros dispositivos? Esta ação não pode ser desfeita.",
                () => {
                    const form = document.getElementById('form-remove-all');
                    if (form) {
                        form.submit();
                    } else {
                        console.error('Formulário de remoção em massa não encontrado');
                    }
                }
            );
        }
    </script>
</body>
</html>
