@extends('layouts.admin')

@section('title', 'Gerenciamento de Planos')

@section('content')
<div class="p-6 space-y-6">
    <!-- Cabeçalho com botão de adicionar -->
    <div class="flex flex-col md:flex-row md:items-center md:justify-between">
        <div>
            <h1 class="text-2xl font-bold text-gray-900">Gerenciamento de Planos</h1>
            <p class="mt-1 text-sm text-gray-500">Gerencie os planos disponíveis para assinatura</p>
        </div>
        <button id="newPlanBtn" class="flex items-center px-4 py-2 mt-4 text-white transition-colors bg-indigo-600 rounded-md md:mt-0 hover:bg-indigo-700">
            <svg xmlns="http://www.w3.org/2000/svg" class="w-5 h-5 mr-2" viewBox="0 0 20 20" fill="currentColor">
                <path fill-rule="evenodd" d="M10 5a1 1 0 011 1v3h3a1 1 0 110 2h-3v3a1 1 0 11-2 0v-3H6a1 1 0 110-2h3V6a1 1 0 011-1z" clip-rule="evenodd" />
            </svg>
            Novo Plano
        </button>
    </div>

    <!-- Cards de estatísticas -->
    <div class="grid grid-cols-1 gap-6 sm:grid-cols-2 lg:grid-cols-3">
        <div class="overflow-hidden bg-white rounded-lg shadow">
            <div class="px-4 py-5 sm:p-6">
                <dt class="text-sm font-medium text-gray-500 truncate">Total de Planos</dt>
                <dd class="mt-1 text-3xl font-semibold text-gray-900">{{ $plans->count() }}</dd>
            </div>
        </div>
        <div class="overflow-hidden bg-white rounded-lg shadow">
            <div class="px-4 py-5 sm:p-6">
                <dt class="text-sm font-medium text-gray-500 truncate">Planos Ativos</dt>
                <dd class="mt-1 text-3xl font-semibold text-green-600">{{ $plans->where('is_active', true)->count() }}</dd>
            </div>
        </div>
        <div class="overflow-hidden bg-white rounded-lg shadow">
            <div class="px-4 py-5 sm:p-6">
                <dt class="text-sm font-medium text-gray-500 truncate">Limite de Dispositivos</dt>
                <dd class="mt-1 text-3xl font-semibold text-indigo-600">{{ $plans->sum('device_limit') }}</dd>
            </div>
        </div>
    </div>

    <!-- Tabela de planos -->
    <div class="overflow-hidden bg-white shadow sm:rounded-lg">
        <div class="px-4 py-5 bg-white border-b border-gray-200 sm:px-6">
            <h3 class="text-lg font-medium leading-6 text-gray-900">Planos Disponíveis</h3>
        </div>
        <div class="overflow-x-auto">
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th scope="col" class="px-6 py-3 text-xs font-medium tracking-wider text-left text-gray-500 uppercase">Nome</th>
                        <th scope="col" class="px-6 py-3 text-xs font-medium tracking-wider text-left text-gray-500 uppercase">Preço</th>
                        <th scope="col" class="px-6 py-3 text-xs font-medium tracking-wider text-left text-gray-500 uppercase">Dispositivos</th>
                        <th scope="col" class="px-6 py-3 text-xs font-medium tracking-wider text-left text-gray-500 uppercase">Prioridade</th>
                        <th scope="col" class="px-6 py-3 text-xs font-medium tracking-wider text-left text-gray-500 uppercase">Status</th>
                        <th scope="col" class="px-6 py-3 text-xs font-medium tracking-wider text-right text-gray-500 uppercase">Ações</th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    @forelse($plans as $plan)
                        <tr class="hover:bg-gray-50">
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="flex items-center">
                                    <div>
                                        <div class="text-sm font-medium text-gray-900">{{ $plan->name }}</div>
                                        <div class="text-sm text-gray-500">{{ $plan->slug }}</div>
                                    </div>
                                </div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900">R$ {{ number_format($plan->price, 2, ',', '.') }}</div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900">{{ $plan->device_limit }}</div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <div class="text-sm text-gray-900">{{ $plan->priority }}</div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                @if($plan->is_active)
                                    <span class="inline-flex px-2 py-1 text-xs font-semibold leading-5 text-green-800 bg-green-100 rounded-full">Ativo</span>
                                @else
                                    <span class="inline-flex px-2 py-1 text-xs font-semibold leading-5 text-red-800 bg-red-100 rounded-full">Inativo</span>
                                @endif
                            </td>
                            <td class="px-6 py-4 text-sm font-medium text-right whitespace-nowrap">
                                <button 
                                    class="mr-3 text-indigo-600 edit-plan-btn hover:text-indigo-900" 
                                    data-plan="{{ json_encode($plan) }}">
                                    <span class="sr-only">Editar</span>
                                    <svg class="w-5 h-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                        <path d="M13.586 3.586a2 2 0 112.828 2.828l-.793.793-2.828-2.828.793-.793zM11.379 5.793L3 14.172V17h2.828l8.38-8.379-2.83-2.828z" />
                                    </svg>
                                </button>
                                <button 
                                    class="text-red-600 delete-plan-btn hover:text-red-900" 
                                    data-id="{{ $plan->id }}" 
                                    data-name="{{ $plan->name }}">
                                    <span class="sr-only">Excluir</span>
                                    <svg class="w-5 h-5" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor" aria-hidden="true">
                                        <path fill-rule="evenodd" d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z" clip-rule="evenodd" />
                                    </svg>
                                </button>
                            </td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="6" class="px-6 py-4 text-sm text-center text-gray-500">
                                Nenhum plano encontrado. Clique em "Novo Plano" para adicionar.
                            </td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Modal de criação/edição de plano -->
<div id="planModalBackdrop" class="fixed inset-0 z-40 hidden transition-opacity bg-gray-500 bg-opacity-75"></div>
<div id="planModal" class="fixed inset-0 z-50 hidden overflow-y-auto">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="w-full max-w-2xl overflow-hidden transition-all transform bg-white rounded-lg shadow-xl">
            <form id="planForm" class="bg-white">
                @csrf
                <input type="hidden" id="plan_id" name="id">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 id="modalTitle" class="text-lg font-medium text-gray-900">Novo Plano</h3>
                </div>
                <div class="p-6">
                    <div id="formErrors" class="hidden p-4 mb-4 rounded-md bg-red-50">
                        <div class="flex">
                            <div class="flex-shrink-0">
                                <svg class="w-5 h-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd" />
                                </svg>
                            </div>
                            <div class="ml-3">
                                <h3 class="text-sm font-medium text-red-800">Ocorreram erros ao processar sua solicitação</h3>
                                <div id="errorList" class="mt-2 text-sm text-red-700"></div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="grid grid-cols-2 gap-6">
                        <div class="col-span-2 sm:col-span-1">
                            <label for="name" class="block text-sm font-medium text-gray-700">Nome do Plano</label>
                            <input type="text" name="name" id="name" class="block w-full mt-1 border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <label for="slug" class="block text-sm font-medium text-gray-700">Slug (opcional)</label>
                            <input type="text" name="slug" id="slug" class="block w-full mt-1 border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                            <p class="mt-1 text-xs text-gray-500">Se vazio, será gerado automaticamente</p>
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <label for="price" class="block text-sm font-medium text-gray-700">Preço (R$)</label>
                            <div class="relative mt-1 rounded-md shadow-sm">
                                <div class="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                    <span class="text-gray-500 sm:text-sm">R$</span>
                                </div>
                                <input type="number" name="price" id="price" step="0.01" min="0" class="block w-full pl-10 border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                            </div>
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <label for="device_limit" class="block text-sm font-medium text-gray-700">Limite de Dispositivos</label>
                            <input type="number" name="device_limit" id="device_limit" min="1" class="block w-full mt-1 border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <label for="priority" class="block text-sm font-medium text-gray-700">Prioridade</label>
                            <input type="number" name="priority" id="priority" min="0" value="0" class="block w-full mt-1 border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
                            <p class="mt-1 text-xs text-gray-500">Maior valor = maior prioridade</p>
                        </div>
                        <div class="col-span-2 sm:col-span-1">
                            <div class="flex items-center h-full">
                                <div class="flex items-center">
                                    <input id="is_active" name="is_active" type="checkbox" checked class="w-4 h-4 text-indigo-600 border-gray-300 rounded focus:ring-indigo-500">
                                    <label for="is_active" class="block ml-2 text-sm text-gray-900">Plano Ativo</label>
                                </div>
                            </div>
                        </div>
                        <div class="col-span-2">
                            <label for="description" class="block text-sm font-medium text-gray-700">Descrição</label>
                            <textarea id="description" name="description" rows="3" class="block w-full mt-1 border-gray-300 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"></textarea>
                        </div>
                        <div class="col-span-2">
                            <label class="block text-sm font-medium text-gray-700">Recursos do Plano</label>
                            <div id="featuresContainer" class="mt-2 space-y-2">
                                <!-- Recursos serão adicionados aqui -->
                            </div>
                            <button type="button" id="addFeatureBtn" class="mt-2 inline-flex items-center px-2.5 py-1.5 border border-transparent text-xs font-medium rounded text-indigo-700 bg-indigo-100 hover:bg-indigo-200 focus:outline-none">
                                <svg class="w-4 h-4 mr-1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                                    <path fill-rule="evenodd" d="M10 5a1 1 0 011 1v3h3a1 1 0 110 2h-3v3a1 1 0 11-2 0v-3H6a1 1 0 110-2h3V6a1 1 0 011-1z" clip-rule="evenodd" />
                                </svg>
                                Adicionar Recurso
                            </button>
                        </div>
                    </div>
                </div>
                <div class="flex justify-end px-6 py-4 space-x-3 border-t border-gray-200 bg-gray-50">
                    <button type="button" id="cancelPlanBtn" class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md shadow-sm hover:bg-gray-50 focus:outline-none">
                        Cancelar
                    </button>
                    <button type="submit" class="px-4 py-2 text-sm font-medium text-white bg-indigo-600 border border-transparent rounded-md shadow-sm hover:bg-indigo-700 focus:outline-none">
                        <span id="submitBtnText">Salvar</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Modal de confirmação de exclusão -->
<div id="deleteModalBackdrop" class="fixed inset-0 z-40 hidden transition-opacity bg-gray-500 bg-opacity-75"></div>
<div id="deleteModal" class="fixed inset-0 z-50 hidden overflow-y-auto">
    <div class="flex items-center justify-center min-h-screen p-4">
        <div class="w-full max-w-lg overflow-hidden transition-all transform bg-white rounded-lg shadow-xl">
            <div class="px-6 py-4 border-b border-gray-200">
                <h3 class="text-lg font-medium text-gray-900">Confirmar Exclusão</h3>
            </div>
            <div class="p-6">
                <div class="mb-4">
                    <p class="text-sm text-gray-600">
                        Tem certeza que deseja excluir o plano <strong id="deletePlanName"></strong>?
                    </p>
                    <p class="mt-2 text-sm text-red-600">
                        Esta ação é irreversível e não poderá ser desfeita.
                    </p>
                </div>
            </div>
            <div class="flex justify-end px-6 py-4 space-x-3 border-t border-gray-200 bg-gray-50">
                <button type="button" id="cancelDeleteBtn" class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md shadow-sm hover:bg-gray-50 focus:outline-none">
                    Cancelar
                </button>
                <button type="button" id="confirmDeleteBtn" class="px-4 py-2 text-sm font-medium text-white bg-red-600 border border-transparent rounded-md shadow-sm hover:bg-red-700 focus:outline-none">
                    Excluir
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
// Adicione este log para confirmar que o script foi carregado
console.log('Plan page script loaded');

document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM fully loaded – initializing plan page');
    
    // Estado
    let isEditMode = false;
    let currentPlanId = null;
    
    // Elementos do DOM
    const newPlanBtn = document.getElementById('newPlanBtn');
    const planModalBackdrop = document.getElementById('planModalBackdrop');
    const planModal = document.getElementById('planModal');
    const planForm = document.getElementById('planForm');
    const cancelPlanBtn = document.getElementById('cancelPlanBtn');
    const addFeatureBtn = document.getElementById('addFeatureBtn');
    const featuresContainer = document.getElementById('featuresContainer');
    const modalTitle = document.getElementById('modalTitle');
    const submitBtnText = document.getElementById('submitBtnText');
    const formErrors = document.getElementById('formErrors');
    const errorList = document.getElementById('errorList');
    
    const deleteModalBackdrop = document.getElementById('deleteModalBackdrop');
    const deleteModal = document.getElementById('deleteModal');
    const deletePlanName = document.getElementById('deletePlanName');
    const cancelDeleteBtn = document.getElementById('cancelDeleteBtn');
    const confirmDeleteBtn = document.getElementById('confirmDeleteBtn');

    // Verifica se os elementos principais existem
    if (!newPlanBtn || !planModal || !planForm) {
        console.error('Elementos principais não encontrados. Confira os IDs.');
        return;
    }
    
    // Eventos para botões principais
    newPlanBtn.addEventListener('click', function() {
        console.log('Botão Novo Plano clicado');
        showCreatePlanModal();
    });
    cancelPlanBtn && cancelPlanBtn.addEventListener('click', function() {
        console.log('Botão Cancelar do modal clicado');
        hidePlanModal();
    });
    planForm.addEventListener('submit', function(e) {
        e.preventDefault();
        console.log('Enviando formulário de plano');
        handleFormSubmit();
    });
    addFeatureBtn && addFeatureBtn.addEventListener('click', function() {
        console.log('Botão para adicionar recurso clicado');
        addFeatureField();
    });
    cancelDeleteBtn && cancelDeleteBtn.addEventListener('click', function() {
        console.log('Botão Cancelar exclusão clicado');
        hideDeleteModal();
    });
    confirmDeleteBtn && confirmDeleteBtn.addEventListener('click', function() {
        console.log('Botão Confirmar exclusão clicado');
        deletePlan();
    });
    
    // Eventos para botões da tabela (delegação)
    document.querySelectorAll('.edit-plan-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const planData = JSON.parse(this.dataset.plan);
            console.log('Editando plano:', planData);
            showEditPlanModal(planData);
        });
    });
    document.querySelectorAll('.delete-plan-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            console.log('Clicou em excluir para o plano ID', this.dataset.id);
            showDeleteModal(this.dataset.id, this.dataset.name);
        });
    });
    
    // Funções dos modais
    function showCreatePlanModal() {
        isEditMode = false;
        currentPlanId = null;
        resetForm();
        modalTitle.textContent = 'Novo Plano';
        submitBtnText.textContent = 'Criar Plano';
        addFeatureField(); // Garante ao menos um campo de recurso
        planModalBackdrop.classList.remove('hidden');
        planModal.classList.remove('hidden');
        console.log('Modal de criação aberto');
    }
    
    function showEditPlanModal(plan) {
        isEditMode = true;
        currentPlanId = plan.id;
        resetForm();
        modalTitle.textContent = 'Editar Plano';
        submitBtnText.textContent = 'Atualizar Plano';
        document.getElementById('plan_id').value = plan.id;
        document.getElementById('name').value = plan.name || '';
        document.getElementById('slug').value = plan.slug || '';
        document.getElementById('price').value = plan.price || 0;
        document.getElementById('device_limit').value = plan.device_limit || 1;
        document.getElementById('priority').value = plan.priority || 0;
        document.getElementById('description').value = plan.description || '';
        document.getElementById('is_active').checked = Boolean(plan.is_active);
        if (plan.features && Array.isArray(plan.features)) {
            plan.features.forEach(feature => {
                addFeatureField(feature);
            });
        } else {
            addFeatureField();
        }
        planModalBackdrop.classList.remove('hidden');
        planModal.classList.remove('hidden');
        console.log('Modal de edição aberto para plano ID', plan.id);
    }
    
    function hidePlanModal() {
        planModalBackdrop.classList.add('hidden');
        planModal.classList.add('hidden');
        console.log('Modal de criação/edição fechado');
    }
    
    function showDeleteModal(planId, planName) {
        currentPlanId = planId;
        deletePlanName.textContent = planName;
        deleteModalBackdrop.classList.remove('hidden');
        deleteModal.classList.remove('hidden');
        console.log('Modal de exclusão aberto para plano ID', planId);
    }
    
    function hideDeleteModal() {
        deleteModalBackdrop.classList.add('hidden');
        deleteModal.classList.add('hidden');
        console.log('Modal de exclusão fechado');
    }
    
    function resetForm() {
        planForm.reset();
        featuresContainer.innerHTML = '';
        hideFormErrors();
        console.log('Formulário resetado');
    }
    
    function addFeatureField(value = '') {
        const featureDiv = document.createElement('div');
        featureDiv.className = 'feature-container flex rounded-md shadow-sm';
        featureDiv.innerHTML = `
            <input type="text" name="features[]" value="${value}" placeholder="Ex: Acesso a todo conteúdo" class="flex-1 block border-gray-300 shadow-sm rounded-l-md focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm">
            <button type="button" class="relative inline-flex items-center px-3 py-2 -ml-px text-sm font-medium text-red-700 border border-gray-300 remove-feature-btn bg-red-50 hover:bg-red-100 rounded-r-md">
                <svg xmlns="http://www.w3.org/2000/svg" class="w-4 h-4" viewBox="0 0 20 20" fill="currentColor">
                    <path fill-rule="evenodd" d="M9 2a1 1 0 00-.894.553L7.382 4H4a1 1 0 000 2v10a2 2 0 002 2h8a2 2 0 002-2V6a1 1 0 100-2h-3.382l-.724-1.447A1 1 0 0011 2H9zM7 8a1 1 0 012 0v6a1 1 0 11-2 0V8zm5-1a1 1 0 00-1 1v6a1 1 0 102 0V8a1 1 0 00-1-1z" clip-rule="evenodd" />
                </svg>
            </button>
        `;
        featuresContainer.appendChild(featureDiv);
        console.log('Campo de recurso adicionado');
    }
    
    function hideFormErrors() {
        errorList.innerHTML = '';
        formErrors.classList.add('hidden');
    }
    
    function showFormErrors(errors) {
        errorList.innerHTML = '';
        if (typeof errors === 'string') {
            const p = document.createElement('p');
            p.textContent = errors;
            errorList.appendChild(p);
        } else {
            const ul = document.createElement('ul');
            ul.className = 'list-disc pl-5';
            for (const key in errors) {
                if (errors.hasOwnProperty(key)) {
                    const li = document.createElement('li');
                    li.textContent = errors[key];
                    ul.appendChild(li);
                }
            }
            errorList.appendChild(ul);
        }
        formErrors.classList.remove('hidden');
        console.log('Erros apresentados:', errors);
    }
    
    function handleFormSubmit() {
        hideFormErrors();
        const formData = new FormData(planForm);
        let url, method;
        if (isEditMode) {
            url = `{{ route('admin.plans.update', '') }}/${currentPlanId}`;
            method = 'PUT';
        } else {
            url = `{{ route('admin.plans.store') }}`;
            method = 'POST';
        }
        console.log('Enviando AJAX para', url);
        fetch(url, {
            method: method,
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            console.log('Resposta do servidor:', data);
            if (data.status === 200) {
                showNotification(isEditMode ? 
                    'Plano atualizado com sucesso!' : 'Plano criado com sucesso!', 'success');
                setTimeout(() => window.location.reload(), 1500);
            } else {
                showFormErrors(data.errors || 'Erro na operação');
            }
        })
        .catch(error => {
            console.error('Erro na requisição AJAX:', error);
            showFormErrors('Erro ao comunicar com servidor.');
        });
    }
    
    function deletePlan() {
        if (!currentPlanId) return;
        console.log('Enviando requisição para excluir plano ID:', currentPlanId);
        fetch(`{{ route('admin.plans.destroy', '') }}/${currentPlanId}`, {
            method: 'DELETE',
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            console.log('Resposta de exclusão:', data);
            if (data.status === 200) {
                showNotification('Plano excluído com sucesso!', 'success');
                setTimeout(() => window.location.reload(), 1500);
            } else {
                showNotification(data.message || 'Erro ao excluir plano', 'error');
            }
        })
        .catch(error => {
            console.error('Erro na exclusão via AJAX:', error);
            showNotification('Erro ao comunicar com servidor', 'error');
        });
    }
    
    function showNotification(message, type) {
        if (typeof toastr !== 'undefined') {
            type === 'success' ? toastr.success(message) : toastr.error(message);
        } else {
            alert(message);
        }
    }
});
</script>
@endsection
