@extends('layouts.admin')

@section('content')
<div class="content-wrapper">
    <div class="content-header">
        <div class="container-fluid">
            <div class="mb-2 row">
                <div class="col-sm-6">
                    <h1 class="m-0">Gerenciamento de Dispositivos</h1>
                </div>
                <div class="col-sm-6">
                    <ol class="breadcrumb float-sm-right">
                        <li class="breadcrumb-item"><a href="{{ route('admin') }}">Dashboard</a></li>
                        <li class="breadcrumb-item active">Dispositivos</li>
                    </ol>
                </div>
            </div>
        </div>
    </div>

    <div class="content">
        <div class="container-fluid">
            <div class="row">
                <div class="col-lg-12">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">Dispositivos Registrados</h3>
                            <div class="card-tools">
                                <div class="input-group input-group-sm" style="width: 250px;">
                                    <input type="text" name="table_search" class="float-right form-control" id="deviceSearch" placeholder="Buscar...">
                                    <div class="input-group-append">
                                        <button type="button" class="btn btn-default">
                                            <i class="fas fa-search"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="p-0 card-body table-responsive">
                            <table class="table table-hover text-nowrap" id="devicesTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Usuário</th>
                                        <th>Dispositivo</th>
                                        <th>Tipo</th>
                                        <th>IP</th>
                                        <th>Último acesso</th>
                                        <th>Status</th>
                                        <th>Ações</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($devices as $device)
                                    <tr>
                                        <td>{{ $device->id }}</td>
                                        <td>
                                            <a href="#" class="user-details" data-user-id="{{ $device->user_id }}">
                                                {{ $device->user->name }}
                                            </a>
                                        </td>
                                        <td>{{ $device->device_name }}</td>
                                        <td>
                                            @if($device->device_type === 'mobile')
                                                <span class="badge badge-primary">Celular</span>
                                            @elseif($device->device_type === 'tablet')
                                                <span class="badge badge-info">Tablet</span>
                                            @else
                                                <span class="badge badge-secondary">Desktop</span>
                                            @endif
                                        </td>
                                        <td>{{ $device->ip_address }}</td>
                                        <td>{{ $device->last_active ? $device->last_active->diffForHumans() : 'N/A' }}</td>
                                        <td>
                                            @if($device->is_current)
                                                <span class="badge badge-success">Atual</span>
                                            @else
                                                <span class="badge badge-secondary">Inativo</span>
                                            @endif
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-danger delete-device" data-device-id="{{ $device->id }}" data-device-name="{{ $device->device_name }}">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                        <div class="clearfix card-footer">
                            {{ $devices->links() }}
                        </div>
                    </div>
                </div>
            </div>

            <div class="row">
                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">Usuários com Mais Dispositivos</h3>
                        </div>
                        <div class="card-body">
                            <div class="chart">
                                <canvas id="usersDevicesChart" style="min-height: 250px; height: 250px; max-height: 250px; max-width: 100%;"></canvas>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-md-6">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">Tipos de Dispositivos</h3>
                        </div>
                        <div class="card-body">
                            <div class="chart">
                                <canvas id="deviceTypesChart" style="min-height: 250px; height: 250px; max-height: 250px; max-width: 100%;"></canvas>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Modal para visualizar usuário e seus dispositivos -->
<div class="modal fade" id="userDetailsModal" tabindex="-1" role="dialog" aria-labelledby="userDetailsModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="userDetailsModalLabel">Detalhes do Usuário</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div class="text-center spinner-container">
                    <div class="spinner-border text-primary" role="status">
                        <span class="sr-only">Carregando...</span>
                    </div>
                </div>

                <div class="user-info d-none">
                    <div class="mb-4 row">
                        <div class="col-md-6">
                            <h5 id="userName"></h5>
                            <p id="userEmail" class="text-muted"></p>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="deviceLimit">Limite de Dispositivos</label>
                                <div class="input-group">
                                    <input type="number" class="form-control" id="deviceLimit" min="1" value="1">
                                    <div class="input-group-append">
                                        <button class="btn btn-primary" id="saveDeviceLimit">Salvar</button>
                                    </div>
                                </div>
                                <small class="form-text text-muted">Limite atual: <span id="currentDeviceLimit">1</span></small>
                            </div>
                        </div>
                    </div>

                    <h6>Dispositivos Registrados</h6>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Dispositivo</th>
                                    <th>Tipo</th>
                                    <th>Último Acesso</th>
                                    <th>Status</th>
                                    <th>Ação</th>
                                </tr>
                            </thead>
                            <tbody id="userDevicesTable">
                                <!-- Dispositivos serão carregados dinamicamente -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Fechar</button>
            </div>
        </div>
    </div>
</div>

<!-- Modal de confirmação para exclusão -->
<div class="modal fade" id="deleteDeviceModal" tabindex="-1" role="dialog" aria-labelledby="deleteDeviceModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="deleteDeviceModalLabel">Confirmar Exclusão</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <p>Tem certeza que deseja excluir o dispositivo <strong id="deviceNameToDelete"></strong>?</p>
                <p class="text-danger">Esta ação não pode ser desfeita.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                <button type="button" class="btn btn-danger" id="confirmDeleteDevice">Excluir</button>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    $(document).ready(function() {
        // Busca na tabela
        $('#deviceSearch').on('keyup', function() {
            const value = $(this).val().toLowerCase();
            $("#devicesTable tbody tr").filter(function() {
                $(this).toggle($(this).text().toLowerCase().indexOf(value) > -1)
            });
        });

        // Gráficos
        const usersDevicesCtx = document.getElementById('usersDevicesChart').getContext('2d');
        const deviceTypesCtx = document.getElementById('deviceTypesChart').getContext('2d');

        // Preparar dados para gráfico de usuários com mais dispositivos
        const userLabels = @json($usersWithMostDevices->pluck('name'));
        const deviceCounts = @json($usersWithMostDevices->pluck('devices_count'));

        new Chart(usersDevicesCtx, {
            type: 'bar',
            data: {
                labels: userLabels,
                datasets: [{
                    label: 'Quantidade de Dispositivos',
                    data: deviceCounts,
                    backgroundColor: 'rgba(60, 141, 188, 0.8)',
                    borderColor: 'rgba(60, 141, 188, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                }
            }
        });

        // Preparar dados para gráfico de tipos de dispositivos
        $.ajax({
            url: '/admin/devices/stats',
            method: 'GET',
            success: function(response) {
                const deviceTypes = response.deviceTypes;

                new Chart(deviceTypesCtx, {
                    type: 'pie',
                    data: {
                        labels: ['Celular', 'Tablet', 'Desktop', 'Outros'],
                        datasets: [{
                            data: [
                                deviceTypes.mobile || 0,
                                deviceTypes.tablet || 0,
                                deviceTypes.desktop || 0,
                                deviceTypes.other || 0
                            ],
                            backgroundColor: [
                                'rgba(60, 141, 188, 0.8)',
                                'rgba(255, 193, 7, 0.8)',
                                'rgba(76, 175, 80, 0.8)',
                                'rgba(108, 117, 125, 0.8)'
                            ]
                        }]
                    },
                    options: {
                        responsive: true
                    }
                });
            }
        });

        // Visualizar dispositivos do usuário
        $('.user-details').on('click', function(e) {
            e.preventDefault();
            const userId = $(this).data('user-id');
            $('#userDetailsModal').modal('show');

            // Mostrar spinner e esconder conteúdo
            $('.spinner-container').removeClass('d-none');
            $('.user-info').addClass('d-none');

            // Carregar os dados do usuário e seus dispositivos
            $.ajax({
                url: `/admin/devices/user/${userId}`,
                method: 'GET',
                success: function(response) {
                    if (response.status === 200) {
                        // Preencher dados do usuário
                        $('#userName').text(response.user.name);
                        $('#userEmail').text(response.user.email);
                        $('#deviceLimit').val(response.user.custom_device_limit || response.device_limit);
                        $('#currentDeviceLimit').text(response.device_limit);
                        $('#saveDeviceLimit').data('user-id', userId);

                        // Limpar e preencher tabela de dispositivos
                        $('#userDevicesTable').empty();

                        if (response.devices.length > 0) {
                            response.devices.forEach(device => {
                                let deviceType = 'Desktop';
                                let deviceBadge = 'secondary';

                                if (device.device_type === 'mobile') {
                                    deviceType = 'Celular';
                                    deviceBadge = 'primary';
                                } else if (device.device_type === 'tablet') {
                                    deviceType = 'Tablet';
                                    deviceBadge = 'info';
                                }

                                const lastActive = device.last_active ?
                                    moment(device.last_active).fromNow() : 'N/A';

                                const tr = `
                                    <tr>
                                        <td>${device.device_name || 'Desconhecido'}</td>
                                        <td><span class="badge badge-${deviceBadge}">${deviceType}</span></td>
                                        <td>${lastActive}</td>
                                        <td>${device.is_current ?
                                            '<span class="badge badge-success">Atual</span>' :
                                            '<span class="badge badge-secondary">Inativo</span>'}</td>
                                        <td>
                                            <button class="btn btn-sm btn-danger delete-user-device" data-device-id="${device.id}">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                `;

                                $('#userDevicesTable').append(tr);
                            });
                        } else {
                            $('#userDevicesTable').append(`
                                <tr>
                                    <td colspan="5" class="text-center">Nenhum dispositivo registrado</td>
                                </tr>
                            `);
                        }

                        // Esconder spinner e mostrar conteúdo
                        $('.spinner-container').addClass('d-none');
                        $('.user-info').removeClass('d-none');
                    } else {
                        toastr.error('Erro ao carregar os dados do usuário');
                    }
                },
                error: function() {
                    toastr.error('Erro ao carregar os dados do usuário');
                    $('#userDetailsModal').modal('hide');
                }
            });
        });

        // Salvar limite de dispositivos do usuário
        $('#saveDeviceLimit').on('click', function() {
            const userId = $(this).data('user-id');
            const limit = $('#deviceLimit').val();

            $.ajax({
                url: `/admin/devices/limit/${userId}`,
                method: 'POST',
                data: {
                    device_limit: limit,
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.status === 200) {
                        toastr.success('Limite de dispositivos atualizado com sucesso!');
                        $('#currentDeviceLimit').text(limit);
                    } else {
                        toastr.error(response.message || 'Erro ao atualizar limite de dispositivos');
                    }
                },
                error: function(xhr) {
                    const message = xhr.responseJSON?.message || 'Erro ao atualizar limite de dispositivos';
                    toastr.error(message);
                }
            });
        });

        // Excluir dispositivo do modal do usuário
        $(document).on('click', '.delete-user-device', function() {
            const deviceId = $(this).data('device-id');
            const row = $(this).closest('tr');

            if (confirm('Tem certeza que deseja excluir este dispositivo?')) {
                $.ajax({
                    url: `/admin/devices/${deviceId}`,
                    method: 'DELETE',
                    data: {
                        _token: $('meta[name="csrf-token"]').attr('content')
                    },
                    success: function(response) {
                        if (response.status === 200) {
                            toastr.success('Dispositivo removido com sucesso!');
                            row.fadeOut(300, function() { $(this).remove(); });
                        } else {
                            toastr.error(response.message || 'Erro ao remover dispositivo');
                        }
                    },
                    error: function() {
                        toastr.error('Erro ao remover dispositivo');
                    }
                });
            }
        });

        // Excluir dispositivo da tabela principal
        $('.delete-device').on('click', function() {
            const deviceId = $(this).data('device-id');
            const deviceName = $(this).data('device-name');

            $('#deviceNameToDelete').text(deviceName);
            $('#confirmDeleteDevice').data('device-id', deviceId);
            $('#confirmDeleteDevice').data('row', $(this).closest('tr'));

            $('#deleteDeviceModal').modal('show');
        });

        // Confirmar exclusão de dispositivo
        $('#confirmDeleteDevice').on('click', function() {
            const deviceId = $(this).data('device-id');
            const row = $(this).data('row');

            $.ajax({
                url: `/admin/devices/${deviceId}`,
                method: 'DELETE',
                data: {
                    _token: $('meta[name="csrf-token"]').attr('content')
                },
                success: function(response) {
                    if (response.status === 200) {
                        toastr.success('Dispositivo removido com sucesso!');
                        row.fadeOut(300, function() { $(this).remove(); });
                        $('#deleteDeviceModal').modal('hide');
                    } else {
                        toastr.error(response.message || 'Erro ao remover dispositivo');
                    }
                },
                error: function() {
                    toastr.error('Erro ao remover dispositivo');
                }
            });
        });
    });
</script>
@endsection
