class FavoriteButton {
    constructor(movieId, buttonElement) {
        this.movieId = movieId;
        this.button = buttonElement;
        this.isFavorite = false;

        this.init();
    }

    init() {
        this.checkFavoriteStatus();
        this.attachEventListeners();
    }

    checkFavoriteStatus() {
        fetch(`/movie/isMovieFavorite/${this.movieId}`)
            .then(response => response.json())
            .then(data => {
                this.isFavorite = data.isFavorite === 1;
                this.updateButtonState();
            })
            .catch(error => {
                console.error('Erro ao verificar status de favorito:', error);
            });
    }

    updateButtonState() {
        if (this.isFavorite) {
            this.button.classList.add('active');
            this.button.querySelector('i').classList.replace('far', 'fas');
            this.button.setAttribute('title', 'Remover dos favoritos');
        } else {
            this.button.classList.remove('active');
            this.button.querySelector('i').classList.replace('fas', 'far');
            this.button.setAttribute('title', 'Adicionar aos favoritos');
        }
    }

    toggleFavorite() {
        const url = this.isFavorite
            ? `/movie/removefromfav/${this.movieId}`
            : `/movie/addtofav/${this.movieId}`;

        fetch(url, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            this.isFavorite = data.isFavorite;
            this.updateButtonState();

            // Mostrar mensagem de feedback ao usuário
            this.showToast(data.message);
        })
        .catch(error => {
            console.error('Erro ao alterar status de favorito:', error);
            this.showToast('Erro ao processar sua solicitação. Tente novamente.');
        });
    }

    showToast(message) {
        // Implementação simples de toast
        const toast = document.createElement('div');
        toast.className = 'toast';
        toast.textContent = message;

        document.body.appendChild(toast);

        setTimeout(() => {
            toast.classList.add('show');
            setTimeout(() => {
                toast.classList.remove('show');
                setTimeout(() => {
                    document.body.removeChild(toast);
                }, 300);
            }, 3000);
        }, 100);
    }

    attachEventListeners() {
        this.button.addEventListener('click', (e) => {
            e.preventDefault();
            this.toggleFavorite();
        });
    }
}

// Adicionar estilo para o toast
document.addEventListener('DOMContentLoaded', () => {
    const style = document.createElement('style');
    style.textContent = `
        .toast {
            position: fixed;
            bottom: 20px;
            left: 50%;
            transform: translateX(-50%) translateY(100px);
            background-color: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 10px 20px;
            border-radius: 4px;
            z-index: 1000;
            opacity: 0;
            transition: transform 0.3s ease, opacity 0.3s ease;
        }
        .toast.show {
            transform: translateX(-50%) translateY(0);
            opacity: 1;
        }
    `;
    document.head.appendChild(style);
});

// Exportar para uso global
window.FavoriteButton = FavoriteButton;
