/**
 * Gerenciador de episódios de séries
 */
const EpisodeManager = {
    /**
     * Inicializa o gerenciador de episódios
     */
    init() {
        console.log('[EpisodeManager] Inicializando...');
        this.setupSeasonButtons();
        this.verifyEpisodes();
    },

    /**
     * Configura os botões de temporada
     */
    setupSeasonButtons() {
        const buttons = document.querySelectorAll('.season-button');
        if (buttons.length === 0) {
            console.warn('[EpisodeManager] Nenhum botão de temporada encontrado');
            return;
        }

        console.log(`[EpisodeManager] ${buttons.length} botões de temporada encontrados`);

        buttons.forEach(button => {
            button.addEventListener('click', e => {
                const seasonNumber = e.target.dataset.season;
                this.switchSeason(seasonNumber, buttons);
            });
        });

        // Ativa a primeira temporada por padrão
        if (buttons[0]) {
            this.switchSeason(buttons[0].dataset.season, buttons);
        }
    },

    /**
     * Troca para a temporada selecionada
     */
    switchSeason(seasonNumber, buttons) {
        console.log(`[EpisodeManager] Mudando para temporada ${seasonNumber}`);

        // Atualiza aparência dos botões
        buttons.forEach(btn => {
            btn.classList.remove('bg-primary', 'text-black');
            btn.classList.add('bg-zinc-800', 'text-white', 'hover:bg-zinc-700');
        });

        const activeButton = Array.from(buttons).find(b => b.dataset.season === seasonNumber);
        if (activeButton) {
            activeButton.classList.remove('bg-zinc-800', 'text-white', 'hover:bg-zinc-700');
            activeButton.classList.add('bg-primary', 'text-black');
        }

        // Mostra/esconde as temporadas
        document.querySelectorAll('.season-episodes').forEach(season => {
            season.classList.add('hidden');
        });

        const activeSeason = document.getElementById(`season_${seasonNumber}`);
        if (activeSeason) {
            activeSeason.classList.remove('hidden');
            console.log(`[EpisodeManager] Temporada ${seasonNumber} ativada com ${activeSeason.querySelectorAll('.episode-item').length} episódios`);
        } else {
            console.error(`[EpisodeManager] Temporada ${seasonNumber} não encontrada`);
        }

        // Se o SerieApp estiver disponível, atualiza o estado
        if (typeof window.SerieApp !== 'undefined') {
            window.SerieApp.state.currentSeason = seasonNumber;
        }
    },

    /**
     * Verifica se há episódios carregados
     */
    verifyEpisodes() {
        const seasons = document.querySelectorAll('.season-episodes');
        let totalEpisodes = 0;

        seasons.forEach(season => {
            const episodes = season.querySelectorAll('.episode-item');
            totalEpisodes += episodes.length;

            const seasonId = season.id.replace('season_', '');
            console.log(`[EpisodeManager] Temporada ${seasonId}: ${episodes.length} episódios`);

            if (episodes.length === 0) {
                const container = season.querySelector('[id^="episodes-container-"]');
                if (container) {
                    container.innerHTML = `
                        <div class="p-4 text-center text-gray-500">
                            <i class="fas fa-exclamation-circle mr-2"></i>
                            Nenhum episódio encontrado para esta temporada
                        </div>
                    `;
                }
            }
        });

        console.log(`[EpisodeManager] Total de episódios encontrados: ${totalEpisodes}`);
    }
};

// Inicializa quando o DOM estiver pronto
document.addEventListener('DOMContentLoaded', () => EpisodeManager.init());
