/**
 * Integração do UltraPlayer com a funcionalidade de conteúdo aleatório
 * Este script substitui o JWPlayer pelo UltraPlayer personalizado no modal de conteúdo aleatório
 */

document.addEventListener('DOMContentLoaded', () => {
    console.log('Inicializando integração do UltraPlayer com conteúdo aleatório...');

    // Verifica se já carregou o HLS.js (necessário para streams HLS)
    if (!window.Hls) {
        loadScript('https://cdn.jsdelivr.net/npm/hls.js@latest', () => {
            console.log('HLS.js carregado com sucesso!');
            initRandomContentPlayerIntegration();
        });
    } else {
        initRandomContentPlayerIntegration();
    }
});

// Função para carregar scripts externos dinamicamente
function loadScript(src, callback) {
    const script = document.createElement('script');
    script.src = src;
    script.async = true;

    script.onload = () => {
        console.log(`Script ${src} carregado com sucesso!`);
        if (callback) callback();
    };

    script.onerror = () => {
        console.error(`Erro ao carregar o script: ${src}`);
    };

    document.head.appendChild(script);
}

function initRandomContentPlayerIntegration() {
    // Verificar se a classe UltraPlayer está disponível
    if (typeof window.UltraPlayer === 'undefined') {
        console.error('UltraPlayer não foi carregado corretamente!');
        return;
    }

    // Armazenar a instância atual do player
    window.currentRandomPlayer = null;

    // Substituir a função openFullscreenPlayer
    const originalOpenFullscreenPlayer = window.openFullscreenPlayer;

    window.openFullscreenPlayer = function(videoUrl, posterImage, playerType, title, subtitle) {
        console.log('Abrindo player em tela cheia com UltraPlayer...', {
            videoUrl,
            posterImage,
            playerType
        });

        // Se não houver URL de vídeo, tentar usar a URL armazenada de conteúdo aleatório
        if (!videoUrl && window.currentContent) {
            if (playerType === 'movie' && window.currentContent.movieUrl) {
                videoUrl = window.currentContent.movieUrl;
            } else if (playerType === 'series' && window.currentContent.episodeUrl) {
                videoUrl = window.currentContent.episodeUrl;
            }
        }

        if (!videoUrl) {
            console.error('Nenhuma URL de vídeo fornecida');
            alert('Não foi possível reproduzir o vídeo: URL não encontrada');
            return;
        }

        // Referências aos elementos DOM
        const playerModal = document.getElementById('fullscreen-player-modal');
        const playerContainer = document.getElementById('modal-player-container');
        const closeFullscreenPlayer = document.getElementById('close-fullscreen-player');

        if (!playerModal || !playerContainer) {
            console.error('Modal ou container do player não encontrado');
            return;
        }

        // Exibir o modal
        playerModal.classList.remove('hidden');
        document.body.style.overflow = 'hidden';

        // Limpar container existente
        playerContainer.innerHTML = '';

        try {
            console.log('Criando instância do UltraPlayer com container:', playerContainer);
            console.log(playerContainer.outerHTML);

            // Criar nova instância do player
            const player = new window.UltraPlayer({
                container: playerContainer,
                source: videoUrl,
                poster: posterImage,
                autoplay: true,
                muted: false,
                primaryColor: '#2BF14A',
                title: title || window.currentContent?.movieTitle || window.currentContent?.seriesTitle || 'Reproduzindo',
                subtitle: subtitle || '',
                // Adicionar verificação de premium baseada no tipo de conteúdo
                requiresPremium: window.currentContent?.requiresPremium || false,
                isPremiumUser: function() {
                    // Verifica primeiro a meta tag
                    const userTypeMeta = document.querySelector('meta[name="user-type"]');
                    if (userTypeMeta) {
                        return userTypeMeta.getAttribute('content') === 'premium';
                    }
                    // Se não encontrar a meta, verifica se existe um elemento que indica premium
                    return !!document.querySelector('.premium-user-indicator');
                },
                upgradeUrl: '/pricing' // Ajuste para a URL correta do seu sistema
            });

            // Armazenar referência global para poder acessar depois
            window.currentRandomPlayer = player;

            // Configurar botão de fechar
            if (closeFullscreenPlayer) {
                closeFullscreenPlayer.onclick = function() {
                    try {
                        // Sair do modo de tela cheia se estiver ativo
                        if (player && player.state && player.state.fullscreen) {
                            player.exitFullscreen();
                        }

                        setTimeout(() => {
                            try {
                                // Destruir o player
                                if (window.currentRandomPlayer) {
                                    window.currentRandomPlayer.destroy();
                                    window.currentRandomPlayer = null;
                                }

                                // Esconder o modal
                                playerModal.classList.add('hidden');
                                document.body.style.overflow = '';
                            } catch (e) {
                                console.error('Erro ao limpar player:', e);
                                playerModal.classList.add('hidden');
                                document.body.style.overflow = '';
                            }
                        }, 100);
                    } catch (e) {
                        console.error('Erro ao fechar player:', e);
                        playerModal.classList.add('hidden');
                        document.body.style.overflow = '';
                    }
                };
            }

            // Adicionar tratamento de erro para o player
            try {
                // Usar o método on se disponível, caso contrário usar addEventListener diretamente
                if (typeof player.on === 'function') {
                    player.on('error', function(error) {
                        handlePlayerError(error);
                    });
                } else {
                    // Fallback para eventos nativos do elemento de vídeo
                    const videoElement = player.elements && player.elements.video;
                    if (videoElement) {
                        videoElement.addEventListener('error', function(event) {
                            handlePlayerError(event);
                        });
                    }
                }

                function handlePlayerError(error) {
                    console.error('Erro no player:', error);

                    // Mostrar mensagem de erro amigável
                    const errorOverlay = document.createElement('div');
                    errorOverlay.style = 'position: absolute; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.7); display: flex; align-items: center; justify-content: center; z-index: 100;';
                    errorOverlay.innerHTML = `
                        <div style="text-align: center; color: white; padding: 20px;">
                            <h3 style="font-size: 24px; margin-bottom: 10px;">Erro ao reproduzir vídeo</h3>
                            <p>Tente novamente ou escolha outro conteúdo.</p>
                            <div style="margin-top: 15px;">
                                <button id="retry-video" style="padding: 8px 16px; background: #2BF14A; color: black; border-radius: 4px; margin-right: 10px; cursor: pointer;">Tentar Novamente</button>
                                <button id="close-video-error" style="padding: 8px 16px; background: #555; color: white; border-radius: 4px; cursor: pointer;">Fechar</button>
                            </div>
                        </div>
                    `;

                    playerContainer.appendChild(errorOverlay);

                    // Botão para tentar novamente
                    document.getElementById('retry-video').addEventListener('click', function() {
                        errorOverlay.remove();
                        if (player && typeof player.load === 'function') {
                            player.load();
                        } else if (player && player.elements && player.elements.video) {
                            player.elements.video.load();
                        }
                    });

                    // Botão para fechar
                    document.getElementById('close-video-error').addEventListener('click', function() {
                        errorOverlay.remove();
                        if (closeFullscreenPlayer) {
                            closeFullscreenPlayer.click();
                        }
                    });
                }
            } catch (e) {
                console.warn('Erro ao configurar tratamento de erros do player:', e);
            }

            // Monitorar o sucesso da inicialização
            setTimeout(() => {
                console.log('UltraPlayer para conteúdo aleatório pronto!');
            }, 1000);

            // Tela cheia somente por gesto do usuário
            playerContainer.addEventListener('click', function() {
                try {
                    // Tela cheia somente após interação direta do usuário
                    if (!player.state.fullscreen && document.fullscreenEnabled) {
                        setTimeout(() => player.requestFullscreen(), 200);
                    }
                } catch (e) {
                    console.warn('Não foi possível entrar em tela cheia por clique:', e);
                }
            });

            // Botão dedicado para tela cheia (mais confiável em dispositivos móveis)
            const fullscreenButton = document.createElement('button');
            fullscreenButton.className = 'fs-button';
            fullscreenButton.innerHTML = '<i class="fas fa-expand"></i>';
            fullscreenButton.style = 'position: absolute; bottom: 70px; right: 20px; z-index: 50; background: rgba(0,0,0,0.5); border: none; color: white; padding: 10px; border-radius: 5px; cursor: pointer;';

            fullscreenButton.addEventListener('click', function() {
                try {
                    if (player && !player.state.fullscreen) {
                        player.requestFullscreen();
                    }
                } catch (e) {
                    console.warn('Erro ao tentar tela cheia pelo botão:', e);
                    // Tentar método alternativo
                    try {
                        const element = playerContainer;
                        if (element.requestFullscreen) {
                            element.requestFullscreen();
                        } else if (element.webkitRequestFullscreen) {
                            element.webkitRequestFullscreen();
                        } else if (element.msRequestFullscreen) {
                            element.msRequestFullscreen();
                        }
                    } catch (e2) {
                        console.error('Todos os métodos de tela cheia falharam:', e2);
                    }
                }
            });

            playerContainer.appendChild(fullscreenButton);

        } catch (error) {
            console.error('Erro ao inicializar o UltraPlayer:', error);

            // Fallback para reprodução simples em caso de falha do UltraPlayer
            playerContainer.innerHTML = `
                <div style="width: 100%; height: 100%; display: flex; flex-direction: column;">
                    <video controls autoplay style="width: 100%; height: calc(100% - 40px);">
                        <source src="${videoUrl}" type="video/mp4">
                        Seu navegador não suporta o elemento de vídeo.
                    </video>
                    <div style="padding: 10px; background: black; color: white;">
                        ${title || 'Reproduzindo vídeo'}
                    </div>
                </div>
            `;

            // Certificar-se de que o botão de fechar ainda funcione
            if (closeFullscreenPlayer) {
                closeFullscreenPlayer.onclick = function() {
                    playerModal.classList.add('hidden');
                    playerContainer.innerHTML = '';
                    document.body.style.overflow = '';
                };
            }
        }
    };

    console.log('Integração do UltraPlayer com conteúdo aleatório concluída!');
}
