/**
 * Script de integração do JWPlayer com a plataforma
 * Este script gerencia a reprodução de vídeos e a integração com outros componentes
 */

// Verifica se o JWPlayer está disponível e configurado corretamente
if (typeof jwplayer === 'undefined') {
    console.error('JWPlayer não está disponível. Verifique se o script foi carregado corretamente.');
}

// Gerenciador de player para centralizar funções relacionadas ao JWPlayer
window.PlayerManager = {
    // Inicializa o player com configurações padrão
    initialize: function(containerId, videoUrl, options = {}) {
        // Verifica se o container existe
        if (!document.getElementById(containerId)) {
            console.error(`Container #${containerId} não encontrado.`);
            return null;
        }

        // Configurações padrão
        const defaultConfig = {
            file: videoUrl,
            image: options.posterImage || '',
            width: "100%",
            aspectratio: "16/9",
            primary: "html5",
            hlshtml: true,
            controls: true,
            autostart: true,
            pipIcon: true,
            playbackRateControls: [0.5, 1, 1.25, 1.5, 2],
            volume: parseInt(localStorage.getItem('playerVolume') || 80),
            abouttext: "UltraFlix Player",
            cast: {},
            skin: {
                name: "netflix"
            }
        };

        // Aplica configurações adicionais, se fornecidas
        const config = { ...defaultConfig, ...options.playerConfig };

        // Inicializa o player
        const player = jwplayer(containerId).setup(config);

        // Expõe o player globalmente para depuração
        window.jwPlayerInstance = player;

        return player;
    },

    // Configura eventos padrão do player
    setupEvents: function(player, options = {}) {
        if (!player) return;

        // Evento para quando o player estiver pronto
        player.on('ready', function() {
            if (options.onReady) options.onReady(player);
        });

        // Salva o volume no localStorage quando alterado
        player.on('volume', function(event) {
            localStorage.setItem('playerVolume', event.volume);
        });

        // Evento para tratamento de erros
        player.on('error', function(error) {
            console.error('Erro no player:', error);
            if (options.onError) options.onError(error);
        });

        // Evento para quando o vídeo for concluído
        player.on('complete', function() {
            if (options.onComplete) options.onComplete(player);
        });

        // Eventos para rastrear tempo de reprodução
        player.on('time', function(e) {
            if (options.onTime) options.onTime(e, player);
        });
    },

    // Destrói o player e limpa recursos
    destroy: function(player) {
        if (player && typeof player.remove === 'function') {
            player.remove();
        }
    }
};

// Adicionar referências globais para informações do conteúdo atual
window.PlayerManager.currentType = null;
window.PlayerManager.currentContentId = null;
window.PlayerManager.currentEpisodeId = null;
window.PlayerManager.currentEpisodeData = null;

// Função para mostrar o player de vídeo em modal
window.showVideoPlayer = function(videoLink, supportedHosts, contentId, contentData) {
    console.log('Inicializando player para:', videoLink, contentId);

    // Remover qualquer player existente
    if (document.getElementById('videoPlayerModal')) {
        document.getElementById('videoPlayerModal').remove();
    }

    // Fechar modal de servidores se estiver aberto
    if (document.getElementById('serverModal')) {
        document.getElementById('serverModal').style.display = 'none';
    }

    // Criar o modal para o player
    const modal = document.createElement('div');
    modal.id = 'videoPlayerModal';
    modal.className = 'player-modal';
    modal.innerHTML = `
        <div class="player-modal-content">
            <div id="jwplayer-container"></div>
            <button class="player-modal-close">&times;</button>
        </div>
    `;

    // Adicionar ao body
    document.body.appendChild(modal);
    document.body.style.overflow = 'hidden';

    // Configurar o botão de fechar
    modal.querySelector('.player-modal-close').addEventListener('click', function() {
        closeVideoPlayer();
    });

    // Determinar tipo de conteúdo (filme ou episódio)
    const isEpisode = contentData && (contentData.episode_number || contentData.season_number);

    // Obter posição salva no localStorage ou via API
    let startPosition = 0;

    // Para filmes usamos callback em função assíncrona
    function initializePlayerWithPosition(position) {
        console.log(`Inicializando player com posição: ${position}s`);

        // CORRIGIDO: Configuração no nível superior
        const playerConfig = {
            file: videoLink,
            image: contentData?.still_path || '',
            width: "100%",
            aspectratio: "16/9",
            primary: "html5",
            hlshtml: true,
            controls: true,
            autostart: true,
            pipIcon: true,
            playbackRateControls: [0.5, 1, 1.25, 1.5, 2],
            volume: parseInt(localStorage.getItem('playerVolume') || 80),
            abouttext: "UltraFlix Player",
            starttime: position, // AQUI É A CORREÇÃO - starttime no nível superior
            cast: {},
            skin: {
                name: "netflix"
            }
        };

        // Inicializar o player com a configuração corrigida
        const player = jwplayer('jwplayer-container').setup(playerConfig);

        // Log de debug para confirmar
        console.log('Player configurado com starttime:', position);
        console.log('Configuração completa:', playerConfig);

        // Expõe o player globalmente para depuração
        window.jwPlayerInstance = player;

        // Salvar progresso a cada 15 segundos
        player.on('time', function(e) {
            const currentPosition = Math.floor(e.position);

            if (currentPosition % 15 === 0 && currentPosition > 0) {
                // Salvar no localStorage
                if (contentId) {
                    localStorage.setItem(`videoPosition-${contentId}`, e.position);
                }

                // Se for um episódio, salvar progresso no servidor
                if (isEpisode) {
                    saveEpisodeProgress(contentId, e.position, e.duration, contentData);
                } else {
                    saveMovieProgress(contentId, e.position, e.duration);
                }
            }
        });

        // Ao completar o vídeo
        player.on('complete', function() {
            if (contentId) {
                localStorage.removeItem(`videoPosition-${contentId}`);
            }

            if (isEpisode) {
                markEpisodeAsWatched(contentId, contentData);
            } else {
                markMovieAsWatched(contentId);
            }
        });

        // Ativar fullscreen após um breve delay
        setTimeout(function() {
            try {
                player.setFullscreen(true);
            } catch (e) {
                console.warn('Não foi possível ativar fullscreen automaticamente:', e);
            }
        }, 1000);

        // Mostrar notificação de continuação se necessário
        if (startPosition > 30) {
            player.once('play', function() {
                if (player.getContainer()) {
                    const notification = document.createElement('div');
                    notification.className = 'player-notification';
                    notification.style.cssText = `
                        position: absolute;
                        bottom: 70px;
                        left: 20px;
                        background-color: rgba(0, 0, 0, 0.7);
                        color: white;
                        padding: 10px 15px;
                        border-radius: 4px;
                        z-index: 10;
                        opacity: 0;
                        transition: opacity 0.3s ease;
                    `;
                    notification.textContent = `Continuando de ${formatTime(startPosition)}`;
                    player.getContainer().appendChild(notification);

                    // Fade in
                    setTimeout(() => {
                        notification.style.opacity = '1';
                    }, 10);

                    // Fade out e remoção após 3 segundos
                    setTimeout(() => {
                        notification.style.opacity = '0';
                        setTimeout(() => notification.remove(), 300);
                    }, 3000);
                }
            });
        }
    }

    // Verificar se temos um ID de conteúdo e se é um episódio
    if (contentId) {
        // Se for um episódio, buscar a posição pela API
        if (isEpisode && contentData && contentData.serie_id) {
            const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';

            fetch(`${baseUrl}/continue-watching/get-position?item_id=${contentData.serie_id}&type=serie&episode_id=${contentId}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                }
            })
            .then(response => response.json())
            .then(data => {
                console.log('Posição recuperada do servidor:', data);

                // Verificar se temos uma posição salva e não está perto do final
                if (data && data.position > 0 && !data.is_near_end) {
                    initializePlayerWithPosition(data.position);
                } else {
                    initializePlayerWithPosition(0);
                }
            })
            .catch(err => {
                console.error('Erro ao buscar posição:', err);
                initializePlayerWithPosition(0);
            });
        } else {
            // Para filmes ou fallback, usar localStorage
            const savedPosition = localStorage.getItem(`videoPosition-${contentId}`);
            if (savedPosition) {
                startPosition = parseFloat(savedPosition);
            }

            initializePlayerWithPosition(startPosition);
        }
    } else {
        // Se não tiver ID, iniciar do começo
        initializePlayerWithPosition(0);
    }
};

// Função para salvar progresso do filme
function saveMovieProgress(movieId, position, duration) {
    if (!movieId) return;

    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';

    fetch(`${baseUrl}/continue-watching`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({
            item_id: movieId,
            type: 'movie',
            position: position,
            duration: duration
        })
    })
    .catch(error => {
        console.warn('Erro ao salvar progresso do filme:', error);
    });
}

// Função para salvar progresso do episódio
function saveEpisodeProgress(episodeId, position, duration, episodeData) {
    if (!episodeId || position <= 0 || !episodeData || !episodeData.serie_id) return;

    console.log('Salvando progresso do episódio:', episodeId, position, duration);

    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';

    fetch(`${baseUrl}/continue-watching`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify({
            item_id: episodeData.serie_id,
            type: 'serie',
            episode_id: episodeId,
            position: position,
            duration: duration,
            season_number: episodeData.season_number || 1,
            episode_number: episodeData.episode_number || 1
        })
    })
    .then(response => response.json())
    .then(data => {
        console.log('Progresso salvo com sucesso:', data);
    })
    .catch(error => {
        console.error('Erro ao salvar progresso do episódio:', error);
    });
}

// Funções para marcar conteúdo como assistido
function markMovieAsWatched(movieId) {
    saveMovieProgress(movieId, 0, 0); // Position 0 indica conclusão
}

function markEpisodeAsWatched(episodeId, episodeData) {
    saveEpisodeProgress(episodeId, 0, 0, episodeData); // Position 0 indica conclusão
}

// Implementar uma função específica para reproduzir episódios de série
window.playEpisode = function(videoLink, episodeId, episodeData) {
    console.log('Iniciando reprodução de episódio:', episodeId, episodeData);

    // Verificar se episodeData contém o ID da série
    if (!episodeData.serie_id) {
        // Se não tiver, tentar obter da meta tag na página
        const serieId = document.querySelector('meta[name="serie-id"]')?.content;
        if (serieId) {
            episodeData.serie_id = parseInt(serieId);
        } else {
            console.error('ID da série não encontrado, não será possível salvar progresso');
        }
    }

    // Armazenar referências do conteúdo atual
    window.PlayerManager.currentType = 'serie';
    window.PlayerManager.currentContentId = episodeData.serie_id;
    window.PlayerManager.currentEpisodeId = episodeId;
    window.PlayerManager.currentEpisodeData = episodeData;

    // Fechar servidor modal se estiver aberto
    if (document.getElementById('serverModal')) {
        document.getElementById('serverModal').style.display = 'none';
    }

    // Se não existir um modal de player, criar um
    if (!document.getElementById('videoPlayerModal')) {
        const modal = document.createElement('div');
        modal.id = 'videoPlayerModal';
        modal.className = 'player-modal';
        modal.innerHTML = `
            <div class="player-modal-content">
                <div id="jwplayer-container"></div>
                <button class="player-modal-close">&times;</button>
            </div>
        `;
        document.body.appendChild(modal);

        // Configurar botão de fechar
        modal.querySelector('.player-modal-close').addEventListener('click', function() {
            closeVideoPlayer();
        });
    } else {
        document.getElementById('videoPlayerModal').style.display = 'block';
    }

    // Usar o sistema centralizado para buscar a posição
    let positionPromise;

    // Verificar se temos o novo sistema centralizado
    if (window.VideoPositionSystem) {
        positionPromise = window.VideoPositionSystem.fetchPosition(episodeId, episodeData.serie_id);
    } else {
        // Fallback para o método antigo de buscar posição
        positionPromise = new Promise((resolve) => {
            const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';

            fetch(`${baseUrl}/continue-watching/get-position?item_id=${episodeData.serie_id}&type=serie&episode_id=${episodeId}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                }
            })
            .then(response => response.json())
            .then(data => resolve(data.position || 0))
            .catch(err => {
                console.error('Erro ao buscar posição:', err);
                resolve(0);
            });
        });
    }

    // Criar o player com a posição obtida
    positionPromise.then(startPosition => {
        console.log(`Configurando player com posição inicial: ${startPosition}s`);

        // Configuração do player com starttime no nível superior
        const playerConfig = {
            file: videoLink,
            image: episodeData.still_path || '',
            width: "100%",
            aspectratio: "16/9",
            primary: "html5",
            hlshtml: true,
            controls: true,
            autostart: true,
            pipIcon: true,
            playbackRateControls: [0.5, 1, 1.25, 1.5, 2],
            volume: parseInt(localStorage.getItem('playerVolume') || 80),
            abouttext: "UltraFlix Player",
            starttime: startPosition,
            title: `${episodeData.name} - T${episodeData.season_number || 1}:E${episodeData.episode_number || 1}`,
            cast: {},
            skin: {
                name: "netflix"
            }
        };

        // Inicializar o player com a configuração
        const player = jwplayer('jwplayer-container').setup(playerConfig);

        // Log para debug
        console.log('Player configurado com starttime:', startPosition);

        // Expõe o player globalmente para depuração
        window.jwPlayerInstance = player;

        // Configurar eventos do player para melhor controle
        player.on('ready', function() {
            console.log('Player pronto, verificando posição inicial:', startPosition);

            // Garantir que a posição inicial seja aplicada
            if (startPosition > 0) {
                // Permitir que o player inicie primeiro
                setTimeout(() => {
                    player.once('play', () => {
                        // Verificar se estamos no início do vídeo, se sim, aplicar posição salva
                        if (player.getPosition() < 3) {
                            console.log(`Ajustando para posição salva: ${startPosition}s`);
                            player.seek(startPosition);

                            // Mostrar notificação após o seek
                            showNotification(player, startPosition);
                        }
                    });
                }, 1000);
            }
        });

        // Salvar progresso periodicamente
        player.on('time', function(e) {
            // Salvar posição a cada 5 segundos
            if (Math.floor(e.position) % 5 === 0 && e.position > 0) {
                saveEpisodeProgress(episodeId, e.position, e.duration, episodeData);
            }
        });

        player.on('complete', function() {
            console.log('Episódio concluído');
            saveEpisodeProgress(episodeId, 0, player.getDuration(), episodeData);
        });

        // Ativar fullscreen após pequeno delay
        setTimeout(function() {
            try {
                player.setFullscreen(true);
            } catch (e) {
                console.warn('Não foi possível ativar fullscreen:', e);
            }
        }, 1000);
    });
};

// Função auxiliar para mostrar notificação de continuação
function showNotification(player, position) {
    if (!player || !player.getContainer()) return;

    const container = player.getContainer();
    const notification = document.createElement('div');
    notification.className = 'player-notification';
    notification.style.cssText = `
        position: absolute;
        bottom: 70px;
        left: 20px;
        background-color: rgba(0, 0, 0, 0.7);
        color: white;
        padding: 10px 15px;
        border-radius: 4px;
        z-index: 10;
        opacity: 0;
        transition: opacity 0.3s ease;
    `;
    notification.textContent = `Continuando de ${formatTime(position)}`;
    container.appendChild(notification);

    // Fade in/out
    setTimeout(() => {
        notification.style.opacity = '1';
        setTimeout(() => {
            notification.style.opacity = '0';
            setTimeout(() => notification.remove(), 300);
        }, 3000);
    }, 10);
}

// Função para salvar progresso do episódio - versão melhorada
function saveEpisodeProgress(episodeId, position, duration, episodeData) {
    // Verificações completas para garantir que temos todos os dados necessários
    if (!episodeId || !episodeData) {
        console.error('Dados insuficientes para salvar progresso:', { episodeId, episodeData });
        return;
    }

    // Garantir que temos o ID da série
    const serieId = episodeData.serie_id || document.querySelector('meta[name="serie-id"]')?.content;
    if (!serieId) {
        console.error('ID da série não encontrado, não é possível salvar progresso');
        return;
    }

    // Para debug
    console.log('Salvando progresso do episódio:', {
        episodeId,
        serieId,
        position,
        duration,
        seasonNumber: episodeData.season_number || 1,
        episodeNumber: episodeData.episode_number || 1
    });

    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';

    // CORREÇÃO: Usar a URL completa com base na meta tag
    const continueWatchingUrl = `${baseUrl}/continue-watching`;

    // Montar payload com todos os dados necessários
    const payload = {
        item_id: serieId,
        type: 'serie',
        episode_id: episodeId,
        position: position,
        duration: duration,
        season_number: episodeData.season_number || 1,
        episode_number: episodeData.episode_number || 1
    };

    // Também salvar no sistema centralizado de cache, se disponível
    if (window.VideoPositionSystem) {
        window.VideoPositionSystem.cache[episodeId] = {
            position: position,
            duration: duration,
            timestamp: Date.now(),
            season: episodeData.season_number || 1,
            episode: episodeData.episode_number || 1
        };

        // Backup no localStorage
        window.VideoPositionSystem.saveToLocalStorage(episodeId, position);
    }

    // Enviar para o servidor
    fetch(continueWatchingUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': csrfToken
        },
        body: JSON.stringify(payload)
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! Status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        console.log('Progresso salvo com sucesso:', data);
    })
    .catch(error => {
        console.error('Erro ao salvar progresso do episódio:', error);
    });
}

// Função para formatar tempo em MM:SS
function formatTime(seconds) {
    if (isNaN(seconds) || !isFinite(seconds)) return '00:00';
    seconds = Math.floor(seconds);
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs < 10 ? '0' : ''}${secs}`;
}

// Função para fechar o player
window.closeVideoPlayer = function() {
    try {
        const player = jwplayer('jwplayer-container');
        if (player && typeof player.remove === 'function') {
            player.remove();
        }

        if (document.getElementById('videoPlayerModal')) {
            document.getElementById('videoPlayerModal').style.display = 'none';
        }
    } catch (e) {
        console.error('Erro ao fechar o player:', e);
    }
};

// Integração com seleção de servidor
if (typeof window.openServerSelection !== 'function') {
    // Implementação de compatibilidade para sites que não têm a função
    window.openServerSelection = function(data) {
        console.log('Abrindo servidor para item:', data);
        // Implementar lógica de fallback aqui se necessário
    };
}

// Modificar a função para salvar progresso de visualização
function saveWatchProgress(videoType, itemId, position, duration, episodeId, seasonNumber, episodeNumber) {
    // Detectar o tipo de conteúdo baseado na URL ou em metadados
    if (!videoType) {
        if (window.location.pathname.includes('filme')) {
            videoType = 'movie';
        } else if (window.location.pathname.includes('serie')) {
            videoType = 'serie';
        } else if (window.location.pathname.includes('anime')) {
            videoType = 'anime';
        }
    }

    // Default para movie se não conseguir detectar
    videoType = videoType || 'movie';

    // Log detalhado para debug
    console.log(`Salvando progresso - Tipo: ${videoType}, ID: ${itemId}, Episódio: ${episodeId || 'N/A'}`);

    // Confirmar que temos pelo menos ID do item e tipo
    if (!itemId) {
        console.error('saveWatchProgress: item_id é necessário');
        return;
    }

    // Preparar dados para envio
    const data = {
        item_id: itemId,
        type: videoType,
        position: position,
        duration: duration
    };

    // Adicionar informações de episódio para séries ou animes
    if ((videoType === 'serie' || videoType === 'anime') && episodeId) {
        data.episode_id = episodeId;

        if (seasonNumber) {
            data.season_number = seasonNumber;
        }

        if (episodeNumber) {
            data.episode_number = episodeNumber;
        }
    }

    // Log para debug
    console.log('Enviando dados de progresso:', data);

    // Obtenha a URL base do meta tag
    const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';

    // CORREÇÃO: Usar a URL completa com base na meta tag
    const continueWatchingUrl = `${baseUrl}/continue-watching`;

    // Enviar dados para o endpoint
    fetch(continueWatchingUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
        },
        body: JSON.stringify(data)
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Falha ao salvar progresso: ' + response.status);
        }
        return response.json();
    })
    .then(data => {
        console.log('Progresso salvo com sucesso:', data);
    })
    .catch(error => {
        console.error('Erro ao salvar progresso:', error);
    });
}

// Adicionar suporte explícito para reprodução de animes
window.playAnimeEpisode = function(videoLink, episodeId, animeData) {
    if (!videoLink) {
        console.error('Link de vídeo não fornecido');
        alert('Erro: Link de vídeo não disponível');
        return;
    }

    // Verificar e garantir que temos o ID do anime
    if (!animeData.anime_id) {
        // Tentar obter da meta tag na página
        const animeId = document.querySelector('meta[name="anime-id"]')?.content;
        if (animeId) {
            animeData.anime_id = parseInt(animeId);
            console.log('Recuperado anime_id da meta tag:', animeData.anime_id);
        } else {
            console.warn('anime_id não encontrado nem na meta tag');
        }
    }

    // Armazenar referências do conteúdo atual
    window.PlayerManager.currentType = 'anime';
    window.PlayerManager.currentContentId = animeData.anime_id;
    window.PlayerManager.currentEpisodeId = episodeId;
    window.PlayerManager.currentEpisodeData = animeData;

    // Fechar modal de servidor se estiver aberto
    const serverModal = document.getElementById('serverModal');
    if (serverModal) {
        serverModal.style.display = 'none';
    }

    // Criar o player fullscreen
    if (typeof openFullscreenPlayer === 'function') {
        const animeId = animeData.anime_id || document.querySelector('meta[name="anime-id"]')?.content;
        const posterImage = animeData.still_path || animeData.backdrop_path || '';
        const title = animeData.anime_name || document.querySelector('meta[name="anime-name"]')?.content || 'Anime';

        // Formatar título do episódio
        let subtitle = '';
        if (animeData.season_number && animeData.episode_number) {
            const seasonNum = String(animeData.season_number).padStart(2, '0');
            const episodeNum = String(animeData.episode_number).padStart(2, '0');
            subtitle = `T${seasonNum}E${episodeNum} ${animeData.name || ''}`;
        }

        console.log('Iniciando reprodução de anime com dados:', {
            videoLink,
            animeId,
            episodeId,
            title,
            subtitle
        });

        // Chamar função de reprodução em tela cheia com todos os parâmetros
        openFullscreenPlayer(
            videoLink,
            posterImage,
            'anime',
            title,
            subtitle,
            animeId,
            episodeId,
            animeData.season_number || 1,
            animeData.episode_number || 1
        );
    } else {
        console.error('Função openFullscreenPlayer não disponível');
        window.open(videoLink, '_blank');
    }
};

// Função para fullscreen player usada pelo sistema
window.openFullscreenPlayer = function(videoUrl, posterImage, contentType, title, subtitle, contentId, episodeId, seasonNumber, episodeNumber) {
    if (!videoUrl) {
        console.error('URL do vídeo não fornecido');
        return;
    }

    console.log('Abrindo player fullscreen com os parâmetros:', {
        contentType,
        contentId,
        episodeId,
        seasonNumber,
        episodeNumber
    });

    // Verificar e criar modal se não existir
    let playerModal = document.getElementById('fullscreen-player-modal');
    if (!playerModal) {
        playerModal = document.createElement('div');
        playerModal.id = 'fullscreen-player-modal';
        playerModal.className = 'fixed inset-0 z-50 flex items-center justify-center hidden bg-black';

        const modalContent = `
            <button id="close-fullscreen-player" class="absolute top-4 right-4 z-50 text-white bg-red-600 hover:bg-red-700 w-10 h-10 rounded-full flex items-center justify-center">
                <i class="fas fa-times"></i>
            </button>
            <div id="modal-player-container" class="w-full h-full"></div>
        `;

        playerModal.innerHTML = modalContent;
        document.body.appendChild(playerModal);
    }

    // Mostrar o modal
    playerModal.classList.remove('hidden');
    document.body.style.overflow = 'hidden';

    // Inicializar o player
    const playerContainer = document.getElementById('modal-player-container');

    // Limpar o container
    playerContainer.innerHTML = '';

    // Usar jwplayer se disponível
    if (typeof jwplayer === 'function') {
        // Buscar posição salva
        let startPosition = 0;

        // Função para inicializar o player com a posição inicial
        const initPlayer = (position) => {
            const playerInstance = jwplayer('modal-player-container').setup({
                file: videoUrl,
                image: posterImage,
                width: '100%',
                height: '100%',
                stretching: 'uniform',
                autostart: true,
                primary: 'html5',
                hlshtml: true,
                playbackRateControls: [0.5, 0.75, 1, 1.25, 1.5, 2],
                title: title,
                description: subtitle,
                starttime: position,
                volume: 80
            });

            // Configurar botão de fechar
            document.getElementById('close-fullscreen-player').onclick = function() {
                try {
                    // Salvar posição atual
                    const currentPosition = playerInstance.getPosition();
                    const totalDuration = playerInstance.getDuration();

                    console.log(`Salvando progresso ao fechar: ${currentPosition}/${totalDuration}`);

                    // Salvar progresso conforme tipo de conteúdo
                    if (contentType === 'anime' && episodeId) {
                        saveWatchProgress('anime', contentId, currentPosition, totalDuration, episodeId, seasonNumber, episodeNumber);
                    } else if (contentType === 'serie' && episodeId) {
                        saveWatchProgress('serie', contentId, currentPosition, totalDuration, episodeId, seasonNumber, episodeNumber);
                    } else {
                        saveWatchProgress('movie', contentId, currentPosition, totalDuration);
                    }

                    // Destruir player e fechar modal
                    playerInstance.remove();
                    playerModal.classList.add('hidden');
                    document.body.style.overflow = '';
                } catch (e) {
                    console.error('Erro ao fechar player:', e);
                    playerModal.classList.add('hidden');
                    document.body.style.overflow = '';
                }
            };

            // Salvar progresso periodicamente (a cada 10 segundos)
            playerInstance.on('time', function(e) {
                if (Math.floor(e.position) % 10 === 0 && e.position > 0) {
                    if (contentType === 'anime' && episodeId) {
                        saveWatchProgress('anime', contentId, e.position, e.duration, episodeId, seasonNumber, episodeNumber);
                    } else if (contentType === 'serie' && episodeId) {
                        saveWatchProgress('serie', contentId, e.position, e.duration, episodeId, seasonNumber, episodeNumber);
                    } else {
                        saveWatchProgress('movie', contentId, e.position, e.duration);
                    }
                }
            });

            // Salvar como concluído ao finalizar
            playerInstance.on('complete', function() {
                if (contentType === 'anime' && episodeId) {
                    saveWatchProgress('anime', contentId, 0, 0, episodeId, seasonNumber, episodeNumber);
                } else if (contentType === 'serie' && episodeId) {
                    saveWatchProgress('serie', contentId, 0, 0, episodeId, seasonNumber, episodeNumber);
                } else {
                    saveWatchProgress('movie', contentId, 0, 0);
                }
            });

            // Tentar ativar fullscreen após inicialização
            setTimeout(() => {
                try {
                    playerInstance.setFullscreen(true);
                } catch (e) {
                    console.warn('Não foi possível ativar fullscreen automaticamente:', e);
                }
            }, 1000);
        };

        // Buscar posição salva para o tipo de conteúdo correto
        if (episodeId) {
            // Para episódios, buscar via API
            const baseUrl = document.querySelector('meta[name="base-url"]')?.content || '';
            const apiType = contentType === 'anime' ? 'anime' : 'serie';

            console.log(`Buscando posição para ${apiType} ID: ${contentId}, episodeId: ${episodeId}`);

            // CORREÇÃO: Usar a URL completa com base na meta tag
            const positionUrl = `${baseUrl}/continue-watching/get-position`;

            fetch(`${positionUrl}?item_id=${contentId}&type=${apiType}&episode_id=${episodeId}`, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || ''
                }
            })
            .then(response => response.json())
            .then(data => {
                console.log('Recuperada posição:', data);
                initPlayer(data.position || 0);
            })
            .catch(error => {
                console.error('Erro ao buscar posição:', error);
                initPlayer(0);
            });
        } else {
            // Para filmes, usar localStorage
            const savedPosition = localStorage.getItem(`videoPosition-${contentId}`);
            startPosition = savedPosition ? parseFloat(savedPosition) : 0;
            initPlayer(startPosition);
        }
    } else {
        // Fallback para HTML5 video player
        playerContainer.innerHTML = `
            <video src="${videoUrl}" controls autoplay class="w-full h-full"></video>
        `;

        // Configurar botão de fechar
        document.getElementById('close-fullscreen-player').onclick = function() {
            playerModal.classList.add('hidden');
            document.body.style.overflow = '';
        };
    }
};

console.log('Player Integration Script carregado com sucesso.');
