/**
 * Script de integração do UltraPlayer com a plataforma {{$settings->app_name}} para filmes
 * Este script facilita a substituição do JWPlayer pelo UltraPlayer personalizado para filmes
 */

document.addEventListener('DOMContentLoaded', () => {
    console.log('Carregando integração do UltraPlayer para filmes...');

    // Verifica se já carregou o HLS.js (necessário para streams HLS)
    if (!window.Hls) {
        loadScript('https://cdn.jsdelivr.net/npm/hls.js@latest', () => {
            console.log('HLS.js carregado com sucesso!');
            initMoviePlayerIntegration();
        });
    } else {
        initMoviePlayerIntegration();
    }
});

// Função para carregar scripts externos dinamicamente
function loadScript(src, callback) {
    const script = document.createElement('script');
    script.src = src;
    script.async = true;

    script.onload = () => {
        console.log(`Script ${src} carregado com sucesso!`);
        if (callback) callback();
    };

    script.onerror = () => {
        console.error(`Erro ao carregar o script: ${src}`);
    };

    document.head.appendChild(script);
}

// Inicializa a integração do player
function initMoviePlayerIntegration() {
    console.log('Inicializando integração do player de filmes...');

    // Verificar se a classe UltraPlayer está disponível
    if (typeof window.UltraPlayer === 'undefined') {
        console.error('UltraPlayer não foi carregado corretamente!');
        return;
    }

    // Armazenar a instância atual do player
    window.currentMoviePlayer = null;

    // Configurar os botões de opção de vídeo
    const videoOptions = document.querySelectorAll('.option');
    if (videoOptions.length > 0) {
        videoOptions.forEach(option => {
            option.addEventListener('click', function() {
                // Remove seleção dos outros botões
                videoOptions.forEach(opt => opt.classList.remove('bg-primary'));
                this.classList.add('bg-primary');

                const videoLink = this.getAttribute('data-link');
                const movieId = this.getAttribute('data-movie-id');
                const movieTitle = document.querySelector('h1').textContent;
                const posterUrl = this.getAttribute('data-poster') || '';

                playMovieWithUltraPlayer(videoLink, movieTitle, posterUrl, movieId);
            });
        });

        // Modificar o watch-button para usar o modal de seleção de servidor
        const watchButton = document.getElementById('watch-button');
        if (watchButton) {
            // Substituir o manipulador de eventos existente
            const originalOnClick = watchButton.onclick;
            watchButton.onclick = function(event) {
                // Verificar se estamos usando openServerSelection agora (nova implementação)
                if (typeof openServerSelection === 'function') {
                    // Nova implementação usa modal, não fazer nada aqui
                    return;
                }

                // Cair para implementação antiga se openServerSelection não existir
                if (originalOnClick) {
                    originalOnClick.call(this, event);
                } else {
                    // Comportamento de fallback caso o evento original não exista
                    const firstOption = videoOptions[0];
                    if (firstOption) {
                        const videoLink = firstOption.getAttribute('data-link');
                        const movieId = firstOption.getAttribute('data-movie-id');
                        const movieTitle = document.querySelector('h1').textContent;
                        const posterUrl = firstOption.getAttribute('data-poster') || '';

                        // Mostrar o player
                        const playerMovie = document.querySelector('.player-movie');
                        if (playerMovie) {
                            playerMovie.style.display = 'block';
                        }

                        playMovieWithUltraPlayer(videoLink, movieTitle, posterUrl, movieId);

                        // Salvar no histórico
                        saveContinueWatching(movieId, 'movie');
                    }
                }
            };
        }
    }

    // Função para reproduzir o filme usando o UltraPlayer
    function playMovieWithUltraPlayer(videoLink, title, posterUrl, movieId) {
        console.log(`Iniciando reprodução do filme: ${title}`);

        // Fecha o player anterior se existir
        if (window.currentMoviePlayer) {
            try {
                window.currentMoviePlayer.destroy();
                window.currentMoviePlayer = null;
            } catch (e) {
                console.error('Erro ao fechar player anterior:', e);
            }
        }

        // Procura o container do player
        const playerContainer = document.getElementById('movie-player-container');
        if (!playerContainer) {
            console.error('Container do player não encontrado!');
            return;
        }

        // Limpa o conteúdo anterior
        playerContainer.innerHTML = '';

        try {
            // Verificar se existe posição salva
            const savedPosition = localStorage.getItem(`moviePosition-${movieId}`);
            let startPosition = 0;

            if (savedPosition) {
                startPosition = parseFloat(savedPosition);
                console.log(`Posição salva encontrada: ${startPosition} segundos`);

                // Se a posição salva for significativa, perguntar se deseja continuar
                if (startPosition > 30) {
                    // Criar modal de confirmação
                    const shouldContinue = confirm(`Deseja continuar de onde parou? (${formatTime(startPosition)})`);
                    if (!shouldContinue) {
                        startPosition = 0;
                        // Remover a posição salva se o usuário optar por começar do início
                        localStorage.removeItem(`moviePosition-${movieId}`);
                    } else {
                        // Mostrar notificação
                        console.log(`Continuando de ${formatTime(startPosition)}`);
                    }
                }
            }

            // Cria nova instância do UltraPlayer com configurações específicas para filmes
            const player = new window.UltraPlayer({
                container: playerContainer,
                source: videoLink,
                poster: posterUrl,
                autoplay: true,
                title: title,
                hlsConfig: {}, // Configuração padrão para HLS
                startTime: startPosition, // Define o tempo inicial baseado na posição salva

                // Configuração importante: definir como NULL para que os botões sejam completamente removidos
                // Usar 'false' faria com que fossem ocultados com CSS, mas NULL os remove do DOM
                prevEpisodeCallback: null,  // Remove completamente o botão de episódio anterior
                nextEpisodeCallback: null,  // Remove completamente o botão de próximo episódio

                // Verificação de premium - a implementação depende de como seu sistema identifica conteúdo premium
                requiresPremium: false, // Altere para true se todos os filmes exigirem premium, ou use lógica dinâmica abaixo

                // Função para verificar se o filme atual requer premium
                isPremiumUser: function() {
                    return true; // Se não conseguir verificar, assume que o usuário tem acesso
                },

                // Callback para quando o player estiver pronto
                onReady: () => {
                    console.log('UltraPlayer para filme pronto!');
                    // Pequeno delay para garantir que o player está completamente inicializado
                    setTimeout(() => {
                        try {
                            // Confirma que a posição inicial foi aplicada
                            if (startPosition > 0) {
                                player.seek(startPosition);
                                player.showNotification(`Continuando de ${formatTime(startPosition)}`);
                            }

                            player.enterFullscreen();
                        } catch (e) {
                            console.warn('Erro ao entrar em fullscreen:', e);
                        }
                    }, 500);
                },

                // Adicionar tratamento para saída do modo fullscreen
                onFullscreenChange: (isFullscreen) => {
                    // Se saiu do fullscreen e não estamos no processo de destruir o player
                    if (!isFullscreen && !window.isClosingMoviePlayer) {
                        console.log('Saindo do fullscreen, abrindo seleção de servidores');

                        // Destruir o player atual
                        if (window.currentMoviePlayer) {
                            try {
                                window.currentMoviePlayer.destroy();
                                window.currentMoviePlayer = null;
                            } catch (e) {
                                console.error('Erro ao destruir player:', e);
                            }
                        }

                        // Se estamos usando o novo modal de player
                        if (document.getElementById('videoPlayerModal')) {
                            document.getElementById('videoPlayerModal').style.display = 'none';
                        }

                        // Exibir o modal de seleção de servidores
                        if (typeof openServerSelection === 'function') {
                            // Pequeno delay para garantir que tudo está limpo
                            setTimeout(() => {
                                openServerSelection();
                            }, 100);
                        }
                    }
                },

                onTimeUpdate: (currentTime, duration) => {
                    // Função para salvar o progresso a cada X segundos
                    if (Math.floor(currentTime) % 30 === 0 && currentTime > 0) {
                        saveMovieProgress(movieId, currentTime, duration);
                    }
                },
                onEnded: () => {
                    // Filme terminou, limpar o progresso
                    localStorage.removeItem(`moviePosition-${movieId}`);

                    // Marcar como assistido completamente
                    saveMovieProgress(movieId, duration, duration);

                    // Mostrar seleção de servidores após um tempo
                    setTimeout(() => {
                        // Se o player estiver em fullscreen, sair primeiro
                        if (player.state && player.state.fullscreen) {
                            player.exitFullscreen();
                        }

                        // Destruir o player
                        if (window.currentMoviePlayer) {
                            try {
                                window.currentMoviePlayer.destroy();
                                window.currentMoviePlayer = null;
                            } catch (e) {
                                console.error('Erro ao destruir player:', e);
                            }
                        }

                        // Fechar modal do player se estiver sendo usado
                        if (document.getElementById('videoPlayerModal')) {
                            document.getElementById('videoPlayerModal').style.display = 'none';
                        }

                        // Abrir seleção de servidores
                        if (typeof openServerSelection === 'function') {
                            openServerSelection();
                        }
                    }, 3000);
                },
                onError: (error) => {
                    console.error('Erro no player:', error);
                }
            });

            // Armazenar referência para uso posterior
            window.currentMoviePlayer = player;

            // Adicionar tratamento de erro adicional
            if (player.elements && player.elements.video) {
                player.elements.video.addEventListener('error', (e) => {
                    console.error('Erro nativo de vídeo:', e);
                });
            }

            // Adicionar um evento de click no overlay para tentar entrar em fullscreen
            // após interação do usuário (necessário em alguns navegadores)
            const overlay = playerContainer.querySelector('.ultra-player-overlay');
            if (overlay) {
                // Adicionar botão de play/pause central para consistência com o player de séries
                const playCenterButton = document.createElement('div');
                playCenterButton.className = 'ultra-player-center-play';
                playCenterButton.innerHTML = `
                    <svg class="ultra-play-icon" viewBox="0 0 24 24" fill="white" width="48" height="48">
                        <path d="M8 5v14l11-7z"></path>
                    </svg>
                    <svg class="ultra-pause-icon" viewBox="0 0 24 24" fill="white" width="48" height="48" style="display:none">
                        <path d="M6 19h4V5H6v14zm8-14v14h4V5h-4z"></path>
                    </svg>
                `;
                playCenterButton.style.cssText = `
                    position: absolute;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    background-color: rgba(0, 0, 0, 0.5);
                    border-radius: 50%;
                    width: 64px;
                    height: 64px;
                    display: flex;
                    justify-content: center;
                    align-items: center;
                    opacity: 0;
                    transition: opacity 0.3s ease;
                    cursor: pointer;
                    z-index: 15;
                    box-shadow: 0 0 20px rgba(0, 0, 0, 0.5);
                `;
                overlay.appendChild(playCenterButton);

                // Mostrar o botão central quando o mouse entra no overlay ou toca em dispositivo móvel
                overlay.addEventListener('mouseenter', () => {
                    playCenterButton.style.opacity = '1';
                });

                overlay.addEventListener('touchstart', () => {
                    playCenterButton.style.opacity = '1';
                    // Ocultar após 3 segundos em dispositivos móveis
                    setTimeout(() => {
                        playCenterButton.style.opacity = '0';
                    }, 3000);
                }, { passive: true });

                // Esconder o botão central quando o mouse sai do overlay
                overlay.addEventListener('mouseleave', () => {
                    playCenterButton.style.opacity = '0';
                });

                // Corrigir o alternador de play/pause
                playCenterButton.addEventListener('click', (e) => {
                    e.stopPropagation(); // Evitar que o clique chegue ao overlay

                    try {
                        // Verifica o estado atual de reprodução
                        if (player.state && player.state.playing) {
                            // Se estiver reproduzindo, pausa o vídeo
                            player.pause();
                        } else {
                            // Se estiver pausado, inicia a reprodução
                            player.play();
                        }
                    } catch (err) {
                        console.error('Erro ao alternar reprodução:', err);
                    }
                });

                // Função para atualizar os ícones do botão
                const updatePlayPauseButton = (isPlaying) => {
                    const playIcon = playCenterButton.querySelector('.ultra-play-icon');
                    const pauseIcon = playCenterButton.querySelector('.ultra-pause-icon');

                    if (isPlaying) {
                        playIcon.style.display = 'none';
                        pauseIcon.style.display = 'block';
                    } else {
                        playIcon.style.display = 'block';
                        pauseIcon.style.display = 'none';
                    }
                };

                // Evento para quando o vídeo começa a reproduzir
                player.elements.video.addEventListener('play', () => {
                    updatePlayPauseButton(true);
                });

                // Evento para quando o vídeo é pausado
                player.elements.video.addEventListener('pause', () => {
                    updatePlayPauseButton(false);
                });

                // Definir o estado inicial baseado no autoplay
                updatePlayPauseButton(player.options.autoplay);

                // Adicionar handler para entrar em fullscreen com clique duplo e pause/play com clique único
                let clickCount = 0;
                overlay.addEventListener('click', (e) => {
                    // Ignorar cliques que vêm do botão de play/pause
                    if (e.target === playCenterButton || playCenterButton.contains(e.target)) {
                        return;
                    }

                    clickCount++;
                    if (clickCount === 1) {
                        setTimeout(() => {
                            if (clickCount === 1) {
                                // Clique único - alternar play/pause
                                if (player.state.playing) {
                                    player.pause();
                                } else {
                                    player.play();
                                }
                            } else {
                                // Clique duplo - entrar em fullscreen
                                if (!player.state.fullscreen) {
                                    player.enterFullscreen();
                                }
                            }
                            clickCount = 0;
                        }, 300);
                    }
                });
            }

            // Verificar se há posição anterior salva
            if (startPosition > 0) {
                // Já aplicamos a posição inicial no onReady, então não precisamos fazer nada aqui
                console.log(`Iniciando de posição salva: ${startPosition} segundos`);
            }

            // Adicionar manipulador para ESC fechar o modal se existir
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' || e.keyCode === 27) {
                    // Se existir um modal de player de vídeo
                    if (document.getElementById('videoPlayerModal') && document.getElementById('videoPlayerModal').style.display === 'block') {
                        // Se houver um botão de fechar, simulamos o clique nele
                        const closeButton = document.querySelector('.close-player-modal');
                        if (closeButton) {
                            closeButton.click();
                        }
                    }
                }
            });
        } catch (error) {
            console.error('Erro ao criar o player:', error);
            playerContainer.innerHTML = `
                <div style="display:flex;align-items:center;justify-content:center;height:100%;background:rgba(0,0,0,0.7);color:white;padding:20px;text-align:center;">
                    <div>
                        <h3 style="margin-bottom:10px;">Erro ao carregar o vídeo</h3>
                        <p style="margin-bottom:15px;">${error.message || 'Tente novamente ou selecione outra opção de vídeo.'}</p>
                        <button onclick="location.reload()" style="background:#E50914;border:none;color:white;padding:8px 16px;border-radius:4px;cursor:pointer;">Recarregar página</button>
                    </div>
                </div>
            `;
        }
    }

    // Função auxiliar para formatar tempo
    function formatTime(seconds) {
        if (isNaN(seconds) || !isFinite(seconds)) return '00:00';
        seconds = Math.floor(seconds);
        const mins = Math.floor(seconds / 60);
        const secs = seconds % 60;
        return `${mins}:${secs < 10 ? '0' : ''}${secs}`;
    }

    // Função para salvar o progresso do filme
    function saveMovieProgress(movieId, currentTime, duration) {
        if (!movieId || !currentTime || currentTime <= 0) return;

        try {
            // Usar o VideoPositionManager para salvar a posição
            window.VideoPositionManager.savePosition({
                item_id: movieId,
                type: 'movie',
                position: currentTime,
                duration: duration
            })
            .then(data => {
                console.log('Progresso do filme salvo com sucesso:', data);
            })
            .catch(err => {
                console.warn('Erro ao enviar progresso ao servidor:', err);
            });
        } catch (e) {
            console.warn('Erro ao salvar progresso:', e);
        }
    }

    // Função para registrar que o usuário começou a assistir o filme
    function saveContinueWatching(movieId, contentType) {
        try {
            // Determinar URL base para as requisições AJAX
            const baseUrl = window.appUrl || document.querySelector('meta[name="base-url"]')?.content || '';
            const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';
            const userId = document.querySelector('meta[name="user-id"]')?.content || '';

            // Criar dados da requisição
            const watchingData = {
                item_id: movieId,
                type: contentType,
                user_id: userId, // Adicionamos o user_id para caso não haja perfil selecionado
                position: 0, // Posição inicial
                duration: 0  // Duração inicial (será atualizada depois)
            };

            // Usar o endpoint correto
            fetch(`${baseUrl}/continue-watching`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(watchingData)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! Status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                console.log('Continue watching registrado com sucesso:', data);
            })
            .catch(e => {
                console.warn('Erro ao salvar continue watching:', e);
            });
        } catch (e) {
            console.warn('Erro ao salvar continue watching:', e);
        }
    }

    // Expor função para uso global
    window.playMovieWithUltraPlayer = playMovieWithUltraPlayer;
    window.saveContinueWatching = saveContinueWatching;

    // Integração com o modal de seleção de servidores e o player do JWPlayer
    if (typeof window.showVideoPlayer !== 'function' && typeof jwplayer === 'function') {
        window.showVideoPlayer = function(videoLink, movieId, posterImage) {
            // Se o modal JWPlayer estiver visível, escondê-lo
            if (document.getElementById('videoPlayerModal')) {
                document.getElementById('videoPlayerModal').style.display = 'none';
            }

            // Usar nossa própria implementação com UltraPlayer
            const container = document.getElementById('movie-player-container');
            if (container) {
                // Garantir que o container está visível
                const playerSection = document.querySelector('.player-movie');
                if (playerSection) {
                    playerSection.style.display = 'block';
                }

                // Buscar título do filme da página
                const movieTitle = document.querySelector('h1')?.textContent || 'Filme';

                // Iniciar player
                playMovieWithUltraPlayer(videoLink, movieTitle, posterImage, movieId);

                // Registrar visualização
                saveContinueWatching(movieId, 'movie');
            } else {
                // Se não encontrar nosso container, usar JWPlayer como fallback
                console.warn('Container UltraPlayer não encontrado, usando JWPlayer como fallback');

                // Repassar para implementação original se existir
                if (window.PlayerManager && window.PlayerManager.initialize) {
                    const player = window.PlayerManager.initialize('jwplayer-container', videoLink, {
                        movieId: movieId,
                        posterImage: posterImage
                    });

                    // Configurar eventos
                    if (window.PlayerManager.setupEvents) {
                        window.PlayerManager.setupEvents({
                            onReady: function() {
                                setTimeout(() => player.setFullscreen(true), 500);
                            }
                        });
                    }

                    // Exibir modal
                    if (document.getElementById('videoPlayerModal')) {
                        document.getElementById('videoPlayerModal').style.display = 'block';
                    }
                }
            }
        };
    }

    console.log('Integração do UltraPlayer para filmes concluída!');
}
