/**
 * UltraPlayer - Um player de vídeo personalizado inspirado no Netflix
 *
 * Características:
 * - Design semelhante ao Netflix
 * - Controles personalizados
 * - Suporte para HLS via hls.js
 * - Compatível com todos os navegadores modernos
 * - Responsivo para diferentes tamanhos de tela
 * - Suporte a legendas
 * - Animações elegantes
 */

// IIFE para evitar poluição do escopo global mas ainda expor a classe
(function(window) {
    'use strict';

    // Verificar se o UltraPlayer já foi declarado para evitar duplicações
    if (window.UltraPlayer) {
        console.log('UltraPlayer já foi declarado, pulando redeclaração');
        return;
    }

    class UltraPlayer {
        constructor(options) {
            // Verifica se options é uma string (seletor) ou um elemento DOM (compatibilidade)
            if (typeof options === 'string') {
                // Suporte para uso antigo: UltraPlayer('player-id', {opções})
                const containerId = options;
                options = arguments[1] || {};
                options.container = document.getElementById(containerId);
            } else if (options instanceof HTMLElement) {
                // Se o primeiro parâmetro é um elemento HTML, adapta para o novo formato
                const containerElement = options;
                options = arguments[1] || {};
                options.container = containerElement;
            }

            // Se options é um objeto e não tem container definido, verifica o segundo parâmetro
            if (options && typeof options === 'object' && !options.container && arguments[1]) {
                if (typeof arguments[1].container === 'string') {
                    options.container = document.querySelector(arguments[1].container);
                } else if (arguments[1].container instanceof HTMLElement) {
                    options.container = arguments[1].container;
                }
            }

            // Opções default
            this.options = {
                container: null,            // Elemento container
                source: null,               // URL do vídeo
                poster: null,               // URL da imagem de poster
                autoplay: false,            // Reproduzir automaticamente
                muted: false,               // Iniciar mudo
                loop: false,                // Loop do vídeo
                preload: 'auto',            // Pré-carregar vídeo
                controls: true,             // Mostrar controles
                startTime: 0,               // Tempo inicial em segundos
                title: '',                  // Título do vídeo
                subtitles: [],              // Array de objetos de legendas
                hlsConfig: {},              // Configuração do hls.js
                primaryColor: '#2BF14A',    // Cor primária (vermelho Netflix)
                secondaryColor: '#FFFFFF',  // Cor secundária
                backgroundColor: '#141414', // Cor de fundo
                onReady: null,              // Callback quando pronto
                onPlay: null,               // Callback quando reproduzir
                onPause: null,              // Callback quando pausar
                onEnded: null,              // Callback quando terminar
                onError: null,              // Callback de erro
                onTimeUpdate: null,         // Callback de atualização de tempo
                onVolumeChange: null,       // Callback de mudança de volume
                onFullscreenChange: null,   // Callback de mudança de tela cheia
                nextEpisodeCallback: null,  // Callback para próximo episódio
                prevEpisodeCallback: null,  // Callback para episódio anterior
                ...(options || {}),         // Substitui opções padrão
            };

            // Log para debug
            console.log('UltraPlayer construído com container:', this.options.container);

            // Estado do player
            this.state = {
                playing: false,
                volume: this.options.muted ? 0 : 1,
                currentTime: this.options.startTime || 0,
                duration: 0,
                buffered: 0,
                fullscreen: false,
                controlsVisible: true,
                controlsTimeout: null,
                dragging: false,
                loading: true,
                error: null,
                touchStartX: 0,
                touchStartY: 0,
                touchStartVolume: 1,
                touchStartTime: 0,
                lastVolume: 0.7,
                countdownInterval: null,
                notificationTimeout: null,
                settingsMenuVisible: false,
                subtitlesMenuVisible: false
            };

            // Instâncias externas
            this.hls = null;
            this.resizeObserver = null;

            // Inicializar
            this.init();
        }

        // Inicializa o player
        init() {
            // Verifica se temos um container válido para trabalhar
            if (!this.options.container) {
                console.error('UltraPlayer: Nenhum container especificado');
                return;
            }

            // Verifica se o container é um elemento DOM
            if (!(this.options.container instanceof HTMLElement)) {
                console.error('UltraPlayer: O container especificado não é um elemento DOM válido');
                return;
            }

            // Verifica se o container está no DOM
            if (!document.body.contains(this.options.container)) {
                console.error('UltraPlayer: O container não está conectado ao DOM');
                return;
            }

            try {
                this.createDOM();
                this.setupVideo();
                this.setupControls();
                this.setupEventListeners();
                this.setupKeyboardControls();
                this.setupTouchControls();
                this.setupSubtitles();
                this.setupHLS();
                this.updatePlaybackState();
                this.setupResizeObserver();

                // Esconder controles após 3 segundos
                this.resetControlsTimeout();

                console.log('UltraPlayer: inicializado com sucesso');
            } catch (error) {
                console.error('UltraPlayer: Erro durante a inicialização:', error);
                this.showError('Erro ao inicializar o player: ' + error.message);
            }
        }

        // Cria a estrutura DOM do player
        createDOM() {
            const container = this.options.container;
            container.classList.add('ultra-player-container');

            // Estilo básico para o container
            container.style.position = 'relative';
            container.style.width = '100%';
            container.style.height = '100%';
            container.style.backgroundColor = this.options.backgroundColor;
            container.style.overflow = 'hidden';

            // Injetando HTML básico
            container.innerHTML = `
                <div class="ultra-player-wrapper">
                    <!-- Vídeo elemento -->
                    <video class="ultra-player-video"></video>

                    <!-- Overlay para interação -->
                    <div class="ultra-player-overlay"></div>

                    <!-- Spinner de carregamento -->
                    <div class="ultra-player-loading">
                        <div class="ultra-player-spinner"></div>
                    </div>

                    <!-- Controles -->
                    <div class="ultra-player-controls">
                        <!-- Barra de progresso -->
                        <div class="ultra-player-progress-container">
                            <div class="ultra-player-progress-bar">
                                <div class="ultra-player-progress-buffered"></div>
                                <div class="ultra-player-progress-current"></div>
                                <div class="ultra-player-progress-handle"></div>
                            </div>
                            <!-- Preview da timeline -->
                            <div class="ultra-player-progress-tooltip">00:00</div>
                        </div>

                        <!-- Controles principais -->
                        <div class="ultra-player-controls-bottom">
                            <div class="ultra-player-controls-left">
                                <button class="ultra-player-btn ultra-player-play-btn">
                                    <svg class="ultra-player-play-icon" viewBox="0 0 24 24">
                                        <path d="M8 5v14l11-7z"></path>
                                    </svg>
                                    <svg class="ultra-player-pause-icon" viewBox="0 0 24 24" style="display:none;">
                                        <path d="M6 19h4V5H6v14zm8-14v14h4V5h-4z"></path>
                                    </svg>
                                </button>

                                <!-- Botão de episódio anterior com texto -->
                                <button class="ultra-player-btn ultra-player-prev-btn" style="${!this.options.prevEpisodeCallback ? 'display:none;' : ''}">
                                    <svg viewBox="0 0 24 24" style="margin-right:4px;">
                                        <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
                                    </svg>
                                    <span class="ultra-player-btn-text">Anterior</span>
                                </button>

                                <!-- Botão de próximo episódio com texto -->
                                <button class="ultra-player-btn ultra-player-next-btn" style="${!this.options.nextEpisodeCallback ? 'display:none;' : ''}">
                                    <span class="ultra-player-btn-text">Próximo</span>
                                    <svg viewBox="0 0 24 24" style="margin-left:4px;">
                                        <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
                                    </svg>
                                </button>

                                <!-- Volume -->
                                <div class="ultra-player-volume-container">
                                    <button class="ultra-player-btn ultra-player-volume-btn">
                                        <svg class="ultra-player-volume-high-icon" viewBox="0 0 24 24">
                                            <path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02zM14 3.23v2.06c2.89.86 5 3.54 5 6.71s-2.11 5.85-5 6.71v2.06c4.01-.91 7-4.49 7-8.77s-2.99-7.86-7-8.77z"></path>
                                        </svg>
                                        <svg class="ultra-player-volume-medium-icon" viewBox="0 0 24 24" style="display:none;">
                                            <path d="M3 9v6h4l5 5V4L7 9H3zm13.5 3c0-1.77-1.02-3.29-2.5-4.03v8.05c1.48-.73 2.5-2.25 2.5-4.02z"></path>
                                        </svg>
                                        <svg class="ultra-player-volume-low-icon" viewBox="0 0 24 24" style="display:none;">
                                            <path d="M7 9v6h4l5 5V4l-5 5H7z"></path>
                                        </svg>
                                        <svg class="ultra-player-volume-mute-icon" viewBox="0 0 24 24" style="display:none;">
                                            <path d="M16.5 12c0-1.77-1.02-3.29-2.5-4.03v2.21l2.45 2.45c.03-.2.05-.41.05-.63zm2.5 0c0 .94-.2 1.82-.54 2.64l1.51 1.51C20.63 14.91 21 13.5 21 12c0-4.28-2.99-7.86-7-8.77v2.06c2.89.86 5 3.54 5 6.71zM4.27 3L3 4.27 7.73 9H3v6h4l5 5v-6.73l4.25 4.25c-.67.52-1.42.93-2.25 1.18v2.06c1.38-.31 2.63-.95 3.69-1.81L19.73 21 21 19.73l-9-9L4.27 3zM12 4L9.91 6.09 12 8.18V4z"></path>
                                        </svg>
                                    </button>

                                    <div class="ultra-player-volume-slider-container">
                                        <div class="ultra-player-volume-slider">
                                            <div class="ultra-player-volume-level"></div>
                                            <div class="ultra-player-volume-handle"></div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Tempo -->
                                <div class="ultra-player-time">
                                    <span class="ultra-player-current-time">00:00</span>
                                    <span class="ultra-player-time-separator">/</span>
                                    <span class="ultra-player-duration">00:00</span>
                                </div>
                            </div>

                            <div class="ultra-player-controls-right">
                                <!-- Botão de legendas -->
                                <button class="ultra-player-btn ultra-player-subtitles-btn">
                                    <svg viewBox="0 0 24 24">
                                        <path d="M20 4H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zM4 12h4v2H4v-2zm10 6H4v-2h10v2zm6 0h-4v-2h4v2zm0-4H10v-2h10v2z"></path>
                                    </svg>
                                </button>

                                <!-- Botão de tela cheia -->
                                <button class="ultra-player-btn ultra-player-fullscreen-btn">
                                    <svg class="ultra-player-fullscreen-icon" viewBox="0 0 24 24">
                                        <path d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z"></path>
                                    </svg>
                                    <svg class="ultra-player-fullscreen-exit-icon" viewBox="0 0 24 24" style="display:none;">
                                        <path d="M5 16h3v3h2v-5H5v2zm3-8H5v2h5V5H8v3zm6 11h2v-3h3v-2h-5v5zm2-11V5h-2v5h5V8h-3z"></path>
                                </button>
                            </div>
                        </div>
                    </div>

                    <!-- Título e descrição -->
                    <div class="ultra-player-title-container">
                        <h3 class="ultra-player-title"></h3>
                    </div>

                    <!-- Menu de legendas (inicialmente oculto) -->
                    <div class="ultra-player-subtitles-menu" style="display:none;">
                        <div class="ultra-player-settings-header">Legendas</div>
                        <div class="ultra-player-settings-options ultra-player-subtitles-options">
                            <button data-subtitle="off" class="active">Desativadas</button>
                            <!-- Opções de legendas serão preenchidas dinamicamente -->
                        </div>
                    </div>

                    <!-- Mensagem de erro (inicialmente oculta) -->
                    <div class="ultra-player-error" style="display:none;">
                        <div class="ultra-player-error-content">
                            <svg viewBox="0 0 24 24" width="48" height="48">
                                <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z"></path>
                            </svg>
                            <h3>Ocorreu um erro</h3>
                            <p class="ultra-player-error-message"></p>
                            <button class="ultra-player-error-retry">Tentar novamente</button>
                        </div>
                    </div>

                    <!-- Notificação flutuante (inicialmente oculta) -->
                    <div class="ultra-player-notification" style="display:none;"></div>

                    <!-- Interface para navegação entre episódios -->
                    <div class="ultra-player-episode-navigation" style="display:none;">
                        <div class="ultra-player-next-episode-info">
                            <h4>Próximo episódio em <span class="ultra-player-countdown">10</span></h4>
                            <h3 class="ultra-player-next-title"></h3>
                            <div class="ultra-player-navigation-buttons">
                                <button class="ultra-player-cancel-next">Cancelar</button>
                                <button class="ultra-player-play-next">Reproduzir agora</button>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            // Armazenar referências a elementos DOM
            this.elements = {
                wrapper: container.querySelector('.ultra-player-wrapper'),
                video: container.querySelector('.ultra-player-video'),
                overlay: container.querySelector('.ultra-player-overlay'),
                loading: container.querySelector('.ultra-player-loading'),
                controls: container.querySelector('.ultra-player-controls'),
                playBtn: container.querySelector('.ultra-player-play-btn'),
                prevBtn: container.querySelector('.ultra-player-prev-btn'),
                nextBtn: container.querySelector('.ultra-player-next-btn'),
                playIcon: container.querySelector('.ultra-player-play-icon'),
                pauseIcon: container.querySelector('.ultra-player-pause-icon'),
                volumeBtn: container.querySelector('.ultra-player-volume-btn'),
                volumeHigh: container.querySelector('.ultra-player-volume-high-icon'),
                volumeMedium: container.querySelector('.ultra-player-volume-medium-icon'),
                volumeLow: container.querySelector('.ultra-player-volume-low-icon'),
                volumeMute: container.querySelector('.ultra-player-volume-mute-icon'),
                volumeSlider: container.querySelector('.ultra-player-volume-slider'),
                volumeLevel: container.querySelector('.ultra-player-volume-level'),
                volumeHandle: container.querySelector('.ultra-player-volume-handle'),
                currentTime: container.querySelector('.ultra-player-current-time'),
                duration: container.querySelector('.ultra-player-duration'),
                progressContainer: container.querySelector('.ultra-player-progress-container'),
                progressBar: container.querySelector('.ultra-player-progress-bar'),
                progressCurrent: container.querySelector('.ultra-player-progress-current'),
                progressBuffered: container.querySelector('.ultra-player-progress-buffered'),
                progressHandle: container.querySelector('.ultra-player-progress-handle'),
                progressTooltip: container.querySelector('.ultra-player-progress-tooltip'),
                fullscreenBtn: container.querySelector('.ultra-player-fullscreen-btn'),
                fullscreenIcon: container.querySelector('.ultra-player-fullscreen-icon'),
                fullscreenExitIcon: container.querySelector('.ultra-player-fullscreen-exit-icon'),
                subtitlesBtn: container.querySelector('.ultra-player-subtitles-btn'),
                subtitlesMenu: container.querySelector('.ultra-player-subtitles-menu'),
                subtitlesOptions: container.querySelector('.ultra-player-subtitles-options'),
                titleContainer: container.querySelector('.ultra-player-title-container'),
                title: container.querySelector('.ultra-player-title'),
                error: container.querySelector('.ultra-player-error'),
                errorMessage: container.querySelector('.ultra-player-error-message'),
                errorRetry: container.querySelector('.ultra-player-error-retry'),
                notification: container.querySelector('.ultra-player-notification'),
                episodeNavigation: container.querySelector('.ultra-player-episode-navigation'),
                nextEpisodeInfo: container.querySelector('.ultra-player-next-episode-info'),
                countdown: container.querySelector('.ultra-player-countdown'),
                nextTitle: container.querySelector('.ultra-player-next-title'),
                cancelNext: container.querySelector('.ultra-player-cancel-next'),
                playNext: container.querySelector('.ultra-player-play-next'),
                container: container // Referência ao container principal
            };

            // Aplicar estilos básicos
            this.applyStyles();
        }

        // Aplica estilos básicos ao player
        applyStyles() {
            // Insere estilos CSS
            if (!document.querySelector('#ultra-player-styles')) {
                const styleSheet = document.createElement('style');
                styleSheet.id = 'ultra-player-styles';
                styleSheet.textContent = `
                    .ultra-player-container {
                        font-family: 'Helvetica Neue', Helvetica, Arial, sans-serif;
                        -webkit-font-smoothing: antialiased;
                        -moz-osx-font-smoothing: grayscale;
                        position: relative;
                        width: 100%;
                        height: 100%;
                        background-color: #000;
                        overflow: hidden;
                        touch-action: manipulation;
                        user-select: none;
                    }

                    .ultra-player-wrapper {
                        position: relative;
                        width: 100%;
                        height: 100%;
                    }

                    .ultra-player-video {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        object-fit: contain;
                        background-color: #000;
                    }

                    .ultra-player-overlay {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        cursor: pointer;
                        z-index: 10;
                    }

                    .ultra-player-loading {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        display: flex;
                        justify-content: center;
                        align-items: center;
                        z-index: 20;
                        background-color: rgba(0, 0, 0, 0.5);
                        transition: opacity 0.3s ease;
                    }

                    .ultra-player-spinner {
                        width: 40px;
                        height: 40px;
                        border-radius: 50%;
                        border: 4px solid rgba(255, 255, 255, 0.2);
                        border-top-color: ${this.options.primaryColor};
                        animation: ultra-spinner 1s linear infinite;
                    }

                    @keyframes ultra-spinner {
                        to { transform: rotate(360deg); }
                    }

                    .ultra-player-controls {
                        position: absolute;
                        bottom: 0;
                        left: 0;
                        width: 100%;
                        z-index: 30;
                        padding: 10px 0;
                        background: linear-gradient(transparent, rgba(0, 0, 0, 0.7));
                        opacity: 1;
                        transition: opacity 0.3s ease, transform 0.3s ease;
                    }

                    .ultra-player-controls.hidden {
                        opacity: 0;
                        transform: translateY(100%);
                    }

                    .ultra-player-btn-text {
                        font-size: 14px;
                        font-weight: 500;
                        white-space: nowrap;
                    }

                    .ultra-player-prev-btn, .ultra-player-next-btn {
                        display: flex;
                        align-items: center;
                        padding: 5px 10px;
                        border-radius: 4px;
                        background-color: rgba(255, 255, 255, 0.1);
                        margin: 0 5px;
                        transition: background-color 0.2s;
                    }

                    .ultra-player-prev-btn:hover, .ultra-player-next-btn:hover {
                        background-color: rgba(255, 255, 255, 0.2);
                    }

                    /* Resto dos estilos existentes */
                    .ultra-player-progress-container {
                        width: 100%;
                        height: 20px;
                        padding: 10px 0;
                        cursor: pointer;
                    }

                    .ultra-player-progress-bar {
                        position: relative;
                        width: 100%;
                        height: 4px;
                        background-color: rgba(255, 255, 255, 0.3);
                        border-radius: 2px;
                        overflow: hidden;
                        transition: height 0.2s ease;
                    }

                    .ultra-player-progress-container:hover .ultra-player-progress-bar {
                        height: 6px;
                    }

                    .ultra-player-progress-buffered {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 0;
                        height: 100%;
                        background-color: rgba(255, 255, 255, 0.5);
                        border-radius: 2px;
                    }

                    .ultra-player-progress-current {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 0;
                        height: 100%;
                        background-color: ${this.options.primaryColor};
                        border-radius: 2px;
                    }

                    .ultra-player-progress-handle {
                        position: absolute;
                        top: 50%;
                        left: 0;
                        transform: translate(-50%, -50%);
                        width: 12px;
                        height: 12px;
                        border-radius: 50%;
                        background-color: ${this.options.primaryColor};
                        opacity: 0;
                        transition: opacity 0.2s ease;
                    }

                    .ultra-player-progress-container:hover .ultra-player-progress-handle {
                        opacity: 1;
                    }

                    .ultra-player-progress-tooltip {
                        position: absolute;
                        bottom: 20px;
                        padding: 4px 8px;
                        background-color: rgba(0, 0, 0, 0.7);
                        color: white;
                        border-radius: 4px;
                        font-size: 12px;
                        pointer-events: none;
                        opacity: 0;
                        transform: translateX(-50%);
                        transition: opacity 0.2s ease;
                    }

                    .ultra-player-controls-bottom {
                        display: flex;
                        justify-content: space-between;
                        align-items: center;
                        padding: 0 20px;
                    }

                    .ultra-player-controls-left,
                    .ultra-player-controls-right {
                        display: flex;
                        align-items: center;
                    }

                    .ultra-player-btn {
                        background: none;
                        border: none;
                        color: white;
                        cursor: pointer;
                        padding: 8px;
                        margin: 0 4px;
                        outline: none;
                        opacity: 0.9;
                        transition: opacity 0.2s ease;
                    }

                    .ultra-player-btn:hover {
                        opacity: 1;
                    }

                    .ultra-player-btn svg {
                        width: 24px;
                        height: 24px;
                        fill: white;
                    }

                    .ultra-player-volume-container {
                        display: flex;
                        align-items: center;
                    }

                    .ultra-player-volume-slider-container {
                        width: 0;
                        overflow: hidden;
                        transition: width 0.3s ease;
                    }

                    .ultra-player-volume-container:hover .ultra-player-volume-slider-container {
                        width: 80px;
                    }

                    .ultra-player-volume-slider {
                        position: relative;
                        width: 80px;
                        height: 4px;
                        margin: 0 10px;
                        background-color: rgba(255, 255, 255, 0.3);
                        border-radius: 2px;
                        cursor: pointer;
                    }

                    .ultra-player-volume-level {
                        position: absolute;
                        top: 0;
                        left: 0;
                        height: 100%;
                        background-color: white;
                        border-radius: 2px;
                    }

                    .ultra-player-volume-handle {
                        position: absolute;
                        top: 50%;
                        transform: translate(-50%, -50%);
                        width: 12px;
                        height: 12px;
                        border-radius: 50%;
                        background-color: white;
                    }

                    .ultra-player-time {
                        font-size: 14px;
                        color: white;
                        margin-left: 10px;
                    }

                    .ultra-player-title-container {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        padding: 20px;
                        background: linear-gradient(rgba(0, 0, 0, 0.7), transparent);
                        transition: opacity 0.3s ease, transform 0.3s ease;
                        opacity: 0;
                        z-index: 25;
                    }

                    .ultra-player-title {
                        color: white;
                        font-size: 18px;
                        font-weight: 500;
                        margin: 0;
                        text-shadow: 0 1px 2px rgba(0, 0, 0, 0.5);
                    }

                    .ultra-player-settings-menu,
                    .ultra-player-subtitles-menu {
                        position: absolute;
                        right: 10px;
                        bottom: 60px;
                        width: 200px;
                        background-color: rgba(28, 28, 28, 0.9);
                        border-radius: 4px;
                        z-index: 40;
                        overflow: hidden;
                        transition: opacity 0.2s ease, transform 0.2s ease;
                        transform-origin: bottom right;
                    }

                    .ultra-player-settings-header {
                        padding: 12px 15px;
                        font-size: 14px;
                        color: white;
                        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
                    }

                    .ultra-player-settings-section {
                        padding: 10px 15px;
                        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
                    }

                    .ultra-player-settings-title {
                        color: #aaa;
                        font-size: 12px;
                        margin-bottom: 8px;
                    }

                    .ultra-player-settings-options {
                        display: flex;
                        flex-wrap: wrap;
                        gap: 8px;
                    }

                    .ultra-player-settings-options button {
                        background: none;
                        border: none;
                        color: white;
                        padding: 4px 8px;
                        font-size: 13px;
                        border-radius: 2px;
                        cursor: pointer;
                        background-color: rgba(255, 255, 255, 0.1);
                    }

                    .ultra-player-settings-options button.active {
                        background-color: ${this.options.primaryColor};
                    }

                    .ultra-player-error {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        background-color: rgba(0, 0, 0, 0.8);
                        display: flex;
                        justify-content: center;
                        align-items: center;
                        z-index: 50;
                    }

                    .ultra-player-error-content {
                        text-align: center;
                        color: white;
                        padding: 20px;
                    }

                    .ultra-player-error-content svg {
                        fill: white;
                        margin-bottom: 15px;
                    }

                    .ultra-player-error-content h3 {
                        font-size: 20px;
                        margin: 0 0 10px 0;
                    }

                    .ultra-player-error-message {
                        font-size: 14px;
                        margin: 0 0 20px 0;
                        color: #aaa;
                    }

                    .ultra-player-error-retry {
                        background-color: ${this.options.primaryColor};
                        color: white;
                        border: none;
                        padding: 8px 16px;
                        border-radius: 4px;
                        font-size: 14px;
                        cursor: pointer;
                    }

                    .ultra-player-notification {
                        position: absolute;
                        bottom: 70px;
                        left: 20px;
                        background-color: rgba(0, 0, 0, 0.7);
                        color: white;
                        padding: 10px 15px;
                        border-radius: 4px;
                        font-size: 14px;
                        z-index: 35;
                        transition: opacity 0.3s ease;
                    }

                    .ultra-player-episode-navigation {
                        position: absolute;
                        bottom: 70px;
                        right: 20px;
                        width: 300px;
                        background-color: rgba(28, 28, 28, 0.9);
                        border-radius: 4px;
                        z-index: 35;
                        overflow: hidden;
                    }

                    .ultra-player-next-episode-info {
                        padding: 15px;
                    }

                    .ultra-player-next-episode-info h4 {
                        font-size: 14px;
                        margin: 0 0 10px 0;
                        color: #aaa;
                    }

                    .ultra-player-next-episode-info h3 {
                        font-size: 16px;
                        margin: 0 0 15px 0;
                    }

                    .ultra-player-navigation-buttons {
                        display: flex;
                        gap: 10px;
                    }

                    .ultra-player-navigation-buttons button {
                        flex: 1;
                        padding: 8px;
                        border: none;
                        border-radius: 2px;
                        font-size: 14px;
                        cursor: pointer;
                    }

                    .ultra-player-cancel-next {
                        background-color: rgba(255, 255, 255, 0.2);
                        color: white;
                    }

                    .ultra-player-play-next {
                        background-color: ${this.options.primaryColor};
                        color: white;
                    }

                    @media (max-width: 768px) {
                        .ultra-player-btn svg {
                            width: 20px;
                            height: 20px;
                        }

                        .ultra-player-time {
                            font-size: 12px;
                        }

                        .ultra-player-btn-text {
                            font-size: 12px;
                        }

                        .ultra-player-controls-bottom {
                            padding: 0 10px;
                        }

                        .ultra-player-episode-navigation {
                            width: 250px;
                        }
                    }

                    @media (max-width: 480px) {
                        .ultra-player-btn-text {
                            display: none;
                        }

                        .ultra-player-volume-container:hover .ultra-player-volume-slider-container {
                            width: 60px;
                        }

                        .ultra-player-volume-slider {
                            width: 60px;
                        }

                        .ultra-player-btn {
                            padding: 6px;
                            margin: 0 2px;
                        }

                        .ultra-player-btn svg {
                            width: 18px;
                            height: 18px;
                        }

                        .ultra-player-time {
                            font-size: 11px;
                        }
                    }
                `;
                document.head.appendChild(styleSheet);
            }

            // Aplicar estilos diretos ao vídeo
            const videoEl = this.elements.video;
            videoEl.style.width = '100%';
            videoEl.style.height = '100%';
            videoEl.style.objectFit = 'contain';
        }

        // Configuração do elemento de vídeo
        setupVideo() {
            const video = this.elements.video;

            // Definir atributos
            video.src = this.options.source;
            video.poster = this.options.poster || '';
            video.autoplay = this.options.autoplay;
            video.muted = this.options.muted;
            video.loop = this.options.loop;
            video.volume = this.state.volume;
            video.preload = this.options.preload;
            video.playsInline = true; // Importante para iOS

            if (this.options.title) {
                // Garantir que os números da temporada/episódio estejam corretos
                let titleText = this.options.title;

                // Verifica se tem o formato "Nome - Temporada X Episódio Y"
                if (titleText.includes(' - Temporada ') && titleText.includes(' Episódio ')) {
                    const showName = titleText.split(' - ')[0];
                    let seasonNumber = this.extractSeasonNumber(titleText);
                    const episodeNumber = this.extractEpisodeNumber(titleText);

                    // Se não conseguiu extrair o número da temporada, use o objeto options
                    if (seasonNumber === undefined || isNaN(seasonNumber)) {
                        // Tentar obter do título ou das options
                        seasonNumber = this.options.seasonNumber || "?";
                    }

                    // Reconstruir o título com informações corretas
                    titleText = `${showName} - Temporada ${seasonNumber} Episódio ${episodeNumber}`;
                }

                this.elements.title.textContent = titleText;
                this.elements.titleContainer.style.opacity = '1';
                setTimeout(() => {
                    this.elements.titleContainer.style.opacity = '0';
                }, 5000);
            }

            // Configuração de eventos para o vídeo
            this.setupEventListeners();
        }

        // Extrai o número da temporada do título
        extractSeasonNumber(title) {
            const match = title.match(/Temporada\s+(\d+)/);
            return match ? parseInt(match[1]) : undefined;
        }

        // Extrai o número do episódio do título
        extractEpisodeNumber(title) {
            const match = title.match(/Episódio\s+(\d+)/);
            return match ? parseInt(match[1]) : undefined;
        }

        // Configuração de eventos para o vídeo
        setupEventListeners() {
            const videoEl = this.elements.video;

            // Eventos de carregamento
            videoEl.addEventListener('loadstart', () => {
                if (!this.elements) return; // Verificação de segurança

                this.state.loading = true;
                if (this.elements.loading) {
                    this.elements.loading.style.display = 'flex';
                }
            });

            videoEl.addEventListener('canplay', () => {
                if (!this.elements) return; // Verificação de segurança

                this.state.loading = false;
                if (this.elements.loading) {
                    this.elements.loading.style.display = 'none';
                }

                this.state.duration = videoEl.duration;
                if (this.elements.duration) {
                    this.elements.duration.textContent = this.formatTime(videoEl.duration);
                }

                if (this.options.startTime > 0 && this.options.startTime < videoEl.duration) {
                    this.seek(this.options.startTime);
                }

                if (this.options.onReady) {
                    this.options.onReady();
                }
            });

            // Eventos de reprodução
            videoEl.addEventListener('play', () => {
                this.state.playing = true;
                this.updatePlaybackState();

                if (this.options.onPlay) {
                    this.options.onPlay();
                }
            });

            videoEl.addEventListener('pause', () => {
                this.state.playing = false;
                this.updatePlaybackState();

                if (this.options.onPause) {
                    this.options.onPause();
                }
            });

            videoEl.addEventListener('ended', () => {
                this.state.playing = false;
                this.updatePlaybackState();

                if (this.options.loop) {
                    this.seek(0);
                    videoEl.play().catch(() => {});
                } else {
                    this.showEpisodeNavigation();
                }

                if (this.options.onEnded) {
                    this.options.onEnded();
                }
            });

            // Atualizações de tempo
            videoEl.addEventListener('timeupdate', () => {
                if (!this.elements || !this.elements.currentTime) return; // Verificação de segurança

                if (!this.state.dragging) {
                    this.state.currentTime = videoEl.currentTime;
                    this.elements.currentTime.textContent = this.formatTime(videoEl.currentTime);

                    if (videoEl.duration && this.elements.progressCurrent && this.elements.progressHandle) {
                        const percent = videoEl.currentTime / videoEl.duration;
                        this.elements.progressCurrent.style.width = `${percent * 100}%`;
                        this.elements.progressHandle.style.left = `${percent * 100}%`;
                    }

                    if (this.options.onTimeUpdate) {
                        this.options.onTimeUpdate(videoEl.currentTime, videoEl.duration);
                    }
                }
            });

            // Atualizações de buffer
            videoEl.addEventListener('progress', () => {
                if (!this.elements || !this.elements.progressBuffered) return; // Verificação de segurança

                if (videoEl.buffered.length > 0) {
                    const bufferedEnd = videoEl.buffered.end(videoEl.buffered.length - 1);
                    const percent = bufferedEnd / videoEl.duration;
                    this.elements.progressBuffered.style.width = `${percent * 100}%`;
                }
            });

            // Volumes
            videoEl.addEventListener('volumechange', () => {
                this.state.volume = videoEl.volume;
                this.updateVolumeUI();

                if (this.options.onVolumeChange) {
                    this.options.onVolumeChange(videoEl.volume, videoEl.muted);
                }
            });

            // Gestão de erros
            videoEl.addEventListener('error', () => {
                if (!this.elements) return; // Verificação de segurança

                this.state.loading = false;
                if (this.elements.loading) {
                    this.elements.loading.style.display = 'none';
                }

                let errorMessage = 'Ocorreu um erro ao reproduzir o vídeo.';
                // Códigos de erro: https://developer.mozilla.org/en-US/docs/Web/API/MediaError/code
                if (videoEl.error) {
                    switch (videoEl.error.code) {
                        case 1:
                            errorMessage = 'A reprodução foi interrompida devido a uma falha de rede.';
                            break;
                        case 2:
                            errorMessage = 'A reprodução foi interrompida devido a um erro de decodificação.';
                            break;
                        case 3:
                            errorMessage = 'A reprodução foi interrompida porque o recurso de mídia não é suportado.';
                            break;
                        case 4:
                            errorMessage = 'A reprodução foi interrompida devido a um problema com a origem do vídeo.';
                            break;
                    }
                }

                this.showError(errorMessage);

                if (this.options.onError) {
                    this.options.onError(videoEl.error);
                }
            });

            // Evento de clique no overlay para alternar reprodução
            this.elements.overlay.addEventListener('click', () => {
                if (this.state.subtitlesMenuVisible) {
                    this.hideAllMenus();
                } else {
                    this.togglePlay();
                    this.showControls();
                    this.resetControlsTimeout();
                }
            });

            // Evento de duplo clique no overlay para alternar tela cheia
            let clickCount = 0;
            this.elements.overlay.addEventListener('click', () => {
                clickCount++;
                setTimeout(() => {
                    if (clickCount === 1) {
                        // Single click
                    } else if (clickCount === 2) {
                        this.toggleFullscreen();
                    }
                    clickCount = 0;
                }, 300);
            });

            // Mostrar controles ao mover o mouse
            this.elements.wrapper.addEventListener('mousemove', () => {
                this.showControls();
                this.resetControlsTimeout();
            });
        }

        // Configuração da barra de progresso
        setupProgressBar() {
            const progressContainer = this.elements.progressContainer;
            const progressBar = this.elements.progressBar;
            const progressCurrent = this.elements.progressCurrent;
            const progressHandle = this.elements.progressHandle;
            const progressTooltip = this.elements.progressTooltip;

            // Manipulador de eventos de clique na barra de progresso
            progressContainer.addEventListener('click', (e) => {
                e.stopPropagation();

                const rect = progressBar.getBoundingClientRect();
                const offsetX = e.clientX - rect.left;
                const seekPercent = offsetX / rect.width;
                const seekTime = seekPercent * this.state.duration;

                this.seek(seekTime);
            });

            // Exibir tooltip de tempo ao passar o mouse
            progressContainer.addEventListener('mousemove', (e) => {
                const rect = progressBar.getBoundingClientRect();
                const offsetX = e.clientX - rect.left;
                const seekPercent = offsetX / rect.width;
                const seekTime = seekPercent * this.state.duration;

                progressTooltip.textContent = this.formatTime(seekTime);
                progressTooltip.style.left = `${e.clientX}px`;
                progressTooltip.style.opacity = '1';
            });

            // Ocultar tooltip ao sair da barra
            progressContainer.addEventListener('mouseleave', () => {
                progressTooltip.style.opacity = '0';
            });

            // Implementação de arrastar na barra de progresso
            let progressDragging = false;

            progressContainer.addEventListener('mousedown', (e) => {
                e.stopPropagation();
                progressDragging = true;
                this.state.dragging = true;
                this.elements.video.pause();
            });

            document.addEventListener('mousemove', (e) => {
                if (progressDragging) {
                    const rect = progressBar.getBoundingClientRect();
                    const offsetX = Math.max(0, Math.min(rect.width, e.clientX - rect.left));
                    const seekPercent = offsetX / rect.width;
                    const seekTime = seekPercent * this.state.duration;

                    // Atualizar visualmente
                    this.elements.progressCurrent.style.width = `${seekPercent * 100}%`;
                    this.elements.progressHandle.style.left = `${seekPercent * 100}%`;

                    // Atualizar tooltip
                    progressTooltip.textContent = this.formatTime(seekTime);
                    progressTooltip.style.left = `${e.clientX}px`;
                    progressTooltip.style.opacity = '1';
                }
            });

            document.addEventListener('mouseup', (e) => {
                if (progressDragging) {
                    const rect = progressBar.getBoundingClientRect();
                    const offsetX = Math.max(0, Math.min(rect.width, e.clientX - rect.left));
                    const seekPercent = offsetX / rect.width;
                    const seekTime = seekPercent * this.state.duration;

                    this.seek(seekTime);
                    this.state.dragging = false;

                    if (this.state.playing) {
                        this.elements.video.play().catch(() => {});
                    }

                    progressDragging = false;
                }
            });
        }

        // Configuração do slider de volume
        setupVolumeSlider() {
            const volumeSlider = this.elements.volumeSlider;
            const volumeLevel = this.elements.volumeLevel;
            const volumeHandle = this.elements.volumeHandle;

            // Manipulador de eventos de clique no slider de volume
            volumeSlider.addEventListener('click', (e) => {
                e.stopPropagation();

                const rect = volumeSlider.getBoundingClientRect();
                const offsetX = e.clientX - rect.left;
                const volumePercent = Math.max(0, Math.min(1, offsetX / rect.width));

                this.setVolume(volumePercent);
            });

            // Implementação de arrastar o controle de volume
            let volumeDragging = false;

            volumeHandle.addEventListener('mousedown', (e) => {
                e.stopPropagation();
                volumeDragging = true;
            });

            document.addEventListener('mousemove', (e) => {
                if (volumeDragging) {
                    const rect = volumeSlider.getBoundingClientRect();
                    const offsetX = e.clientX - rect.left;
                    const volumePercent = Math.max(0, Math.min(1, offsetX / rect.width));

                    this.setVolume(volumePercent);
                }
            });

            document.addEventListener('mouseup', () => {
                volumeDragging = false;
            });
        }

        // Configuração dos controles personalizados
        setupControls() {
            // Botão de reprodução/pausa
            this.elements.playBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.togglePlay();
            });

            // Botão de episódio anterior
            if (this.options.prevEpisodeCallback) {
                this.elements.prevBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    if (typeof this.options.prevEpisodeCallback === 'function') {
                        this.options.prevEpisodeCallback();
                    }
                });
            } else {
                // Esconde o botão caso o callback não esteja definido
                if (this.elements.prevBtn) {
                    this.elements.prevBtn.style.display = 'none';
                }
            }

            // Botão de próximo episódio
            if (this.options.nextEpisodeCallback) {
                this.elements.nextBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    if (typeof this.options.nextEpisodeCallback === 'function') {
                        this.options.nextEpisodeCallback();
                    }
                });
            } else {
                // Esconde o botão caso o callback não esteja definido
                if (this.elements.nextBtn) {
                    this.elements.nextBtn.style.display = 'none';
                }
            }

            // Botão de volume
            this.elements.volumeBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleMute();
            });

            // Botão de tela cheia
            this.elements.fullscreenBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleFullscreen();
            });

            // Botão de legendas
            this.elements.subtitlesBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.toggleSubtitlesMenu();
            });

            // Botão de tentar novamente em caso de erro
            this.elements.errorRetry.addEventListener('click', () => {
                this.retry();
            });

            // Botões de navegação de episódios
            this.elements.cancelNext.addEventListener('click', () => {
                this.hideEpisodeNavigation();
            });

            this.elements.playNext.addEventListener('click', () => {
                if (typeof this.options.nextEpisodeCallback === 'function') {
                    this.options.nextEpisodeCallback();
                }
            });

            // Controle deslizante de volume
            this.setupVolumeSlider();

            // Barra de progresso
            this.setupProgressBar();
        }

        // Configuração de controles de teclado
        setupKeyboardControls() {
            // Só processa eventos de teclado se o player estiver em foco
            document.addEventListener('keydown', (e) => {
                const isPlayerFocused = this.elements.container === document.activeElement ||
                                        this.elements.container.contains(document.activeElement) ||
                                        this.state.fullscreen;

                if (!isPlayerFocused) return;

                switch (e.code) {
                    case 'Space':
                        e.preventDefault();
                        this.togglePlay();
                        break;
                    case 'ArrowRight':
                        e.preventDefault();
                        this.skip(10);
                        break;
                    case 'ArrowLeft':
                        e.preventDefault();
                        this.skip(-10);
                        break;
                    case 'ArrowUp':
                        e.preventDefault();
                        this.adjustVolume(0.1);
                        break;
                    case 'ArrowDown':
                        e.preventDefault();
                        this.adjustVolume(-0.1);
                        break;
                    case 'KeyM':
                        e.preventDefault();
                        this.toggleMute();
                        break;
                    case 'KeyF':
                        e.preventDefault();
                        this.toggleFullscreen();
                        break;
                    case 'Escape':
                        if (this.state.fullscreen) {
                            this.exitFullscreen();
                        } else if (this.state.settingsMenuVisible || this.state.subtitlesMenuVisible) {
                            this.hideAllMenus();
                        }
                        break;
                }
            });
        }

        // Configuração para controles de toque (mobile)
        setupTouchControls() {
            const videoEl = this.elements.video;
            const overlay = this.elements.overlay;

            let touchStartX = 0;
            let touchStartY = 0;
            let touchStartTime = 0;
            let touchStartVolume = 0;
            let touchStartCurrentTime = 0;
            let isSeeking = false;
            let isVolumeChange = false;
            let isTap = false;

            overlay.addEventListener('touchstart', (e) => {
                if (e.touches.length !== 1) return;

                touchStartX = e.touches[0].clientX;
                touchStartY = e.touches[0].clientY;
                touchStartTime = Date.now();
                touchStartVolume = videoEl.volume;
                touchStartCurrentTime = videoEl.currentTime;
                isSeeking = false;
                isVolumeChange = false;
                isTap = true;

                e.preventDefault();
            }, { passive: false });

            overlay.addEventListener('touchmove', (e) => {
                if (e.touches.length !== 1) return;

                const touchX = e.touches[0].clientX;
                const touchY = e.touches[0].clientY;
                const deltaX = touchX - touchStartX;
                const deltaY = touchY - touchStartY;
                const absDeltaX = Math.abs(deltaX);
                const absDeltaY = Math.abs(deltaY);

                if (!isSeeking && !isVolumeChange) {
                    // Determinar se é um movimento horizontal (seek) ou vertical (volume)
                    if (absDeltaX > 20 && absDeltaX > absDeltaY) {
                        isSeeking = true;
                        this.showControls();
                        videoEl.pause();
                    } else if (absDeltaY > 20 && absDeltaY > absDeltaX) {
                        isVolumeChange = true;
                        this.showControls();
                    }

                    if (isSeeking || isVolumeChange) {
                        isTap = false;
                    }
                }

                // Aplicar seek ou mudança de volume
                if (isSeeking) {
                    const seekDelta = (deltaX / overlay.clientWidth) * this.state.duration;
                    let newTime = touchStartCurrentTime + seekDelta;
                    newTime = Math.max(0, Math.min(this.state.duration, newTime));

                    // Atualizar visualmente
                    const newPercent = newTime / this.state.duration;
                    this.elements.progressCurrent.style.width = `${newPercent * 100}%`;
                    this.elements.progressHandle.style.left = `${newPercent * 100}%`;
                    this.elements.currentTime.textContent = this.formatTime(newTime);

                    // Mostrar indicação de seek
                    this.showNotification(`${this.formatTime(newTime)} / ${this.formatTime(this.state.duration)}`);
                }

                if (isVolumeChange) {
                    const volumeDelta = -deltaY / 150; // Ajuste a sensibilidade aqui
                    let newVolume = touchStartVolume + volumeDelta;
                    newVolume = Math.max(0, Math.min(1, newVolume));

                    this.setVolume(newVolume);

                    // Mostrar indicação de volume
                    const volumePercent = Math.round(newVolume * 100);
                    this.showNotification(`Volume: ${volumePercent}%`);
                }
            }, { passive: true });

            overlay.addEventListener('touchend', (e) => {
                const touchDuration = Date.now() - touchStartTime;

                if (isSeeking) {
                    const deltaX = e.changedTouches[0].clientX - touchStartX;
                    const seekDelta = (deltaX / overlay.clientWidth) * this.state.duration;
                    let newTime = touchStartCurrentTime + seekDelta;
                    newTime = Math.max(0, Math.min(this.state.duration, newTime));

                    this.seek(newTime);

                    if (this.state.playing) {
                        videoEl.play().catch(() => {});
                    }
                } else if (isTap && touchDuration < 300) {
                    this.togglePlay();
                    this.showControls();
                    this.resetControlsTimeout();
                }

                isSeeking = false;
                isVolumeChange = false;
                isTap = false;
            }, { passive: true });

            // Manipular double tap (avançar/retroceder)
            let lastTapTime = 0;
            let tapCount = 0;
            let tapX = 0;

            overlay.addEventListener('touchend', (e) => {
                const currentTime = Date.now();
                const tapDelay = currentTime - lastTapTime;

                if (tapDelay < 300) {
                    tapCount++;

                    if (tapCount === 2) {
                        const touch = e.changedTouches[0];
                        const screenWidth = overlay.clientWidth;
                        const touchX = touch.clientX;

                        // Double tap detectado
                        if (touchX < screenWidth / 2) {
                            // Lado esquerdo - retroceder 10s
                            this.skip(-10);
                            this.showNotification('Retroceder 10s');
                        } else {
                            // Lado direito - avançar 10s
                            this.skip(10);
                            this.showNotification('Avançar 10s');
                        }

                        tapCount = 0;
                    }
                } else {
                    tapCount = 1;
                    tapX = e.changedTouches[0].clientX;
                }

                lastTapTime = currentTime;
            }, { passive: true });
        }

        // Configuração de legendas
        setupSubtitles() {
            if (this.options.subtitles && this.options.subtitles.length > 0) {
                // Adicionar elementos de track para legendas
                this.options.subtitles.forEach((subtitle, index) => {
                    const track = document.createElement('track');
                    track.kind = 'subtitles';
                    track.label = subtitle.label || `Legenda ${index + 1}`;
                    track.srclang = subtitle.srclang || 'pt';
                    track.src = subtitle.src;
                    track.default = subtitle.default || false;
                    this.elements.video.appendChild(track);

                    // Adicionar opção ao menu de legendas
                    const option = document.createElement('button');
                    option.textContent = subtitle.label || `Legenda ${index + 1}`;
                    option.dataset.subtitle = index.toString();
                    if (subtitle.default) {
                        option.classList.add('active');
                    }
                    this.elements.subtitlesOptions.appendChild(option);
                });
            }
        }

        // Configuração do HLS para reprodução de streams
        setupHLS() {
            // Verifica se a URL do vídeo é um stream HLS (.m3u8)
            if (this.options.source && this.options.source.includes('.m3u8')) {
                // Verifica se o navegador suporta HLS nativamente
                if (this.elements.video.canPlayType('application/vnd.apple.mpegurl')) {
                    console.log('Este navegador suporta HLS nativamente');
                    // Alguns navegadores como Safari suportam HLS nativamente
                } else if (window.Hls) {
                    if (Hls.isSupported()) {
                        this.hls = new Hls(this.options.hlsConfig);
                        this.hls.loadSource(this.options.source);
                        this.hls.attachMedia(this.elements.video);
                        // Adiciona manipuladores de eventos do HLS
                        this.hls.on(Hls.Events.MANIFEST_PARSED, () => {
                            console.log('HLS manifest carregado');
                            // Configura as opções de qualidade no menu de configurações
                            const levels = this.hls.levels;
                            if (levels.length > 1) {
                                const qualitySection = this.elements.settingsMenu.querySelector('.ultra-player-quality-section');
                                const qualityOptions = qualitySection.querySelector('.ultra-player-quality-options');
                                // Limpa opções anteriores
                                qualityOptions.innerHTML = '';
                                // Adiciona opções de qualidade automática
                                const autoOption = document.createElement('button');
                                autoOption.textContent = 'Automático';
                                autoOption.dataset.level = '-1';
                                autoOption.classList.add('active');
                                autoOption.addEventListener('click', () => {
                                    this.setQuality(-1);
                                    this.toggleSettingsMenu();
                                });
                                qualityOptions.appendChild(autoOption);
                                // Adiciona opções para cada nível de qualidade
                                levels.forEach((level, index) => {
                                    const height = level.height;
                                    const option = document.createElement('button');
                                    option.textContent = `${height}p`;
                                    option.dataset.level = index.toString();
                                    option.addEventListener('click', () => {
                                        this.setQuality(index);
                                        this.toggleSettingsMenu();
                                    });
                                    qualityOptions.appendChild(option);
                                });
                                // Exibe a seção de qualidade
                                qualitySection.style.display = 'block';
                            }
                        });
                        // Gerenciar erros do HLS
                        this.hls.on(Hls.Events.ERROR, (event, data) => {
                            if (data.fatal) {
                                switch(data.type) {
                                    case Hls.ErrorTypes.NETWORK_ERROR:
                                        console.error('Erro fatal de rede', data);
                                        // Tenta recuperar erros de rede
                                        this.hls.startLoad();
                                        break;
                                    case Hls.ErrorTypes.MEDIA_ERROR:
                                        console.error('Erro fatal de mídia', data);
                                        this.hls.recoverMediaError();
                                        break;
                                    default:
                                        console.error('Erro fatal', data);
                                        this.showError('Erro ao carregar o vídeo: ' + data.details);
                                        this.destroy();
                                        break;
                                }
                            }
                        });
                    } else {
                        console.error('HLS.js não está disponível e este navegador não suporta HLS nativamente');
                        this.showError('Seu navegador não suporta a reprodução de streams HLS');
                    }
                } else {
                    console.error('HLS.js não está disponível e este navegador não suporta HLS nativamente');
                    this.showError('Seu navegador não suporta a reprodução de streams HLS');
                }
            }
        }

        // Configura um observador para redimensionamento para ajustar a UI
        setupResizeObserver() {
            if ('ResizeObserver' in window) {
                this.resizeObserver = new ResizeObserver(entries => {
                    for (const entry of entries) {
                        const width = entry.contentRect.width;
                        // Adapta a interface com base na largura
                        this.adaptUIToSize(width);
                    }
                });
                this.resizeObserver.observe(this.options.container);
            }
        }

        // Adapta a interface baseada no tamanho do container
        adaptUIToSize(width) {
            if (width < 600) {
                // Ajustes para telas pequenas
                this.elements.time?.classList.add('ultra-player-mobile');
                this.elements.volumeSlider?.classList.add('ultra-player-mobile');
            } else {
                // Ajustes para telas maiores
                this.elements.time?.classList.remove('ultra-player-mobile');
                this.elements.volumeSlider?.classList.remove('ultra-player-mobile');
            }
        }

        // Reproduz ou pausa o vídeo
        togglePlay() {
            const video = this.elements.video;
            if (video.paused || video.ended) {
                video.play().catch(error => {
                    console.error('Erro ao reproduzir vídeo:', error);
                    if (error.name === 'NotAllowedError') {
                        // Políticas de autoplay podem impedir a reprodução automática
                        // mas geralmente podemos reproduzir se o vídeo estiver mudo
                        this.showNotification('Reprodução automática bloqueada. Ativando mudo.');
                        this.elements.video.muted = true;
                        this.updateVolumeUI();
                        this.elements.video.play().catch(e => console.error('Falha na reprodução mesmo mudo:', e));
                    } else {
                        this.showError('Não foi possível iniciar a reprodução');
                    }
                });
            } else {
                video.pause();
            }
        }

        // Atualiza o estado visual dos botões de reprodução/pausa
        updatePlaybackState() {
            if (this.state.playing) {
                this.elements.playIcon.style.display = 'none';
                this.elements.pauseIcon.style.display = 'inline';
            } else {
                this.elements.playIcon.style.display = 'inline';
                this.elements.pauseIcon.style.display = 'none';
            }
        }

        // Avança ou retrocede no vídeo
        skip(seconds) {
            const video = this.elements.video;
            const newTime = video.currentTime + seconds;
            this.seek(Math.max(0, Math.min(video.duration, newTime)));
            // Mostra uma notificação
            const action = seconds > 0 ? 'Avançou' : 'Retrocedeu';
            this.showNotification(`${action} ${Math.abs(seconds)} segundos`);
        }

        // Define a posição atual do vídeo
        seek(time) {
            const video = this.elements.video;
            if (!isNaN(time) && isFinite(time)) {
                // Atualiza a UI antes da operação de seek para resposta mais responsiva
                const percent = time / video.duration;
                this.elements.progressCurrent.style.width = `${percent * 100}%`;
                this.elements.progressHandle.style.left = `${percent * 100}%`;
                this.elements.currentTime.textContent = this.formatTime(time);
                // Aplica o seek
                video.currentTime = time;
            }
        }

        // Muta ou desmuta o vídeo
        toggleMute() {
            const video = this.elements.video;
            if (!video.muted && video.volume > 0) {
                // Salva o volume atual antes de mutar, se não estiver já salvo
                this.state.lastVolume = video.volume;
            }
            // Alterna o estado de mudo
            video.muted = !video.muted;
            // Se estiver desmutando mas o volume for 0, restaura o volume anterior ou define para 30%
            if (!video.muted && video.volume === 0) {
                this.setVolume(this.state.lastVolume || 0.3);
            }
            this.updateVolumeUI();
        }

        // Define o volume do vídeo
        setVolume(volume) {
            const video = this.elements.video;
            // Garante que o volume está entre 0 e 1
            volume = Math.max(0, Math.min(1, volume));
            // Define o volume
            video.volume = volume;
            // Se o volume for definido para 0, muta o vídeo
            if (volume === 0) {
                video.muted = true;
            } else {
                video.muted = false;
            }
            // Atualiza a UI
            this.updateVolumeUI();
        }

        // Ajusta o volume relativamente ao valor atual
        adjustVolume(amount) {
            const video = this.elements.video;
            const newVolume = Math.max(0, Math.min(1, video.volume + amount));
            this.setVolume(newVolume);
            this.showNotification(`Volume: ${Math.round(newVolume * 100)}%`);
        }

        // Atualiza a interface de volume
        updateVolumeUI() {
            const video = this.elements.video;
            const volumePercent = video.muted ? 0 : video.volume;
            // Atualiza o slider de volume
            this.elements.volumeLevel.style.width = `${volumePercent * 100}%`;
            this.elements.volumeHandle.style.left = `${volumePercent * 100}%`;
            // Atualiza o ícone de volume
            this.elements.volumeHigh.style.display = 'none';
            this.elements.volumeMedium.style.display = 'none';
            this.elements.volumeLow.style.display = 'none';
            this.elements.volumeMute.style.display = 'none';
            if (video.muted || volumePercent === 0) {
                this.elements.volumeMute.style.display = 'inline';
            } else if (volumePercent < 0.3) {
                this.elements.volumeLow.style.display = 'inline';
            } else if (volumePercent < 0.7) {
                this.elements.volumeMedium.style.display = 'inline';
            } else {
                this.elements.volumeHigh.style.display = 'inline';
            }
        }

        // Entra ou sai do modo tela cheia
        toggleFullscreen() {
            if (this.state.fullscreen) {
                this.exitFullscreen();
            } else {
                this.enterFullscreen();
            }
        }

        // Entra no modo tela cheia
        enterFullscreen() {
            try {
                const container = this.options.container;

                // Verifica se já está em fullscreen
                if (this.state.fullscreen) {
                    return; // Já está em fullscreen, não precisa fazer nada
                }

                // Diferentes métodos para cada navegador
                if (container.requestFullscreen) {
                    container.requestFullscreen().catch(err => {
                        console.warn('Erro ao entrar em fullscreen:', err);
                        // Para alguns navegadores, o fullscreen precisa ser iniciado por uma ação do usuário
                        this.showNotification('Clique na tela para ativar o modo tela cheia', 3000);
                    });
                } else if (container.webkitRequestFullscreen) { /* Safari */
                    container.webkitRequestFullscreen();
                } else if (container.msRequestFullscreen) { /* IE11 */
                    container.msRequestFullscreen();
                } else if (container.mozRequestFullScreen) { /* Firefox */
                    container.mozRequestFullScreen();
                }

                this.state.fullscreen = true;
                this.updateFullscreenUI();

                if (this.options.onFullscreenChange) {
                    this.options.onFullscreenChange(true);
                }
            } catch (error) {
                console.error('Erro ao entrar em modo tela cheia:', error);
                this.state.fullscreen = false; // Garante que o estado está correto
                this.showNotification('Não foi possível entrar em modo tela cheia. Tente clicar no botão de tela cheia.', 3000);
            }
        }

        // Sai do modo tela cheia
        exitFullscreen() {
            try {
                // Verifica primeiro se o documento está ativo e em fullscreen antes de tentar sair
                if (document.fullscreenElement ||
                    document.webkitFullscreenElement ||
                    document.mozFullScreenElement ||
                    document.msFullscreenElement) {

                    if (document.exitFullscreen) {
                        document.exitFullscreen().catch(e => {
                            console.warn('Erro ao sair do modo tela cheia:', e);
                            // Continua mesmo com erro
                            this.state.fullscreen = false;
                            this.updateFullscreenUI();
                        });
                    } else if (document.webkitExitFullscreen) { /* Safari */
                        document.webkitExitFullscreen();
                    } else if (document.msExitFullscreen) { /* IE11 */
                        document.msExitFullscreen();
                    } else if (document.mozCancelFullScreen) { /* Firefox */
                        document.mozCancelFullScreen();
                    }
                }

                // Atualiza o estado independente do resultado da operação de saída
                this.state.fullscreen = false;
                this.updateFullscreenUI();

                if (this.options.onFullscreenChange) {
                    this.options.onFullscreenChange(false);
                }
            } catch (error) {
                console.error('Erro ao sair do modo tela cheia:', error);
                // Atualiza a UI mesmo com o erro
                this.state.fullscreen = false;
                this.updateFullscreenUI();
            }
        }

        // Atualiza a interface para modo tela cheia
        updateFullscreenUI() {
            if (this.state.fullscreen) {
                this.elements.fullscreenIcon.style.display = 'none';
                this.elements.fullscreenExitIcon.style.display = 'inline';
                this.options.container.classList.add('ultra-player-fullscreen');
            } else {
                this.elements.fullscreenIcon.style.display = 'inline';
                this.elements.fullscreenExitIcon.style.display = 'none';
                this.options.container.classList.remove('ultra-player-fullscreen');
            }
        }

        // Define a velocidade de reprodução
        setPlaybackSpeed(speed) {
            this.elements.video.playbackRate = speed;
            this.showNotification(`Velocidade: ${speed === 1.0 ? 'Normal' : speed + 'x'}`);
        }

        // Define a qualidade do vídeo (para streams HLS)
        setQuality(levelIndex) {
            if (this.hls && this.hls.levels && this.hls.levels.length > 1) {
                this.hls.currentLevel = levelIndex;
                // Mostra uma notificação com a qualidade selecionada
                let qualityText = 'Automático';
                if (levelIndex >= 0) {
                    const height = this.hls.levels[levelIndex].height;
                    qualityText = `${height}p`;
                }
                this.showNotification(`Qualidade: ${qualityText}`);
            }
        }

        // Alterna a visibilidade do menu de configurações
        toggleSettingsMenu() {
            const menu = this.elements.settingsMenu;
            if (window.getComputedStyle(menu).display === 'none') {
                this.hideAllMenus();
                menu.style.display = 'block';
                this.state.settingsMenuVisible = true;
                this.resetControlsTimeout();
            } else {
                menu.style.display = 'none';
                this.state.settingsMenuVisible = false;
            }
        }

        // Alterna a visibilidade do menu de legendas
        toggleSubtitlesMenu() {
            const menu = this.elements.subtitlesMenu;
            if (window.getComputedStyle(menu).display === 'none') {
                this.hideAllMenus();
                menu.style.display = 'block';
                this.state.subtitlesMenuVisible = true;
                this.resetControlsTimeout();
            } else {
                menu.style.display = 'none';
                this.state.subtitlesMenuVisible = false;
            }
        }

        // Esconde todos os menus
        hideAllMenus() {
            this.elements.settingsMenu.style.display = 'none';
            this.elements.subtitlesMenu.style.display = 'none';
            this.state.settingsMenuVisible = false;
            this.state.subtitlesMenuVisible = false;
        }

        // Mostra a interface para navegação para o próximo episódio
        showEpisodeNavigation() {
            // Só mostra a navegação de episódio se tivermos um callback para o próximo episódio
            if (this.options.nextEpisodeCallback && typeof this.options.nextEpisodeCallback === 'function') {
                this.elements.episodeNavigation.style.display = 'block';
                this.elements.nextTitle.textContent = 'Próximo episódio';

                // Inicia a contagem regressiva para o próximo episódio
                let countdown = 10;
                this.elements.countdown.textContent = countdown;
                // Limpa o intervalo anterior se existir
                if (this.state.countdownInterval) {
                    clearInterval(this.state.countdownInterval);
                }
                // Configurar a contagem regressiva
                this.state.countdownInterval = setInterval(() => {
                    countdown--;
                    this.elements.countdown.textContent = countdown;
                    if (countdown <= 0) {
                        clearInterval(this.state.countdownInterval);
                        if (typeof this.options.nextEpisodeCallback === 'function') {
                            this.options.nextEpisodeCallback();
                        }
                    }
                }, 1000);
            }
        }

        // Esconde a interface de navegação para o próximo episódio
        hideEpisodeNavigation() {
            if (this.state.countdownInterval) {
                clearInterval(this.state.countdownInterval);
                this.state.countdownInterval = null;
            }
            this.elements.episodeNavigation.style.display = 'none';
        }

        // Exibe os controles do player
        showControls() {
            this.elements.controls.classList.remove('hidden');
            this.elements.titleContainer.style.opacity = '1';
            this.state.controlsVisible = true;
        }

        // Esconde os controles do player
        hideControls() {
            if (!this.state.playing) return; // Não esconde os controles se o vídeo estiver pausado
            if (this.state.settingsMenuVisible || this.state.subtitlesMenuVisible) return; // Não esconde se os menus estiverem abertos
            this.elements.controls.classList.add('hidden');
            this.elements.titleContainer.style.opacity = '0';
            this.state.controlsVisible = false;
        }

        // Reinicia o temporizador para esconder os controles
        resetControlsTimeout() {
            // Limpa timeout anterior
            if (this.state.controlsTimeout) {
                clearTimeout(this.state.controlsTimeout);
            }
            // Configura novo timeout para esconder os controles
            this.state.controlsTimeout = setTimeout(() => {
                this.hideControls();
            }, 3000); // Esconde após 3 segundos de inatividade
        }

        // Exibe mensagem de erro
        showError(message) {
            if (!this.elements || !this.elements.errorMessage) return;

            this.elements.errorMessage.textContent = message;
            if (this.elements.error) {
                this.elements.error.style.display = 'flex';
            }
        }

        // Esconde mensagem de erro
        hideError() {
            if (!this.elements || !this.elements.error) return;

            this.elements.error.style.display = 'none';
        }

        // Tenta reproduzir novamente após erro
        retry() {
            this.hideError();
            if (this.hls) {
                this.hls.destroy();
                this.setupHLS();
            } else {
                this.elements.video.load();
                this.elements.video.play().catch(error => {
                    console.error('Erro ao tentar novamente:', error);
                    this.showError('Não foi possível reproduzir o vídeo. Por favor, tente novamente mais tarde.');
                });
            }
        }

        // Exibe uma notificação temporária
        showNotification(message, duration = 2000) {
            if (!this.elements || !this.elements.notification) return;

            const notification = this.elements.notification;
            notification.textContent = message;
            notification.style.display = 'block';
            notification.style.opacity = '1';

            // Limpa timeout anterior
            if (this.state.notificationTimeout) {
                clearTimeout(this.state.notificationTimeout);
            }

            // Configura novo timeout para esconder a notificação
            this.state.notificationTimeout = setTimeout(() => {
                if (!this.elements || !this.elements.notification) return;

                notification.style.opacity = '0';
                setTimeout(() => {
                    if (!this.elements || !this.elements.notification) return;
                    notification.style.display = 'none';
                }, 300); // Aguarda a transição terminar
            }, duration);
        }

        // Formata segundos para formato MM:SS ou HH:MM:SS
        formatTime(seconds) {
            if (isNaN(seconds) || !isFinite(seconds)) return '00:00';

            // Arredonda os segundos para evitar números com muitas casas decimais
            seconds = Math.floor(seconds);

            let hours = Math.floor(seconds / 3600);
            let minutes = Math.floor((seconds % 3600) / 60);
            let secs = Math.floor(seconds % 60);

            // Formata com zeros à esquerda
            if (hours > 0) {
                return `${hours}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
            } else {
                return `${minutes}:${secs.toString().padStart(2, '0')}`;
            }
        }

        // Método para registrar ouvintes de eventos
        on(eventName, callback) {
            if (!this.eventListeners) {
                this.eventListeners = {};
            }
            if (!this.eventListeners[eventName]) {
                this.eventListeners[eventName] = [];
            }
            this.eventListeners[eventName].push(callback);

            // Adiciona também como listener para eventos nativos do vídeo
            if (this.elements && this.elements.video && typeof this.elements.video.addEventListener === 'function') {
                if (['play', 'pause', 'ended', 'timeupdate', 'error', 'volumechange', 'seeking', 'seeked'].includes(eventName)) {
                    this.elements.video.addEventListener(eventName, (e) => callback(e));
                }
            }

            return this; // Para encadeamento de métodos
        }

        // Método para disparar eventos
        trigger(eventName, data) {
            if (this.eventListeners && this.eventListeners[eventName]) {
                this.eventListeners[eventName].forEach(callback => {
                    callback(data);
                });
            }
            return this;
        }

        // Exibe uma tela informando que o conteúdo requer assinatura premium
        showPremiumRequiredScreen() {
            // Esta função foi removida para eliminar a dependência do route()
            console.warn('Função showPremiumRequiredScreen não está mais disponível');
        }

        // Ajusta a cor para mais claro ou mais escuro para efeitos de hover
        adjustColor(color, amount) {
            // Remove o # se existir
            color = color.replace('#', '');

            // Converte para RGB
            let r = parseInt(color.substring(0, 2), 16);
            let g = parseInt(color.substring(2, 4), 16);
            let b = parseInt(color.substring(4, 6), 16);

            // Ajusta cada componente
            r = Math.max(0, Math.min(255, r + amount));
            g = Math.max(0, Math.min(255, g + amount));
            b = Math.max(0, Math.min(255, b + amount));

            // Converte de volta para hex
            return "#" + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1);
        }

        // Libera recursos e destrói o player
        destroy() {
            // Criamos uma flag para indicar que a destruição está em andamento
            this.isBeingDestroyed = true;

            try {
                // Remover todos os event listeners do vídeo primeiro
                // Fazemos isso antes de pausar para evitar erros de eventos disparados durante a destruição
                if (this.elements && this.elements.video) {
                    // Clonando o elemento para remover todos os listeners de forma eficaz
                    const oldVideo = this.elements.video;
                    if (oldVideo.parentNode) {
                        const newVideo = oldVideo.cloneNode(false);
                        oldVideo.parentNode.replaceChild(newVideo, oldVideo);
                    }

                    // Agora que os listeners foram removidos, podemos lidar com o vídeo
                    try {
                        this.elements.video.pause();
                        this.elements.video.removeAttribute('src');
                        this.elements.video.load();
                    } catch (e) {
                        console.warn('Erro ao limpar elemento de vídeo:', e);
                    }
                }

                // Limpar todos os timeouts
                if (this.state) {
                    if (this.state.controlsTimeout) {
                        clearTimeout(this.state.controlsTimeout);
                    }

                    if (this.state.notificationTimeout) {
                        clearTimeout(this.state.notificationTimeout);
                    }

                    if (this.state.countdownInterval) {
                        clearInterval(this.state.countdownInterval);
                    }
                }

                // Sair do modo tela cheia se estiver ativo
                if (this.state && this.state.fullscreen) {
                    try {
                        this.exitFullscreen();
                    } catch (e) {
                        console.warn('Erro ao sair do fullscreen durante destroy:', e);
                    }
                }

                // Desconectar o ResizeObserver
                if (this.resizeObserver) {
                    this.resizeObserver.disconnect();
                    this.resizeObserver = null;
                }

                // Destruir instância HLS se existir
                if (this.hls) {
                    this.hls.destroy();
                    this.hls = null;
                }

                // Limpar o conteúdo do container
                if (this.options && this.options.container) {
                    // Armazenamos uma referência separada para o container
                    // pois vamos limpar this.elements logo em seguida
                    const container = this.options.container;

                    // Remover todos os event listeners do container e seus filhos
                    // A maneira mais segura é clonar e substituir
                    const oldContainer = container;
                    const parentNode = oldContainer.parentNode;

                    if (parentNode) {
                        const newContainer = oldContainer.cloneNode(false); // shallow clone
                        parentNode.replaceChild(newContainer, oldContainer);
                        this.options.container = newContainer;
                    }

                    container.innerHTML = '';
                }
            } catch (error) {
                console.error('Erro durante a limpeza do player:', error);
            }

            // Limpar referências para ajudar o GC
            this.elements = null;
        }
    }

    // Exporta a classe se estivermos em um ambiente que suporta módulos
    if (typeof module !== 'undefined' && typeof module.exports !== 'undefined') {
        module.exports = UltraPlayer;
    }

    // Expõe a classe para uso global
    window.UltraPlayer = UltraPlayer;

})(window);

/**
 * Gerenciador de posição do vídeo para UltraFlix
 * Responsável por armazenar e recuperar as posições de reprodução de filmes e episódios
 */

// Verificar se VideoPositionManager já foi declarado para evitar redeclaração
if (typeof window.VideoPositionManager === 'undefined') {
    class VideoPositionManager {
        constructor() {
            this.baseUrl = window.appUrl || document.querySelector('meta[name="base-url"]')?.content || '';
            this.csrfToken = document.querySelector('meta[name="csrf-token"]')?.content || '';
            this.userId = document.querySelector('meta[name="user-id"]')?.content || '';

            // Adicionar nomes de rotas
            this.routes = {
                savePosition: 'continueWatching',
                getPosition: 'continueWatching.getPosition'
            };
        }

        /**
         * Salva a posição atual do vídeo
         * @param {Object} data Objeto contendo os dados do vídeo
         * @param {number} data.item_id - ID do item (filme, série ou anime)
         * @param {string} data.type - Tipo do conteúdo (movie, serie, anime)
         * @param {number} data.position - Posição atual em segundos
         * @param {number} data.duration - Duração total em segundos
         * @param {number} [data.episode_id] - ID do episódio (apenas para séries e animes)
         * @param {number} [data.season_number] - Número da temporada (apenas para séries e animes)
         * @param {number} [data.episode_number] - Número do episódio (apenas para séries e animes)
         * @returns {Promise} Promise com resultado da operação
         */
        savePosition(data) {
            // Adicionar o user_id aos dados
            const saveData = {
                ...data,
                user_id: this.userId
            };

            // Salvar no localStorage também como backup
            if (data.type === 'movie') {
                localStorage.setItem(`moviePosition-${data.item_id}`, data.position);
            } else if (data.episode_id) {
                localStorage.setItem(`videoPosition-${data.episode_id}`, data.position);
            }

            // Enviar para o servidor usando a rota nomeada
            const url = `${this.baseUrl}/continue-watching`;

            return fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken
                },
                body: JSON.stringify(saveData)
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`Erro ao salvar posição: ${response.status}`);
                }
                return response.json();
            });
        }

        /**
         * Carrega a posição salva para um item específico
         * @param {Object} query Parâmetros para buscar a posição
         * @param {number} query.item_id - ID do item
         * @param {string} query.type - Tipo do conteúdo
         * @param {number} [query.episode_id] - ID do episódio (apenas para séries e animes)
         * @returns {Promise<Object>} Promise com a posição salva
         */
        loadPosition(query) {
            // Primeiro tentamos obter do localStorage para resposta rápida
            let localPosition = null;

            if (query.type === 'movie') {
                localPosition = localStorage.getItem(`moviePosition-${query.item_id}`);
            } else if (query.episode_id) {
                localPosition = localStorage.getItem(`videoPosition-${query.episode_id}`);
            }

            // Criamos o objeto de consulta para a API
            const queryParams = new URLSearchParams({
                item_id: query.item_id,
                type: query.type
            });

            if (query.episode_id) {
                queryParams.append('episode_id', query.episode_id);
            }

            // Consultamos o servidor usando a rota nomeada
            const url = `${this.baseUrl}/continue-watching/get-position?${queryParams.toString()}`;

            return fetch(url, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': this.csrfToken
                }
            })
            .then(response => {
                if (!response.ok) {
                    // Se o servidor falhar, usamos o localStorage
                    if (localPosition) {
                        return {
                            position: parseFloat(localPosition),
                            source: 'localStorage'
                        };
                    }
                    throw new Error(`Erro ao carregar posição: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                // Se o servidor retornou dados, usamos essa posição
                if (data && data.position) {
                    return data;
                }

                // Caso contrário, usamos o localStorage se disponível
                if (localPosition) {
                    return {
                        position: parseFloat(localPosition),
                        source: 'localStorage'
                    };
                }

                // Se não há posição salva em nenhum lugar
                return { position: 0 };
            })
            .catch(error => {
                console.warn('Erro ao recuperar posição:', error);

                // Em caso de erro, retornamos a posição do localStorage, se existir
                if (localPosition) {
                    return {
                        position: parseFloat(localPosition),
                        source: 'localStorage'
                    };
                }

                // Ou zero se não houver nada salvo
                return { position: 0 };
            });
        }

        /**
         * Verifica se devemos continuar de uma posição salva
         * @param {number} savedPosition - Posição salva em segundos
         * @param {number} duration - Duração total do vídeo em segundos
         * @returns {boolean} Verdadeiro se deve continuar da posição salva
         */
        shouldResumeFrom(savedPosition, duration) {
            // Não continuar se não houver posição ou duração
            if (!savedPosition || !duration) return false;

            // Não continuar se já estiver perto do fim (95% ou mais)
            if (savedPosition / duration > 0.95) return false;

            // Não continuar se estiver muito no início (menos de 30 segundos)
            if (savedPosition < 30) return false;

            return true;
        }

        /**
         * Formata segundos para formato MM:SS
         * @param {number} seconds - Segundos para formatar
         * @returns {string} Tempo formatado
         */
        formatTime(seconds) {
            if (!seconds || isNaN(seconds)) return '00:00';

            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return `${mins}:${secs < 10 ? '0' : ''}${secs}`;
        }
    }

    // Exportamos a instância global
    window.VideoPositionManager = new VideoPositionManager();
}

// Função auxiliar para mostrar o modal de confirmação para continuar assistindo
// Esta função é usada tanto para filmes quanto para séries
function showContinueWatchingPrompt(savedPosition, onContinue, onRestart) {
    // Verificar se já existe um prompt para evitar duplicação
    if (document.getElementById('continue-watching-prompt')) {
        return;
    }

    // Criar o modal
    const modal = document.createElement('div');
    modal.className = 'fixed inset-0 z-[999] flex items-center justify-center bg-black/80';
    modal.id = 'continue-watching-prompt';
    modal.style.opacity = '0';
    modal.style.transition = 'opacity 0.3s ease';

    // Converter a posição salva para formato legível
    const formattedTime = typeof window.VideoPositionManager !== 'undefined'
        ? window.VideoPositionManager.formatTime(savedPosition)
        : formatFallbackTime(savedPosition);

    // Conteúdo do modal
    modal.innerHTML = `
        <div class="w-full max-w-md p-6 mx-4 rounded-lg bg-zinc-900 shadow-2xl">
            <h3 class="mb-4 text-xl font-medium text-white">Continuar Assistindo</h3>
            <p class="mb-6 text-gray-300">Você assistiu até ${formattedTime}. Deseja continuar de onde parou?</p>
            <div class="flex justify-end space-x-3">
                <button id="restart-btn" class="px-4 py-2 text-sm font-medium text-white transition bg-zinc-700 rounded-md hover:bg-zinc-600">
                    Começar do início
                </button>
                <button id="continue-btn" class="px-4 py-2 text-sm font-medium text-white transition rounded-md bg-primary hover:bg-green-600">
                    Continuar
                </button>
            </div>
        </div>
    `;

    // Adicionar ao DOM
    document.body.appendChild(modal);

    // Fade in
    setTimeout(() => {
        modal.style.opacity = '1';
    }, 10);

    // Adicionar eventos aos botões
    document.getElementById('continue-btn').addEventListener('click', () => {
        modal.style.opacity = '0';
        setTimeout(() => {
            modal.remove();
            if (onContinue) onContinue();
        }, 300);
    });

    document.getElementById('restart-btn').addEventListener('click', () => {
        modal.style.opacity = '0';
        setTimeout(() => {
            modal.remove();
            if (onRestart) onRestart();
        }, 300);
    });

    // Adicionar timeout para auto-continuar após 10 segundos
    setTimeout(() => {
        if (document.getElementById('continue-watching-prompt')) {
            modal.style.opacity = '0';
            setTimeout(() => {
                modal.remove();
                if (onContinue) onContinue();
            }, 300);
        }
    }, 10000);

    // Função de fallback para formatar tempo caso VideoPositionManager não esteja disponível
    function formatFallbackTime(seconds) {
        if (!seconds || isNaN(seconds)) return '00:00';
        const mins = Math.floor(seconds / 60);
        const secs = Math.floor(seconds % 60);
        return `${mins}:${secs < 10 ? '0' : ''}${secs}`;
    }
}

// Se já existe no escopo global, não redefine
if (typeof window.showContinueWatchingPrompt === 'undefined') {
    window.showContinueWatchingPrompt = showContinueWatchingPrompt;
}
