/**
 * Gerenciamento otimizado de carrosséis
 * Este script utiliza lazy initialization para carregar carrosséis à medida que se tornam visíveis
 */

(function() {
    // Evitar execução duplicada
    if (window.carouselInitialized) return;
    window.carouselInitialized = true;

    // Armazenar referências de carrosséis inicializados
    const initializedCarousels = {};

    // Verificar se Splide está disponível
    if (typeof Splide === 'undefined') {
        console.error('Splide não está carregado. Carrosséis não serão inicializados.');
        return;
    }

    /**
     * Inicializa um carrossel quando ele se torna visível
     * @param {string} carouselId - ID do elemento do carrossel
     * @param {Object} config - Configuração do carrossel
     */
    function initCarousel(carouselId, config) {
        if (!carouselId || initializedCarousels[carouselId]) return;

        const element = document.getElementById(carouselId);
        if (!element) return;

        try {
            console.log(`Inicializando carrossel: ${carouselId}`);
            const splide = new Splide(`#${carouselId}`, config);
            splide.mount();
            initializedCarousels[carouselId] = splide;

            // Personalizar setas
            customizeArrows(carouselId, splide);

            return splide;
        } catch (error) {
            console.error(`Erro ao inicializar carrossel ${carouselId}:`, error);
            return null;
        }
    }

    /**
     * Personaliza as setas de navegação do carrossel
     * @param {string} carouselId - ID do elemento do carrossel
     * @param {Splide} splide - Instância do carrossel
     */
    function customizeArrows(carouselId, splide) {
        const element = document.getElementById(carouselId);
        if (!element) return;

        // Substituir os ícones das setas
        const prevArrows = element.querySelectorAll('.splide__arrow--prev');
        const nextArrows = element.querySelectorAll('.splide__arrow--next');

        prevArrows.forEach(arrow => {
            arrow.innerHTML = '<i class="fas fa-chevron-left"></i>';
        });

        nextArrows.forEach(arrow => {
            arrow.innerHTML = '<i class="fas fa-chevron-right"></i>';
        });

        // Função para atualizar o estado das setas
        function updateArrowState() {
            const prevButton = element.querySelector('.splide__arrow--prev');
            const nextButton = element.querySelector('.splide__arrow--next');

            if (prevButton && nextButton && splide) {
                prevButton.classList.toggle('is-disabled', splide.index === 0);
                nextButton.classList.toggle('is-disabled',
                    splide.index >= splide.length - splide.options.perPage);
            }
        }

        // Atualizar setas quando o carrossel muda
        splide.on('moved', updateArrowState);
        splide.on('resized', updateArrowState);

        // Atualização inicial
        updateArrowState();

        // Melhorar performance de eventos de clique
        element.querySelectorAll('.splide__arrow').forEach(arrow => {
            arrow.addEventListener('click', function(e) {
                if (arrow.classList.contains('is-disabled')) {
                    e.preventDefault();
                    e.stopPropagation();
                    return false;
                }

                e.stopPropagation();

                // Pequena otimização para evitar múltiplos cliques rápidos
                if (arrow.dataset.processing) return;
                arrow.dataset.processing = true;

                setTimeout(() => {
                    delete arrow.dataset.processing;
                }, 300);

                if (splide) {
                    if (arrow.classList.contains('splide__arrow--next')) {
                        splide.go('>');
                    } else {
                        splide.go('<');
                    }

                    // Atualizar estados após mudança
                    setTimeout(updateArrowState, 50);
                }
            });
        });
    }

    /**
     * Observa elementos e inicializa carrosséis quando se tornam visíveis
     */
    function observeCarousels() {
        // Verificar suporte a IntersectionObserver
        if (!('IntersectionObserver' in window)) {
            // Fallback para navegadores sem suporte
            initializeAllCarousels();
            return;
        }

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const element = entry.target;
                    const id = element.id;

                    let config;
                    if (id.includes('continue-watching') || id.includes('choosed') ||
                        id.includes('latest-episodes')) {
                        config = window.wideCarouselConfig;
                    } else if (id.includes('top10')) {
                        config = window.top10CarouselConfig;
                    } else {
                        config = window.carouselConfig;
                    }

                    // Inicializar com um pequeno atraso para melhorar o rendering
                    setTimeout(() => {
                        initCarousel(id, config);
                    }, 50);

                    // Parar de observar após inicialização
                    observer.unobserve(element);
                }
            });
        }, {
            rootMargin: '200px 0px',
            threshold: 0.01
        });

        // Começar observando todos os carrosséis não inicializados
        document.querySelectorAll('.splide:not(.is-initialized)').forEach(carousel => {
            observer.observe(carousel);
        });
    }

    /**
     * Inicializa todos os carrosséis de uma vez (fallback)
     */
    function initializeAllCarousels() {
        document.querySelectorAll('.splide:not(.is-initialized)').forEach(carousel => {
            const id = carousel.id;
            if (!id) return;

            let config;
            if (id.includes('continue-watching') || id.includes('choosed') ||
                id.includes('latest-episodes')) {
                config = window.wideCarouselConfig;
            } else if (id.includes('top10')) {
                config = window.top10CarouselConfig;
            } else {
                config = window.carouselConfig;
            }

            initCarousel(id, config);
        });
    }

    // Expor funções ao escopo global para uso em outros scripts
    window.carouselManager = {
        initCarousel,
        customizeArrows,
        refreshAllCarousels: function() {
            Object.values(initializedCarousels).forEach(splide => {
                if (splide && typeof splide.refresh === 'function') {
                    splide.refresh();
                }
            });
        }
    };

    // Iniciar observação após carregar o documento
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', observeCarousels);
    } else {
        // DOMContentLoaded já disparou
        observeCarousels();
    }
})();
