<?php

namespace App\Http\Controllers;

use Illuminate\Validation\Rule;
use App\Http\Requests\AvatarRequest;
use App\Http\Requests\PasswordUpdateRequest;
use App\Http\Requests\PasswordAppRequest;
use App\Http\Requests\DeviceStoreRequest;
use App\Http\Requests\UserRequest;
use Illuminate\Http\Request;
use App\Http\Requests\UserRequestStore;
use App\Http\Requests\UserUpdateRequest;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Laravel\Passport\Client;
use App\Setting;
use App\Profile;
use App\Device;
use Illuminate\Support\Carbon;
use Sentinel;
use App\Services\BackblazeService;
use Firebase\JWT\JWT;
use Firebase\JWT\Key;

class UserController extends Controller
{
    protected $backblaze;

    public function __construct(BackblazeService $backblaze)
    {
        $this->middleware('doNotCacheResponse', ['only' => ['checkExpirationDate']]);
        $this->backblaze = $backblaze;
    }


    const MESSAGE = "successfully updated";



    public function showUser($id)
    {

        $user = User::query()
        ->where('id', '=', $id)->select(['id','name'])->get();

        return response()->json($user);


    }

    public function deleteInactiveAccounts()
    {
        // Data atual
        $now = Carbon::now();

        // Data limite para contas expiradas (30 dias atrás)
        $thresholdDate = $now->subDays(30);

        // Data limite para ignorar contas recém-criadas (15 dias atrás)
        $ignoreDate = $now->subDays(15);

        // Selecionar usuários que atendem aos critérios
        $users = User::where('premuim', '!=', 1)
            ->where(function ($query) use ($thresholdDate) {
                $query->whereNull('expired_in')
                      ->orWhere('expired_in', '<', $thresholdDate);
            })
            ->where('created_at', '<', $ignoreDate)
            ->get();

        // Deletar perfis relacionados e depois os usuários
        foreach ($users as $user) {
            $user->profiles()->delete();
            $user->delete();
        }

        return response()->json(['message' => 'Inactive accounts and their profiles deleted']);
    }

    public function index(Request $request)
    {
        $query = User::query();

        if ($request->filled('search')) {
            $query->where('email', 'like', '%' . $request->search . '%');
        }

        $users = $query->paginate(10);

        return view('ativar_tv', compact('users'));
    }

    public function updateTv(Request $request, $id)
    {
        $user = User::findOrFail($id);

        $tv_access = $request->has('tv_access') ? 1 : 0; // Isso verifica se o checkbox 'tv_access' foi marcado.

        $expired_in_tv = $user->expired_in_tv; // Assume a data atual de expiração.

        // Se o checkbox 'add_30_days' foi marcado, adicione 30 dias à data de expiração.
        if ($request->has('add_30_days')) {
            $expired_in_tv = Carbon::parse($user->expired_in_tv)->addDays(30);
        }

        // Atualiza o usuário.
        $user->update([
            'tv_access' => $tv_access,
            'expired_in_tv' => $expired_in_tv,
        ]);

        return back()->with('success', 'Usuário atualizado com sucesso!');
        }
    
    public function keepOnlyOldestProfileForAllUsers()
{
    $users = User::with('profiles')->get(); // Carrega antecipadamente os perfis para otimizar

    foreach ($users as $user) {
        // Obtenha todos os IDs de perfis, exceto o mais antigo
        $profileIdsToDelete = $user->profiles()->orderBy('created_at', 'asc')
                                    ->pluck('id') // Obtém apenas os IDs
                                    ->slice(1); // Ignora o primeiro perfil (o mais antigo)

        // Exclui todos os perfis obtidos, exceto o mais antigo
        if ($profileIdsToDelete->isNotEmpty()) {
            $user->profiles()->whereIn('id', $profileIdsToDelete)->delete();
        }
    }

    return response()->json(['message' => 'All users have been processed, and only the oldest profiles have been kept.']);
}

    
// No seu controlador ou onde a função getAvatarList será usada
public function getAvatarList()
{
    // Cria uma instância do serviço Backblaze
    $backblazeService = new \App\Services\BackblazeService();
    
    // Obtém a lista de arquivos do serviço Backblaze
    $files = $backblazeService->listFiles();

    // Verifica se a resposta contém arquivos válidos
    if (!is_array($files)) {
        return []; // Retorna um array vazio se a resposta não for válida
    }

    $baseUrl = 'https://avatar.payutx.com/file/';
    $bucketName = env('BACKBLAZE_BUCKET_NAME');
    $avatars = [];

    // Itera sobre os arquivos retornados na resposta
    foreach ($files as $file) {
        // Verifica se o arquivo tem um nome de arquivo válido
        if (isset($file['fileName'])) {
            $avatars[] = $baseUrl . $bucketName . '/' . $file['fileName'];
        }
    }

    // Embaralha os avatares para dar uma ordem aleatória
    shuffle($avatars);

    return $avatars; // Retorna a lista de URLs dos avatares
}



    public function devices (Request $request){


        $auth = Auth()->user();

        $user = Device::query()->where('user_id', '=', $auth->id)->get();


         return response()->json(['devices' => $user], 200);
     }

    public function createDevice (Request $request){


            $user = Auth()->user();


            $dev = Device::where('serial_number', $request->serial_number)->
            where('user_id', $user->id)->first();


            if (!$dev) {
            
            $this->validate($request, [
            'serial_number' => [
            'required',
            ],
            ]);


   
        $device = new Device();
        $device->name = $request->name;
        $device->model = $request->model;
        $device->serial_number = $request->serial_number;
        $device->user_id = $user->id;
        $user->devices()->save($device);

                
            }

        
            
        if ($device != null) {
        
            $data = ['status' => 200, 'message' => 'Device successfully Attached',];
        } else {
            $data = ['status' => 400, 'message' => 'Device Already Attached',];
        }

            return response()->json($data, 200);
       
     }

     
     public function deleteDevice ($id,Request $request){


        $user = Auth()->user();

        $deleteProfile = Device::find($id)->delete();


        if ($deleteProfile != null) {
        
            $data = ['status' => 200, 'message' => 'successfully removed',];
        } else {
            $data = ['status' => 400, 'message' => 'could not be deleted',];
        }

         return response()->json($data, 200);
     }
     
     public function createProfilex(Request $request) {
        // Assume que a autenticação já foi tratada pelo middleware
        $user = Auth::user(); // Obtém o usuário autenticado

        // Validação dos dados do pedido
        $request->validate([
            'name' => 'required|string|max:255',
            'avatar' => 'nullable|url', // Garante que, se fornecido, o avatar deve ser uma URL válida
        ]);

        // Verifica o limite de perfis
        if ($user->profiles()->count() >= 5) {
            return response()->json([
                'message' => 'Não é permitido criar mais de 2 perfis.'
            ], 403);
        }

        // Cria um novo perfil e associa ao usuário autenticado
        $profile = new Profile([
            'name' => $request->name,
            'avatar' => $request->avatar ?? '/api/avatars/image/avatar_default.png', // Usa um avatar padrão se nenhum for fornecido
        ]);

        $user->profiles()->save($profile);

        return response()->json([
            'message' => 'Perfil criado com sucesso.',
            'profile' => $profile
        ], 201);
    }



    public function createProfile(Request $request) {
    $this->validate($request, [
        'name' => 'required',
    ]);

    $user = Auth()->user();

    $profileCount = $user->profiles()->count();
    if ($profileCount >= 2) {
        return response()->json([
            'status' => 403,
            'message' => 'Não é permitido criar mais de 2 perfis.',
        ], 403);
    }

    $profile = new Profile();
    $profile->name = $request->name;
    if ($request->avatar == null) {
        $profile->avatar = $request->root() . '/api/avatars/image/avatar_default.png';
    } else {
        
        $profile->avatar = $request->avatar;
    }


    $user->profiles()->save($profile);

    if ($profile != null) {
        $data = [
            'status' => 200,
            'message' => 'Perfil adicionado com sucesso.',
        ];
    } else {
        $data = [
            'status' => 400,
            'message' => 'O perfil não pôde ser criado.',
        ];
    }

    return response()->json($data, $data['status']);
}

   


     public function showDefaultAvatar($filename)
    {
   
       $image = Storage::disk('avatars')->get($filename);
   
       $mime = Storage::disk('avatars')->mimeType($filename);
   
       return (new Response($image, 200))->header('Content-Type', $mime);
    }

     public function deleteProfile (Request $request){


        $user = Auth()->user();


        $deleteProfile = Profile::find($request->profile_id)->delete();


        if ($deleteProfile != null) {
        
            $data = ['status' => 200, 'message' => 'successfully removed',];
        } else {
            $data = ['status' => 400, 'message' => 'could not be deleted',];
        }

         return response()->json($data, 200);
     }
   


    public function QrAutoGenerate(Request $request)
	{	
		$result=0;
		if ($request->action = 'updateqr') {
			$user = Sentinel::getUser();
			if ($user) {
				$qrLogin=bcrypt($user->personal_number.$user->email.str_random(40));
		        $user->QRpassword= $qrLogin;
		        $user->update();
		        $result=1;
			}
		
		}
		
        return $result;
	}

    public function checkPremium($id)
    {
        $user = User::find($id);
        // Verifica se o usuário existe, é premium e a data de expiração é no futuro
        if ($user && $user->premuim == 1 && new Carbon($user->expired_in) > Carbon::now()) {
            return response()->json(['isPremium' => true]);
        } else {
            return response()->json(['isPremium' => false]);
        }
    }

    public function checkPremiumTV($id)
    {
        $user = User::find($id);
        // Verifica se o usuário existe, é premium e a data de expiração é no futuro
        if ($user && $user->tv_access == 1 && new Carbon($user->expired_in_tv) > Carbon::now()) {
            return response()->json(['isPremium' => true]);
        } else {
            return response()->json(['isPremium' => false]);
        }
    }


    public function checkExpirationDate(Request $request)
{
    $user = Auth::user();

    if (!$user || !$user->expired_in) {
        return response()->json(['status' => 400, 'error' => 'User not found or expiration date not set'], 400);
    }

    $expirationDate = Carbon::parse($user->expired_in);
    $currentDate = Carbon::now();

    $data = [
        'status' => 200,
        'subscription' => $currentDate->gte($expirationDate) ? 'expired' : 'notexpired'
    ];

    return response()->json($data, 200);
}




    public function create(Request $request)
    {

        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:6'
        ]);

        $user = User::create([
            'name' => request('name'),
            'email' => request('email'),
            'premuim' => false,
            'password' => bcrypt(request('password'))


        ]);

        return $this->issueToken($request, 'password');

    }





    // returns the authenticated user for admin panel

    public function data()
    {

    
        $user = Auth()->user();
        return response()
        ->json( $user, 200);

    
    }




    public function allusers()
    {

        return response()->json(User::orderByDesc('created_at')
        ->paginate(12), 200);
    }


    // return the logo checking the format
    public function showAvatar()

    {
        if (Storage::disk('public')->exists('users/users.jpg')) {
            $image = Storage::disk('public')->get('users/users.jpg');
            $mime = Storage::disk('public')->mimeType('/users/users.jpg');
            $type = 'jpg';
        } else {
            $image = Storage::disk('public')->get('users/users.png');
            $mime = Storage::disk('public')->mimeType('users/users.png');
            $type = 'png';
        }
        return (new Response($image, 200))
            ->header('Content-Type', $mime)->header('type', $type);
    }


    public function updateAvatar(AvatarRequest $request)
    {
        if ($request->hasFile('image')) {
            Storage::disk('public')->deleteDirectory('users');
            $extension = $request->image->getClientOriginalExtension();
            $filename = Storage::disk('public')->putFileAs('users', $request->image, "users.$extension");
            $data = [
                'status' => 200,
                'image_path' => $request->root() . '/api/image/users?' . time(),
            ];
        } else {
            $data = [
                'status' => 400,
            ];
        }

        return response()->json($data, $data['status']);
    }



      // update user data in the database
      public function phoneUpdate(Request $request
)
      {
        $this->validate($request, [
            'id' => 'required']);
        
       
        $user = Auth::user();


        DB::table('users')
            ->where('id',$request->id)
            ->update(
                array( 
                    "verified" => 1));

      return response()->noContent();

      }


    // update user data in the database
    public function update(UserRequest $request)
    {
        $user = Auth()->user();
        
        if ($request->get('password') == '') {
            $user->update($request->except('password'));
        } else {
            $user->update($request->all());
        }  
        $user->save();

        
        $data = [
            'status' => 200,
            self::MESSAGE,
            'body' => $user
        ];

        return response()->json($data, $data['status']);
    }


    public function isSubscribed(Request $request)
    {

     $user = Auth()->user();
        
      return response()->json($data = [
        'active' => $user->subscriptions()->active()->count()]);



    }

    public function store(Request $request){


        $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|min:6'
        ]);

        $user = User::create([
            'name' => request('name'),
            'email' => request('email'),
            'premuim' => false,
            'password' => bcrypt(request('password'))


        ]);

    }


    public function addUser(Request $request)
    {

        //validation for inputs
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'email' => 'required|email'
        ]);
        $return['message'] = 'Please enter valid inputs';
        if ($validator->fails()) {
            return $return;
        }

        //Add user
        $user = new User();
        $user->name = $request->name;
        $user->email = $request->email;
        $user->role = $request->role;
        $user->password = Hash::make(rand(10000, 99999));
        $user->save();
        return $user;
    }


    public function updateUser(UserUpdateRequest $request, User $user)


    {

        if ($user != null) {

            $user->fill($request->user);
            $user->save();

            $data = [
                'status' => 200,
                self::MESSAGE,
            ];
        } else {
            $data = [
                'status' => 400,
                'message' => 'Error',
            ];
        }

        return response()->json($data, $data['status']);
    }


    public function destroy(User $user)
{
    if ($user != null) {
        // Excluir registros favoritos relacionados a cada perfil do usuário
        foreach ($user->profiles as $profile) {
            $profile->delete(); // Exclui o perfil após remover os favoritos relacionados
        }

        // Após excluir todos os perfis e seus favoritos relacionados, exclua o usuário
        $user->delete();

        return response()->json([
            'status' => 200,
            'message' => 'User and related profiles successfully removed',
        ]);
    } else {
        return response()->json([
            'status' => 400,
            'message' => 'User could not be deleted',
        ]);
    }
}


    // update user password in the database
    public function passwordUpdate(PasswordUpdateRequest $request)
    {
        $user = Auth()->user();
        $user->password = bcrypt($request->password);
        $user->save();
        $data = [
            'status' => 200,
            self::MESSAGE,
        ];

        return response()->json($data, $data['status']);
    }




    public function passwordUpdateApp(PasswordAppRequest $request)
    {

        $settings = Setting::first();
        $settings->password = bcrypt($request->password);
        $settings->save();
        $data = [
            'status' => 200,
            self::MESSAGE,
        ];

        return response()->json($data, $data['status']);
    }


 // update user password in the database
 public function updateUserPassword (PasswordUpdateRequest $request)
 {
     $user = Auth()->user();
     $user->password = bcrypt($request->password);
     $user->save();
     $data = [
         'status' => 200,
         self::MESSAGE,
     ];

     return response()->json($data, $data['status']);
 }




 public function show($filename)
 {

    $image = Storage::disk('avatars')->get($filename);

    $mime = Storage::disk('avatars')->mimeType($filename);

    return (new Response($image, 200))->header('Content-Type', $mime);
 }

    public function getAccountDetails()
    {
        $user = Auth::user();

        return response()->json([
            'name' => $user->name,
            'email' => $user->email,
        ], 200);
    }

    public function getSubscriptionDetails()
    {
        $user = Auth::user();

        return response()->json([
            'subscription_expiry_date' => $user->expired_in,
        ], 200);
    }

    
}