<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Subscription extends Model
{
    /**
     * Os atributos que podem ser preenchidos em massa.
     *
     * @var array
     */
    protected $fillable = [
        'user_id',
        'plan_id',
        'status',
        'start_date',
        'end_date',
        'transaction_id',
        'payment_method',
        'is_active',
        'device_limit'
    ];

    /**
     * Os atributos que devem ser convertidos.
     *
     * @var array
     */
    protected $casts = [
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'is_active' => 'boolean',
        'device_limit' => 'integer'
    ];

    protected $dates = [
        'trial_ends_at', 'ends_at', 'created_at', 'updated_at'
    ];

    /**
     * Relacionamento com o usuário dono da assinatura
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Relacionamento com o plano associado
     */
    public function plan()
    {
        return $this->belongsTo(Plan::class);
    }

    /**
     * Escopos de consulta para assinaturas ativas
     */
    public function scopeActive($query)
    {
        return $query->where(function ($query) {
            $query->whereNull('ends_at')
                ->orWhere('ends_at', '>', Carbon::now());
        })->whereNotNull('stripe_status')
          ->where('stripe_status', 'active');
    }

    /**
     * Verifica se a assinatura está ativa
     *
     * @return bool
     */
    public function isActive()
    {
        if ($this->stripe_status !== 'active') {
            return false;
        }

        return $this->ends_at === null || $this->ends_at->isFuture();
    }

    /**
     * Retorna o limite de dispositivos para esta assinatura
     * Se não estiver definido na assinatura, usa o limite do plano
     * Se o plano não tiver limite, usa o valor padrão
     *
     * @return int
     */
    public function getDeviceLimit()
    {
        if (!empty($this->device_limit)) {
            return $this->device_limit;
        }
        if ($this->plan) {
            return $this->plan->getDeviceLimit();
        }
        return config('app.default_device_limit', 1);
    }

    /**
     * Define o limite de dispositivos para esta assinatura específica
     */
    public function setDeviceLimit($limit)
    {
        $this->device_limit = $limit;
        $this->save();
        return $this;
    }
}
