<?php

namespace App\Services;

use App\Device;
use App\User;
use Illuminate\Http\Request;
use Jenssegers\Agent\Agent;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Exception;

class DeviceService
{
    /**
     * Determina o limite de dispositivos para um usuário
     *
     * @param User $user
     * @return int
     */
    public function getUserDeviceLimit(User $user)
    {
        try {
            // LOG para debug
            Log::debug("Determinando limite de dispositivos para usuário {$user->id}");

            // 1. Verificar cache
            $cacheKey = "user_{$user->id}_device_limit";
            $cachedLimit = cache($cacheKey);
            if ($cachedLimit) {
                Log::debug("Limite obtido do cache: {$cachedLimit}");
                return $cachedLimit;
            }

            // 2. Verificar atributo direto no usuário (se o admin definiu manualmente)
            if (isset($user->device_limit) && $user->device_limit > 0) {
                Log::debug("Limite obtido diretamente do usuário: {$user->device_limit}");
                cache([$cacheKey => $user->device_limit], now()->addHours(6));
                return $user->device_limit;
            }

            // 3. Verificar assinatura ativa atual do usuário
            $subscription = $user->subscription;
            if ($subscription && $subscription->isActive()) {
                // Obter o limite de dispositivos através do método do modelo
                $limit = $subscription->getDeviceLimit();
                Log::debug("Limite obtido da assinatura ativa: {$limit}");
                cache([$cacheKey => $limit], now()->addHours(6));
                return $limit;
            }

            // 4. Verificar no pack_id do usuário (método legado)
            if ($user->pack_id) {
                $plan = \App\Plan::find($user->pack_id);
                if ($plan && $plan->device_limit > 0) {
                    Log::debug("Limite obtido do plano via pack_id: {$plan->device_limit}");
                    cache([$cacheKey => $plan->device_limit], now()->addHours(6));
                    return $plan->device_limit;
                }
            }

            // 5. Verificar configuração global
            $defaultLimit = config('app.default_device_limit', 1);
            Log::debug("Usando limite padrão: {$defaultLimit}");
            cache([$cacheKey => $defaultLimit], now()->addHours(6));
            return $defaultLimit;
        } catch (Exception $e) {
            Log::error("Erro ao determinar limite de dispositivos: " . $e->getMessage(), [
                'user_id' => $user->id,
                'trace' => $e->getTraceAsString()
            ]);

            // Em caso de erro, retorna o valor padrão mais conservador
            return 1;
        }
    }

    /**
     * Registra um dispositivo para o usuário
     *
     * @param User $user
     * @param Request $request
     * @return Device|null
     */
    public function registerDevice(User $user, Request $request)
    {
        try {
            if (!$user) {
                Log::warning('Tentativa de registrar dispositivo sem usuário autenticado');
                return null;
            }

            Log::debug("Iniciando registro de dispositivo para usuário {$user->id}");

            // Garantir que a tabela de dispositivos tem todas as colunas necessárias
            $this->ensureRequiredColumns();

            // Detectar informações do dispositivo
            $agent = new Agent();
            $agent->setUserAgent($request->userAgent());

            // Determinar tipo de dispositivo
            if ($agent->isPhone()) {
                $deviceType = 'mobile';
            } elseif ($agent->isTablet()) {
                $deviceType = 'tablet';
            } else {
                $deviceType = 'desktop';
            }

            $deviceName = $agent->platform() . ' - ' . $agent->browser();
            $userAgent = $request->userAgent();
            $ipAddress = $request->ip();
            $serialNumber = md5($userAgent . $ipAddress);

            // Marcar todos os outros dispositivos como não-atuais
            DB::transaction(function () use ($user, $deviceType, $deviceName, $userAgent, $ipAddress, $serialNumber) {
                // Desmarcar todos os dispositivos do usuário como não atuais
                $user->devices()->update(['is_current' => false]);

                // Verificar se este dispositivo já existe para este usuário
                $existingDevice = $user->devices()
                    ->where(function($query) use ($userAgent, $ipAddress, $serialNumber) {
                        $query->where(function($q) use ($userAgent, $ipAddress) {
                            $q->where('user_agent', $userAgent)
                              ->where('ip_address', $ipAddress);
                        })
                        ->orWhere('serial_number', $serialNumber);
                    })
                    ->first();

                // Garantir que last_active seja sempre uma data válida
                $lastActive = now();

                // Se existir, atualiza; se não, cria um novo
                if ($existingDevice) {
                    Log::debug("Atualizando dispositivo existente ID: {$existingDevice->id}");

                    $existingDevice->update([
                        'last_active' => $lastActive,
                        'is_current' => true,
                        'name' => $deviceName,
                        'device_type' => $deviceType,
                        'user_agent' => $userAgent,
                        'ip_address' => $ipAddress
                    ]);

                    return $existingDevice;
                } else {
                    Log::debug("Criando novo dispositivo para usuário {$user->id}");

                    // Obter o limite de dispositivos do usuário
                    $limit = $this->getUserDeviceLimit($user);

                    // Contar dispositivos ativos
                    $activeDevicesCount = $user->devices()->count();

                    // Se estiver no limite, remove o dispositivo mais antigo
                    if ($activeDevicesCount >= $limit) {
                        Log::debug("Usuário atingiu o limite de dispositivos ({$limit}). Removendo dispositivo mais antigo.");

                        $oldestDevice = $user->devices()
                            ->where('is_current', false)
                            ->orderBy('last_active', 'asc')
                            ->first();

                        if ($oldestDevice) {
                            $oldestDevice->delete();
                            Log::debug("Dispositivo mais antigo ID: {$oldestDevice->id} removido.");
                        }
                    }

                    return $user->devices()->create([
                        'name' => $deviceName,
                        'device_type' => $deviceType,
                        'ip_address' => $ipAddress,
                        'user_agent' => $userAgent,
                        'last_active' => $lastActive,
                        'is_current' => true,
                        'serial_number' => $serialNumber
                    ]);
                }
            });

            // Busca o dispositivo atualizado após a transação
            return $user->devices()
                ->where('is_current', true)
                ->first();
        } catch (Exception $e) {
            Log::error("Erro ao registrar dispositivo: " . $e->getMessage(), [
                'user_id' => $user->id,
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * Verifica se um usuário excedeu o limite de dispositivos
     *
     * @param User $user
     * @param Request $request
     * @return bool true se pode acessar, false se excedeu o limite
     */
    public function checkDeviceLimit(User $user, Request $request)
    {
        try {
            // Obter o limite de dispositivos
            $limit = $this->getUserDeviceLimit($user);

            // Obter identificadores do dispositivo atual
            $userAgent = $request->userAgent();
            $ipAddress = $request->ip();
            $serialNumber = md5($userAgent . $ipAddress);

            // Verificar se este dispositivo já está registrado
            $isExistingDevice = $user->devices()
                ->where(function($query) use ($userAgent, $ipAddress, $serialNumber) {
                    $query->where(function($q) use ($userAgent, $ipAddress) {
                        $q->where('user_agent', $userAgent)
                          ->where('ip_address', $ipAddress);
                    })
                    ->orWhere('serial_number', $serialNumber);
                })
                ->exists();

            // Se já existir, permitir acesso
            if ($isExistingDevice) {
                Log::debug("Dispositivo já registrado. Acesso liberado para usuário {$user->id}");
                return true;
            }

            // Contar dispositivos ativos
            $activeDevices = $user->devices()->count();

            // Verificar se há espaço para um novo dispositivo
            $canAccess = $activeDevices < $limit;

            Log::debug("Verificação de limite para usuário {$user->id}: limite {$limit}, ativos {$activeDevices}, pode acessar? " . ($canAccess ? 'Sim' : 'Não'));

            return $canAccess;
        } catch (Exception $e) {
            Log::error("Erro ao verificar limite de dispositivos: " . $e->getMessage(), [
                'user_id' => $user->id,
                'trace' => $e->getTraceAsString()
            ]);

            // Em caso de erro, permite o acesso
            return true;
        }
    }

    /**
     * Garante que todas as colunas necessárias existam na tabela de dispositivos
     */
    private function ensureRequiredColumns()
    {
        try {
            // Verifica se as colunas necessárias existem na tabela
            if (!Schema::hasColumn('devices', 'device_type') ||
                !Schema::hasColumn('devices', 'ip_address') ||
                !Schema::hasColumn('devices', 'user_agent') ||
                !Schema::hasColumn('devices', 'last_active') ||
                !Schema::hasColumn('devices', 'is_current') ||
                !Schema::hasColumn('devices', 'serial_number')) {

                // Adiciona as colunas faltantes
                Schema::table('devices', function ($table) {
                    if (!Schema::hasColumn('devices', 'device_type')) {
                        $table->string('device_type')->nullable();
                    }
                    if (!Schema::hasColumn('devices', 'ip_address')) {
                        $table->string('ip_address')->nullable();
                    }
                    if (!Schema::hasColumn('devices', 'user_agent')) {
                        $table->text('user_agent')->nullable();
                    }
                    if (!Schema::hasColumn('devices', 'last_active')) {
                        $table->timestamp('last_active')->nullable();
                    }
                    if (!Schema::hasColumn('devices', 'is_current')) {
                        $table->boolean('is_current')->default(false);
                    }
                    if (!Schema::hasColumn('devices', 'serial_number')) {
                        $table->string('serial_number')->nullable();
                    }
                });

                Log::info("Colunas necessárias adicionadas à tabela de dispositivos");
            }
        } catch (Exception $e) {
            Log::error("Erro ao garantir colunas na tabela de dispositivos: " . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
