<?php

namespace App;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class Plan extends Model
{
    protected $fillable = [
        'name',
        'slug',
        'stripe_plan',
        'price',
        'device_limit',
        'features',
        'is_active',
        'priority',
        'color',
        'description'
    ];

    /**
     * Os atributos que devem ser convertidos.
     *
     * @var array
     */
    protected $casts = [
        'price' => 'float',
        'device_limit' => 'integer',
        'features' => 'array',
        'is_active' => 'boolean',
        'priority' => 'integer'
    ];

    /**
     * Retorna o limite de dispositivos do plano ou um valor padrão
     *
     * @return int
     */
    public function getDeviceLimit()
    {
        // Registrar a chamada para depuração
        Log::debug("Obtendo limite de dispositivos do plano {$this->id} ({$this->name}): {$this->device_limit}");

        return $this->device_limit ?? config('app.default_device_limit', 1);
    }

    /**
     * Define o limite de dispositivos para o plano
     *
     * @param int $limit
     * @return $this
     */
    public function setDeviceLimit($limit)
    {
        $this->device_limit = $limit;
        return $this;
    }

    /**
     * Relacionamento com as assinaturas deste plano
     */
    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Escopo para buscar apenas planos ativos
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Retorna os planos ativos ordenados por prioridade
     */
    public static function getActivePlans()
    {
        return self::active()->orderBy('priority', 'desc')->get();
    }

    /**
     * Verifica se um plano possui uma característica específica
     */
    public function hasFeature($featureName)
    {
        if (empty($this->features)) {
            return false;
        }
        return in_array($featureName, $this->features);
    }
}
