<?php

namespace App\Notifications;

use App\Models\SupportTicket;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TicketUpdatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $ticket;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct(SupportTicket $ticket)
    {
        $this->ticket = $ticket;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        // Começar com database que é mais confiável
        $channels = ['database'];

        // Adicionar email apenas se estiver configurado
        if (config('mail.default') && config('mail.mailers.smtp.host')) {
            $channels[] = 'mail';
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $status = '';
        if ($this->ticket->status === 'aberto') {
            $status = 'Aberto';
        } elseif ($this->ticket->status === 'em_andamento') {
            $status = 'Em Andamento';
        } elseif ($this->ticket->status === 'resolvido') {
            $status = 'Resolvido';
        }

        $mail = (new MailMessage)
            ->subject('Atualização no Seu Ticket de Suporte #' . $this->ticket->id)
            ->greeting('Olá ' . $notifiable->name . '!')
            ->line('Seu ticket de suporte foi atualizado.')
            ->line('Assunto: ' . $this->ticket->subject)
            ->line('Status: ' . $status)
            ->action('Ver Ticket', route('support.show', $this->ticket->id));

        if ($this->ticket->admin_response) {
            $mail->line('Resposta da Equipe:')
                 ->line($this->ticket->admin_response);
        }

        return $mail;
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        $statusText = '';
        if ($this->ticket->status === 'aberto') {
            $statusText = 'aberto';
        } elseif ($this->ticket->status === 'em_andamento') {
            $statusText = 'em andamento';
        } elseif ($this->ticket->status === 'resolvido') {
            $statusText = 'resolvido';
        }

        return [
            'ticket_id' => $this->ticket->id,
            'subject' => $this->ticket->subject,
            'status' => $this->ticket->status,
            'type' => 'ticket_updated',
            'message' => "Seu ticket foi atualizado para: {$statusText}",
            'url' => route('support.show', $this->ticket->id)
        ];
    }
}
