<?php

namespace App\Notifications;

use App\Models\SupportTicket;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TicketCreatedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $ticket;

    /**
     * Create a new notification instance.
     *
     * @return void
     */
    public function __construct(SupportTicket $ticket)
    {
        $this->ticket = $ticket;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        // É melhor verificar se os canais estão configurados antes de tentar usá-los
        $channels = ['database']; // Começar com database que é mais confiável

        if (config('mail.default') && config('mail.mailers.smtp.host')) {
            $channels[] = 'mail'; // Adicionar mail apenas se configurado
        }

        return $channels;
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        return (new MailMessage)
            ->subject('Novo Ticket de Suporte #' . $this->ticket->id)
            ->line('Um novo ticket de suporte foi criado por ' . $this->ticket->user->name . '.')
            ->line('Assunto: ' . $this->ticket->subject)
            ->line('Categoria: ' . (SupportTicket::getCategories()[$this->ticket->category] ?? $this->ticket->category))
            ->action('Ver Ticket', route('admin.support.show', $this->ticket->id))
            ->line('Mensagem: ' . \Illuminate\Support\Str::limit($this->ticket->message, 300));
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        return [
            'ticket_id' => $this->ticket->id,
            'subject' => $this->ticket->subject,
            'user_id' => $this->ticket->user_id,
            'user_name' => $this->ticket->user->name,
            'type' => 'ticket_created',
            'message' => "Novo ticket de suporte: {$this->ticket->subject}",
            'url' => route('admin.support.show', $this->ticket->id)
        ];
    }
}
