<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class LimitDownloadMiddleware
{
    /**
     * Manipula uma requisição recebida.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $ip = $request->ip();

        // Verifica se é uma requisição de validação ou download direto
        $isValidation = $request->has('password');
        $isDownload = $request->has('download_token');

        // Se for uma requisição de download, verifica o token
        if ($isDownload) {
            $downloadToken = $request->input('download_token');
            $tokenKey = 'sql_download_token_' . $downloadToken;

            if (!Cache::has($tokenKey)) {
                return redirect()->route('download.view')->withErrors([
                    'token' => 'Token de download inválido ou expirado. Por favor, tente novamente.'
                ]);
            }

            // Token válido, prossegue com o download
            Cache::forget($tokenKey); // Usa o token apenas uma vez
            return $next($request);
        }

        // A partir daqui é apenas validação de senha
        if (!$isValidation) {
            return redirect()->route('download.view');
        }

        // Verifica tentativas de senha - limitação contra força bruta
        $attemptsCacheKey = 'password_attempts_' . $ip;
        $attempts = Cache::get($attemptsCacheKey, 0);

        // Se excedeu tentativas, bloqueia por 30 minutos
        if ($attempts >= 5) {
            Log::warning("Muitas tentativas de senha para download SQL do IP: {$ip}");
            return redirect()->back()->withErrors([
                'blocked' => 'Muitas tentativas incorretas. Tente novamente em 30 minutos.'
            ]);
        }

        // Senha definida para acesso
        $password = env('DOWNLOAD_PASSWORD', 'PAL0194JMMDK8ALLOS9E');
        $maxDownloadsPerDay = env('MAX_DOWNLOADS_PER_DAY', 4);

        // Verifica a senha
        if ($request->input('password') !== $password) {
            // Incrementa contagem de tentativas
            Cache::put($attemptsCacheKey, $attempts + 1, now()->addMinutes(30));

            Log::info("Tentativa de download SQL com senha incorreta do IP: {$ip}");
            return redirect()->back()->withErrors([
                'password' => 'Senha incorreta. ' . (5 - ($attempts + 1)) . ' tentativas restantes.'
            ]);
        }

        // Senha correta, resetar contador de tentativas
        Cache::forget($attemptsCacheKey);

        // Chave para rastrear downloads por IP
        $downloadCacheKey = 'downloads_' . date('Y-m-d') . '_' . $ip;

        // Obtém contagem atual de downloads
        $downloads = Cache::get($downloadCacheKey, 0);
        $downloads++;

        // Limita ao número máximo de downloads por dia
        if ($downloads > $maxDownloadsPerDay) {
            Log::info("Limite de download excedido para o IP: {$ip}. Tentativa: {$downloads}");
            return redirect()->back()->withErrors([
                'download_limit' => "Você atingiu o limite de {$maxDownloadsPerDay} downloads por dia. Tente novamente amanhã."
            ]);
        }

        // Gera um token temporário para o download
        $downloadToken = Str::random(32);
        $tokenKey = 'sql_download_token_' . $downloadToken;

        // Armazena o token por 5 minutos
        Cache::put($tokenKey, $ip, now()->addMinutes(5));

        // Incrementa o contador e define expiração
        $endOfDay = now()->endOfDay();
        $secondsUntilEndOfDay = $endOfDay->diffInSeconds(now());
        Cache::put($downloadCacheKey, $downloads, $secondsUntilEndOfDay);

        // Registra validação bem-sucedida
        Log::info("Validação para download SQL concedida para o IP: {$ip}. Downloads hoje: {$downloads}");

        // Redireciona com token e mensagem de sucesso
        return redirect()->route('download.view')
            ->with('download_token', $downloadToken)
            ->with('success', 'Senha validada com sucesso! Clique no botão abaixo para baixar o arquivo SQL.');
    }
}
