<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\DeviceService;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;

class CheckDeviceLimit
{
    protected $deviceService;

    /**
     * Construtor com injeção do serviço de dispositivos
     */
    public function __construct(DeviceService $deviceService)
    {
        $this->deviceService = $deviceService;
    }

    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // Se o usuário não está logado, permitir (a autenticação ocorrerá em outro middleware)
        if (!Auth::check()) {
            return $next($request);
        }

        $user = Auth::user();

        try {
            // Registrar ou atualizar o dispositivo atual
            $currentDevice = $this->deviceService->registerDevice($user, $request);

            if (!$currentDevice) {
                Log::warning('Não foi possível registrar o dispositivo atual', [
                    'user_id' => $user->id,
                    'ip' => $request->ip()
                ]);
            }

            // Verificar se está dentro do limite de dispositivos
            $canAccess = $this->deviceService->checkDeviceLimit($user, $request);

            // Se não pode acessar por limite de dispositivos
            if (!$canAccess) {
                if ($request->expectsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Limite de dispositivos atingido',
                        'deviceLimitExceeded' => true
                    ], 403);
                }

                return redirect()->route('devices.index')->with(
                    'error',
                    'Você atingiu o limite de dispositivos para sua conta. Remova um dispositivo para continuar.'
                );
            }

            return $next($request);
        } catch (\Exception $e) {
            Log::error('Erro ao verificar o limite de dispositivos', [
                'user_id' => $user->id,
                'ip' => $request->ip(),
                'error' => $e->getMessage()
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Erro interno ao verificar o limite de dispositivos'
                ], 500);
            }

            return redirect()->route('devices.index')->with(
                'error',
                'Erro interno ao verificar o limite de dispositivos. Tente novamente mais tarde.'
            );
        }
    }
}
