<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\StreamedResponse;

class VideoLinkController extends Controller
{
    /**
     * Export the list of animes without video links to a .txt file.
     *
     * @return StreamedResponse
     */
    public function exportAnimesWithoutVideoLink()
    {
        return $this->exportWithoutVideoLink('Anime', 'seasons', 'anime_id');
    }

    /**
     * Export the list of series without video links to a .txt file.
     *
     * @return StreamedResponse
     */
    public function exportSeriesWithoutVideoLink()
    {
        return $this->exportWithoutVideoLink('Serie', 'seasons', 'serie_id');
    }

    /**
     * Export the list of movies without video links to a .txt file.
     *
     * @return StreamedResponse
     */
    public function exportMoviesWithoutVideoLink()
    {
        return $this->exportMoviesWithoutVideos();
    }

    /**
     * Helper method to export animes or series without video links.
     *
     * @param string $modelName
     * @param string $relationTable
     * @param string $foreignKey
     * @return StreamedResponse
     */
    protected function exportWithoutVideoLink($modelName, $relationTable, $foreignKey)
    {
        $fileName = strtolower($modelName) . '_without_video_links.txt';
        $modelClass = "App\\$modelName";

        return new StreamedResponse(function () use ($modelClass, $relationTable, $foreignKey) {
            // Write header
            echo "=== $modelName Sem Links de Vídeo ===\n";

            // Get IDs of items without video links
            $items = $modelClass::select('id', 'name')
                ->whereDoesntHave('seasons.episodes.videos', function ($query) {
                    $query->whereNotNull('link');
                })
                ->cursor(); // Use cursor for memory efficiency

            foreach ($items as $item) {
                echo "{$item->name} (ID: {$item->id})\n";
            }
        }, 200, [
            'Content-Type' => 'text/plain',
            'Content-Disposition' => "attachment; filename=\"$fileName\"",
        ]);
    }

    /**
     * Helper method to export movies without video links.
     *
     * @return StreamedResponse
     */
    protected function exportMoviesWithoutVideos()
    {
        $fileName = 'movies_without_video_links.txt';

        return new StreamedResponse(function () {
            // Write header
            echo "=== Filmes Sem Links de Vídeo ===\n";

            // Get IDs of movies without video links
            $movies = DB::table('movies')
                ->select('movies.id', 'movies.title')
                ->leftJoin('movie_videos', 'movies.id', '=', 'movie_videos.movie_id')
                ->whereNull('movie_videos.link')
                ->groupBy('movies.id')
                ->havingRaw('COUNT(movie_videos.id) = 0')
                ->cursor(); // Use cursor for memory efficiency

            foreach ($movies as $movie) {
                echo "{$movie->title} (ID: {$movie->id})\n";
            }
        }, 200, [
            'Content-Type' => 'text/plain',
            'Content-Disposition' => "attachment; filename=\"$fileName\"",
        ]);
    }
}
