<?php

namespace App\Http\Controllers;

use App\Models\SupportTicket;
use App\User;
use App\Notifications\TicketCreatedNotification;
use App\Notifications\TicketUpdatedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

class SupportTicketController extends Controller
{
    public function index()
    {
        $tickets = Auth::user()->supportTickets()->orderBy('created_at', 'desc')->paginate(10);
        return view('support.index', compact('tickets'));
    }

    public function create()
    {
        $categories = SupportTicket::getCategories();
        return view('support.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'subject' => 'required|max:255',
            'message' => 'required',
            'category' => 'required|in:' . implode(',', array_keys(SupportTicket::getCategories())),
        ]);

        $ticket = new SupportTicket([
            'subject' => $request->subject,
            'message' => $request->message,
            'category' => $request->category,
            'status' => 'aberto',
            'user_id' => Auth::id(),
        ]);

        $ticket->save();

        // Modificando aqui para usar o campo 'role' em vez de 'is_admin'
        // Notificar administradores sobre o novo ticket
        $admins = User::where('role', 'admin')->get();

        // Se quiser ser mais flexível, pode verificar se há registros
        if ($admins->isEmpty()) {
            // Tenta uma alternativa se 'role' não retornar resultados
            $admins = User::where('premuim', 1)->get(); // Exemplo de alternativa
        }

        // Só enviar notificação se houver administradores
        if ($admins->isNotEmpty()) {
            try {
                Notification::send($admins, new TicketCreatedNotification($ticket));
            } catch (\Exception $e) {
                // Log do erro mas continue o fluxo
                \Log::error('Erro ao enviar notificação: ' . $e->getMessage());
            }
        }

        return redirect()->route('support.show', $ticket->id)
            ->with('success', 'Seu ticket de suporte foi enviado com sucesso!');
    }

    public function show($id)
    {
        $ticket = SupportTicket::findOrFail($id);

        // Verificar se o usuário tem acesso a este ticket
        if (Auth::user()->id !== $ticket->user_id && !Auth::user()->is_admin) {
            return redirect()->route('support.index')
                ->with('error', 'Você não tem permissão para acessar este ticket.');
        }

        return view('support.show', compact('ticket'));
    }

    public function adminIndex()
    {
        // Verificar se o usuário é administrador
        if (!Auth::user() || Auth::user()->role !== 'admin') {
            return redirect()->route('support.index')
                ->with('error', 'Acesso não autorizado.');
        }

        $openTickets = SupportTicket::where('status', 'aberto')
            ->orderBy('created_at', 'desc')
            ->paginate(10, ['*'], 'open_page');

        $inProgressTickets = SupportTicket::where('status', 'em_andamento')
            ->orderBy('updated_at', 'desc')
            ->paginate(10, ['*'], 'progress_page');

        $resolvedTickets = SupportTicket::where('status', 'resolvido')
            ->orderBy('updated_at', 'desc')
            ->paginate(10, ['*'], 'resolved_page');

        return view('admin.support.index', compact('openTickets', 'inProgressTickets', 'resolvedTickets'));
    }

    public function adminShow($id)
    {
        // Verificar se o usuário é administrador
        if (!Auth::user() || Auth::user()->role !== 'admin') {
            return redirect()->route('support.index')
                ->with('error', 'Acesso não autorizado.');
        }

        $ticket = SupportTicket::findOrFail($id);
        return view('admin.support.show', compact('ticket'));
    }

    public function adminUpdate(Request $request, $id)
    {
        // Verificar se o usuário é administrador
        if (!Auth::user() || Auth::user()->role !== 'admin') {
            return redirect()->route('support.index')
                ->with('error', 'Acesso não autorizado.');
        }

        $request->validate([
            'status' => 'required|in:aberto,em_andamento,resolvido',
            'admin_response' => 'required_if:status,resolvido',
        ]);

        $ticket = SupportTicket::findOrFail($id);
        $oldStatus = $ticket->status;

        $ticket->status = $request->status;

        if ($request->has('admin_response')) {
            $ticket->admin_response = $request->admin_response;
        }

        if ($request->status == 'resolvido' && !$ticket->admin_id) {
            $ticket->admin_id = Auth::id();
        }

        $ticket->save();

        // Se o status foi alterado ou uma resposta foi adicionada/modificada, notificar o usuário
        if ($oldStatus !== $ticket->status || $request->has('admin_response')) {
            $ticket->user->notify(new TicketUpdatedNotification($ticket));
        }

        return redirect()->route('admin.support.show', $ticket->id)
            ->with('success', 'Ticket atualizado com sucesso!');
    }

    public function getOpenTicketsCount()
    {
        // Verificar se o usuário é administrador
        if (!Auth::user() || Auth::user()->role !== 'admin') {
            return response()->json(['error' => 'Acesso não autorizado'], 403);
        }

        try {
            $count = SupportTicket::where('status', 'aberto')->count();
            return response()->json(['count' => $count]);
        } catch (\Exception $e) {
            \Log::error('Erro ao contar tickets abertos: ' . $e->getMessage());
            return response()->json(['count' => 0, 'error' => 'Erro ao processar solicitação']);
        }
    }
}
