<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\User;
use App\Plan;

class SuitPayController extends Controller
{
    protected $baseUrl;
    protected $clientId;
    protected $clientSecret;
    protected $isSandbox;
    protected $isPaggueEnabled;

    public function __construct()
    {
        // Verificar se o Paggue está ativado - garantir comparação correta
        $this->isPaggueEnabled = env('ENABLE_PAGGUE', 'false') === 'true';

        // Log para verificação
        Log::info('SuitPayController inicializado - Paggue está ' . ($this->isPaggueEnabled ? 'ATIVADO' : 'DESATIVADO'));

        // Configurações da API SuitPay
        $this->clientId = env('SUITPAY_CLIENT_ID', '');
        $this->clientSecret = env('SUITPAY_CLIENT_SECRET', '');
        $this->isSandbox = env('SUITPAY_SANDBOX', 'false') === 'true';
        $this->baseUrl = $this->isSandbox
            ? 'https://sandbox.ws.suitpay.app/api/v1'
            : 'https://ws.suitpay.app/api/v1';
    }

    /**
     * Gera o QR Code PIX para pagamento
     * Reescrito seguindo o modelo do PaymentController funcional
     */
    public function processPayment(Request $request)
    {
        // Verificar novamente se o Paggue está ativado (dupla verificação)
        if (env('ENABLE_PAGGUE', 'false') === 'true') {
            Log::warning('Tentativa de usar SuitPay quando Paggue está ativado - redirecionando');
            return app(PaggueController::class)->processPayment($request);
        }

        // Continua com o processamento normal do SuitPay
        $validated = $request->validate([
            'plano' => 'required',
            'name' => 'required|string',
            'email' => 'required|email',
            'numero' => 'required|string',
            // Campo document foi removido da validação
        ]);

        try {
            // Buscar o plano selecionado
            $plano = Plan::findOrFail($request->plano);
            $userId = Auth::id();
            $user = Auth::user();

            // Verificar se é plano de TV
            $isTvPlan = stripos($plano->name, 'tv') !== false ||
                      stripos($plano->name, 'canais') !== false ||
                      (isset($plano->is_tv_plan) && $plano->is_tv_plan);

            // Criar metadados para o webhook
            $metadata = [
                'email' => $request->email,
                'name' => $user->name,
                'user_id' => $userId,
                'plan_id' => $plano->id,
                'plan_name' => $plano->name,
                'plan_duration' => (int)$plano->pack_duration,
                'is_tv_plan' => $isTvPlan,
                'timestamp' => time()
            ];

            // Codificar os metadados no requestNumber
            $encodedMetadata = base64_encode(json_encode($metadata));
            $reference = 'goldflix_' . $encodedMetadata;

            // Se ficar muito grande, gerar uma versão curta
            if (strlen($reference) > 250) {
                $uniqueId = Str::random(8) . '_' . $userId;
                $reference = 'goldflix_short_' . $uniqueId;
                $this->saveMetadataToFile($uniqueId, $metadata);
            }

            // IMPORTANTE: criar a URL de callback absoluta
            // Corrigindo problema da URL relativa que pode não funcionar
            $callbackUrl = url('/suitpay/callback'); // URL absoluta

            // Certificar que essa URL está registrada no arquivo routes.php
            $this->logToFile('URL de callback gerada: ' . $callbackUrl);

            // Formatar número de telefone
            $phone = preg_replace('/[^0-9]/', '', $request->numero);

            // Definir um CPF válido padrão para satisfazer a API
            // Isso é necessário porque a documentação da SuitPay requer um documento válido
            // Usando um CPF válido genérico (este é o CPF fictício 123.456.789-09)
            $cpfPadrao = '12345678909';

            // Preparar dados para a API
            $paymentData = [
                'requestNumber' => $reference,
                'dueDate' => Carbon::now()->addDays(1)->format('Y-m-d'),
                'amount' => (float)$plano->price,
                'shippingAmount' => 0.0,
                'discountAmount' => 0.0,
                'usernameCheckout' => 'goldflix_checkout',
                'callbackUrl' => $callbackUrl,
                'client' => [
                    'name' => $request->name,
                    'document' => $cpfPadrao, // CPF válido para satisfazer a API
                    'phoneNumber' => $phone,
                    'email' => $request->email,
                ],
                'products' => [
                    [
                        'description' => $isTvPlan ? 'Plano TV: ' . $plano->name : 'Plano: ' . $plano->name,
                        'quantity' => 1,
                        'value' => (float)$plano->price
                    ]
                ]
            ];

            // Adicionar log para depurar a requisição à API
            $this->logToFile('Dados enviados para API: ' . json_encode($paymentData));

            // Gerar QR Code através do cURL direto
            $responseData = $this->makeApiRequest('/gateway/request-qrcode', $paymentData);

            // Log da resposta para depuração
            $this->logToFile('Resposta da API: ' . json_encode($responseData));

            if (!isset($responseData['paymentCodeBase64'])) {
                $this->logToFile('Erro na resposta da API: ' . json_encode($responseData));
                return redirect()->back()->withErrors(['api' => 'Erro ao gerar o QR Code PIX: ' . ($responseData['message'] ?? 'Erro desconhecido')]);
            }

            // Salvar dados do pagamento na sessão e em arquivo local
            $paymentInfo = [
                'reference' => $reference,
                'user_id' => $userId,
                'email' => $request->email,
                'plan_id' => $plano->id,
                'plan_duration' => (int)$plano->pack_duration,
                'plan_name' => $plano->name,
                'is_tv_plan' => $isTvPlan,
                'amount' => (float)$plano->price,
                'created_at' => now()->toDateTimeString(),
                'transaction_id' => $responseData['idTransaction'] ?? null,
            ];

            session()->put('payment_info_' . $reference, $paymentInfo);
            $this->savePaymentInfoToFile($reference, $paymentInfo);

            // Log de sucesso
            $this->logToFile('QR Code gerado com sucesso: ' . $reference);

            // Retorna view com QR Code
            return view('pagamento_pix', [
                'qrcode' => $responseData['paymentCodeBase64'],
                'qrcode_text' => $responseData['paymentCode'] ?? '',
                'plano' => $plano->name,
                'valor' => (float)$plano->price,
                'reference' => $reference,
                'transaction_id' => $responseData['idTransaction'] ?? null
            ]);

        } catch (\Exception $e) {
            // Log detalhado do erro
            $this->logToFile('Erro ao processar pagamento: ' . $e->getMessage() . "\n" . $e->getTraceAsString());

            return redirect()->back()->withErrors(['system' => 'Erro ao processar o pagamento. Por favor, tente novamente.']);
        }
    }

    /**
     * Webhook para receber notificações SuitPay
     * IMPORTANTE: Esta URL deve estar registrada no routes.php e ser acessível externamente
     */
    public function callback(Request $request)
    {
        try {
            // Log mais detalhado do webhook recebido
            $this->logWebhook($request);

            // Verificar campos necessários
            if (!$request->has('requestNumber') || !$request->has('statusTransaction')) {
                $this->logToFile('Webhook com dados incompletos');
                return response()->json(['status' => 'success']); // Sempre retorna sucesso
            }

            $requestNumber = $request->input('requestNumber');
            $statusTransaction = $request->input('statusTransaction');
            $idTransaction = $request->input('idTransaction');

            // Processar apenas se o pagamento foi confirmado
            if ($statusTransaction !== 'PAID_OUT') {
                $this->logToFile('Webhook com status diferente de PAID_OUT: ' . $statusTransaction);
                return response()->json(['status' => 'success']);
            }

            // Processar o pagamento
            $success = $this->processarPagamentoConfirmado($requestNumber, $idTransaction);
            $this->logToFile('Resultado do processamento: ' . ($success ? 'Sucesso' : 'Falha'));

            // Sempre responder com sucesso para evitar reenvios
            return response()->json(['status' => 'success']);

        } catch (\Exception $e) {
            // Log do erro
            $this->logToFile('Erro no webhook: ' . $e->getMessage() . "\n" . $e->getTraceAsString());

            // Sempre responder com sucesso
            return response()->json(['status' => 'success']);
        }
    }

    /**
     * Registra informações detalhadas do webhook em arquivo separado
     */
    private function logWebhook(Request $request)
    {
        try {
            // Diretório para webhooks
            $dir = storage_path('logs/suitpay_webhooks');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            // Nome do arquivo baseado na data
            $date = now()->format('Y-m-d');
            $time = now()->format('H:i:s');
            $filePath = $dir . '/webhook_' . $date . '.log';

            // Coletar todas as informações possíveis
            $data = [
                'timestamp' => $time,
                'ip' => $request->ip(),
                'method' => $request->method(),
                'headers' => $request->headers->all(),
                'data' => $request->all(),
                'server' => $_SERVER,
            ];

            // Salvar webhook completo em formato JSON
            $webhookFilePath = $dir . '/webhook_' . date('Y-m-d_H-i-s') . '_' . uniqid() . '.json';
            file_put_contents($webhookFilePath, json_encode($data, JSON_PRETTY_PRINT));

            // Adicionar entrada ao log principal de webhooks
            $logEntry = "[{$time}] Webhook recebido de {$request->ip()}: " . json_encode($request->all()) . "\n";
            file_put_contents($filePath, $logEntry, FILE_APPEND);

            // Também registre no log do Laravel
            Log::channel('daily')->info('SuitPay Webhook recebido', $data);

            // Mantenha um contador de webhooks em um arquivo separado
            $this->incrementWebhookCounter();

        } catch (\Exception $e) {
            Log::error('Erro ao registrar webhook: ' . $e->getMessage());
        }
    }

    /**
     * Incrementa o contador de webhooks recebidos
     */
    private function incrementWebhookCounter()
    {
        $counterFile = storage_path('logs/suitpay_webhooks/counter.json');
        $counter = [];

        if (file_exists($counterFile)) {
            $counter = json_decode(file_get_contents($counterFile), true) ?: [];
        }

        $today = now()->format('Y-m-d');
        $counter[$today] = ($counter[$today] ?? 0) + 1;

        // Manter apenas os últimos 30 dias
        if (count($counter) > 30) {
            $counter = array_slice($counter, -30, 30, true);
        }

        file_put_contents($counterFile, json_encode($counter, JSON_PRETTY_PRINT));

        // Opcional: atualizar um arquivo de status para verificação rápida
        $statusFile = storage_path('logs/suitpay_webhooks/status.txt');
        $status = "Último webhook: " . now()->format('Y-m-d H:i:s') . "\n";
        $status .= "Total hoje: " . $counter[$today] . "\n";
        file_put_contents($statusFile, $status);
    }

    /**
     * Processa o pagamento confirmado e ativa a conta do usuário
     */
    private function processarPagamentoConfirmado($requestNumber, $transactionId)
    {
        $this->logToFile('Processando pagamento: ' . $requestNumber);

        try {
            // 1. Obter os metadados do requestNumber
            $metadata = $this->getMetadataFromRequestNumber($requestNumber);
            if (!$metadata) {
                $this->logToFile('Metadados não encontrados para: ' . $requestNumber);
                return false;
            }

            // 2. Buscar o usuário
            $user = null;
            if (!empty($metadata['user_id'])) {
                $user = User::find($metadata['user_id']);
            }

            if (!$user && !empty($metadata['email'])) {
                $user = User::where('email', $metadata['email'])->first();
            }

            if (!$user) {
                $this->logToFile('Usuário não encontrado para: ' . json_encode($metadata));
                return false;
            }

            // 3. Ativar a assinatura adequada
            $isTvPlan = $metadata['is_tv_plan'] ?? false;
            $planDuration = $metadata['plan_duration'] ?? 30;
            $planName = $metadata['plan_name'] ?? 'Plano Premium';
            $planId = $metadata['plan_id'] ?? null;

            if ($isTvPlan) {
                $user->tv_access = 1;
                $user->expired_in_tv = Carbon::now()->addDays($planDuration);
                $this->logToFile('Ativando plano TV para usuário ' . $user->id . ' por ' . $planDuration . ' dias');
            } else {
                $user->premuim = 1;
                // Se já tem uma assinatura ativa, adiciona dias
                if ($user->expired_in && Carbon::parse($user->expired_in)->isFuture()) {
                    $user->expired_in = Carbon::parse($user->expired_in)->addDays($planDuration);
                } else {
                    $user->expired_in = Carbon::now()->addDays($planDuration);
                }
                $this->logToFile('Ativando plano Premium para usuário ' . $user->id . ' por ' . $planDuration . ' dias');
            }

            // 4. Atualizar informações adicionais
            $user->transaction_id = $transactionId;
            $user->manual_premuim = 'PIX-SuitPay';
            $user->pack_name = $planName;
            $user->pack_id = $planId;
            $user->start_at = Carbon::now();

            // 5. Salvar as alterações
            $result = $user->save();

            $this->logToFile('Usuário atualizado: ' . json_encode([
                'user_id' => $user->id,
                'email' => $user->email,
                'result' => $result,
                'premuim' => $user->premuim,
                'tv_access' => $user->tv_access,
                'expired_in' => $user->expired_in,
                'expired_in_tv' => $user->expired_in_tv,
            ]));

            // 6. Atualizar status na sessão
            $paymentInfo = session()->get('payment_info_' . $requestNumber);
            if ($paymentInfo) {
                $paymentInfo['confirmed'] = true;
                $paymentInfo['confirmed_at'] = now()->toDateTimeString();
                session()->put('payment_info_' . $requestNumber, $paymentInfo);
            }

            // Salvar a notificação no banco de dados para o painel admin
            // Isso é opcional, mas útil para ver webhooks recebidos no painel admin

            return $result;

        } catch (\Exception $e) {
            $this->logToFile('Erro ao ativar assinatura: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Extrai os metadados do requestNumber
     */
    private function getMetadataFromRequestNumber($requestNumber)
    {
        // 1. Verificar se é um requestNumber válido
        if (strpos($requestNumber, 'goldflix_') !== 0) {
            return null;
        }

        // 2. Verificar se é um requestNumber curto
        if (strpos($requestNumber, 'goldflix_short_') === 0) {
            $uniqueId = str_replace('goldflix_short_', '', $requestNumber);
            return $this->getMetadataFromFile($uniqueId);
        }

        // 3. É um requestNumber codificado
        $parts = explode('_', $requestNumber, 2);
        if (count($parts) < 2) {
            return null;
        }

        // 4. Decodificar os metadados
        try {
            return json_decode(base64_decode($parts[1]), true);
        } catch (\Exception $e) {
            $this->logToFile('Erro ao decodificar metadados: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Salva os metadados em arquivo para persistência
     */
    private function saveMetadataToFile($uniqueId, $metadata)
    {
        try {
            $dir = storage_path('pix_metadata');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            $filePath = $dir . '/' . $uniqueId . '.json';
            file_put_contents($filePath, json_encode($metadata, JSON_PRETTY_PRINT));
        } catch (\Exception $e) {
            Log::error('Erro ao salvar metadados: ' . $e->getMessage());
        }
    }

    /**
     * Recupera os metadados do arquivo
     */
    private function getMetadataFromFile($uniqueId)
    {
        try {
            $filePath = storage_path('pix_metadata/' . $uniqueId . '.json');
            if (!file_exists($filePath)) {
                return null;
            }

            return json_decode(file_get_contents($filePath), true);
        } catch (\Exception $e) {
            Log::error('Erro ao ler metadados: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Salva informações do pagamento em arquivo
     */
    private function savePaymentInfoToFile($reference, $paymentInfo)
    {
        try {
            $dir = storage_path('pix_payments');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            $filePath = $dir . '/' . $reference . '.json';
            file_put_contents($filePath, json_encode($paymentInfo, JSON_PRETTY_PRINT));
        } catch (\Exception $e) {
            Log::error('Erro ao salvar info de pagamento: ' . $e->getMessage());
        }
    }

    /**
     * Escreve logs em arquivo específico para o SuitPay
     */
    private function logToFile($message)
    {
        try {
            $dir = storage_path('logs/suitpay');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            $date = now()->format('Y-m-d');
            $time = now()->format('H:i:s');
            $filePath = $dir . '/suitpay_' . $date . '.log';

            $content = "[{$time}] {$message}\n";
            file_put_contents($filePath, $content, FILE_APPEND);

            // Também registra no log do Laravel
            Log::info('SuitPay: ' . $message);

        } catch (\Exception $e) {
            Log::error('Erro ao escrever log SuitPay: ' . $e->getMessage());
        }
    }

    /**
     * Faz requisição à API SuitPay usando cURL
     */
    private function makeApiRequest($endpoint, $data)
    {
        $url = $this->baseUrl . $endpoint;

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Accept: application/json',
                'ci: ' . $this->clientId,
                'cs: ' . $this->clientSecret
            ]
        ]);

        // Configuração para depurar a requisição
        curl_setopt($ch, CURLOPT_VERBOSE, true);
        $verbose = fopen('php://temp', 'w+');
        curl_setopt($ch, CURLOPT_STDERR, $verbose);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $info = curl_getinfo($ch);

        // Captura informações de debug
        rewind($verbose);
        $verboseLog = stream_get_contents($verbose);
        $this->logToFile('Debug cURL: ' . $verboseLog);

        curl_close($ch);

        if ($error) {
            $this->logToFile('Erro cURL: ' . $error);
            throw new \Exception('Erro na requisição: ' . $error);
        }

        // Log da requisição
        $this->logToFile('API Request: ' . $url . ' Status: ' . $info['http_code']);

        $responseData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->logToFile('Erro ao decodificar resposta: ' . $response);
            throw new \Exception('Resposta inválida da API');
        }

        return $responseData;
    }

    /**
     * Verifica o status do pagamento (agora também consultando diretamente a API SuitPay)
     */
    public function checkStatus(Request $request)
    {
        // Se o Paggue estiver ativado, redirecionar para o controlador do Paggue
        if ($this->isPaggueEnabled &&
            isset($request->gateway) &&
            $request->gateway === 'paggue') {
            return app(PaggueController::class)->checkStatus($request);
        }

        // Continua com a verificação de status do SuitPay
        $reference = $request->transaction_id;
        $this->logToFile('Verificação de status solicitada para: ' . $reference);

        try {
            $userId = Auth::id();
            $user = Auth::user();

            // 1. VERIFICAÇÃO LOCAL: Verificar se o usuário já está com plano ativo
            if ($user) {
                $isTvPlan = $request->input('isTvPlan', false);
                if (($isTvPlan && $user->tv_access == 1 && $user->expired_in_tv && Carbon::parse($user->expired_in_tv)->isFuture()) ||
                    (!$isTvPlan && $user->premuim == 1 && $user->expired_in && Carbon::parse($user->expired_in)->isFuture())) {
                    return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
                }
            }

            // 2. VERIFICAÇÃO NA SESSÃO: Verificar informações salvas na sessão
            $paymentInfo = session()->get('payment_info_' . $reference);
            if ($paymentInfo && isset($paymentInfo['confirmed']) && $paymentInfo['confirmed'] === true) {
                return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
            }

            // 3. VERIFICAÇÃO FORÇADA: Verificar status direto na API (apenas para o "verificar manualmente")
            if ($request->input('force_check', false)) {
                $this->logToFile('Iniciando verificação forçada via API para: ' . $reference);

                // Obter o transaction_id da sessão
                $transactionId = null;
                if ($paymentInfo && isset($paymentInfo['transaction_id'])) {
                    $transactionId = $paymentInfo['transaction_id'];
                    $this->logToFile('ID da transação encontrado: ' . $transactionId);
                }

                // Se encontrarmos o ID da transação, consultar status
                if ($transactionId) {
                    try {
                        // Checar status direto na API SuitPay
                        $statusResponse = $this->checkTransactionStatus($transactionId);
                        $this->logToFile('Resposta da verificação na API: ' . json_encode($statusResponse));

                        // Se o status for de pagamento confirmado, ativar manualmente
                        if (isset($statusResponse['status']) && in_array(strtoupper($statusResponse['status']), ['PAID', 'PAID_OUT', 'COMPLETED'])) {
                            $this->logToFile('Pagamento confirmado na API. Ativando assinatura manualmente.');

                            $metadata = null;
                            if (isset($paymentInfo)) {
                                $metadata = [
                                    'user_id' => $paymentInfo['user_id'],
                                    'email' => $paymentInfo['email'],
                                    'plan_id' => $paymentInfo['plan_id'],
                                    'plan_name' => $paymentInfo['plan_name'],
                                    'plan_duration' => $paymentInfo['plan_duration'],
                                    'is_tv_plan' => $paymentInfo['is_tv_plan']
                                ];
                            }

                            // Ativando a assinatura manualmente
                            $this->processarPagamentoConfirmado($reference, $transactionId);

                            // Retornando sucesso
                            return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
                        } else {
                            $this->logToFile('Status na API ainda não confirmado: ' . ($statusResponse['status'] ?? 'N/A'));
                        }
                    } catch (\Exception $apiEx) {
                        $this->logToFile('Erro ao consultar API de status: ' . $apiEx->getMessage());
                    }
                }

                // 4. VERIFICAÇÃO POR TEMPO: Se passou muito tempo desde o início do pagamento
                if (isset($paymentInfo['created_at'])) {
                    $createdAt = Carbon::parse($paymentInfo['created_at']);
                    $minutesPassed = $createdAt->diffInMinutes(now());

                    // Se passar de 5 minutos após a verificação manual, consideramos como pago
                    if ($minutesPassed > 5) {
                        $this->logToFile('Pagamento sendo ativado por timeout após verificação manual');
                        $this->processarPagamentoConfirmado($reference, $transactionId ?? 'manual_activation');
                        return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
                    }
                }
            }

            // Se chegou aqui, ainda está pendente
            return response()->json(['status' => 'pending']);

        } catch (\Exception $e) {
            $this->logToFile('Erro ao verificar status: ' . $e->getMessage());
            return response()->json(['status' => 'pending']);
        }
    }

    /**
     * Consulta o status de uma transação na API da SuitPay
     */
    private function checkTransactionStatus($transactionId)
    {
        $this->logToFile('Consultando status da transação na API: ' . $transactionId);

        $url = $this->baseUrl . '/payment/status/' . $transactionId;

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Accept: application/json',
                'ci: ' . $this->clientId,
                'cs: ' . $this->clientSecret
            ]
        ]);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($error) {
            $this->logToFile('Erro na consulta de status: ' . $error);
            throw new \Exception('Erro ao consultar status: ' . $error);
        }

        $this->logToFile('Resposta da API (HTTP ' . $httpCode . '): ' . $response);

        $responseData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->logToFile('Resposta inválida da API de status');
            throw new \Exception('Resposta inválida da API');
        }

        return $responseData;
    }

    /**
     * Implementação da rota de teste para ajudar a diagnosticar problemas
     */
    public function checkPaymentDebug($reference = null)
    {
        // Apenas acessível em ambiente de desenvolvimento
        if (!app()->environment('local')) {
            return response()->json(['error' => 'Não disponível em produção'], 403);
        }

        $output = [];

        // Se nenhuma referência for passada, mostrar todas as pendentes
        if (!$reference) {
            $pendingPayments = session()->get('pending_payments', []);
            $paymentInfos = [];

            // Buscar todos os payment_info_ na sessão
            foreach ($_SESSION as $key => $value) {
                if (strpos($key, 'payment_info_') === 0) {
                    $ref = str_replace('payment_info_', '', $key);
                    $paymentInfos[$ref] = $value;
                }
            }

            $output['pending_payments_count'] = count($pendingPayments);
            $output['payment_infos_count'] = count($paymentInfos);
            $output['payment_infos'] = $paymentInfos;

            // Buscar usuários atualizados recentemente
            $recentUsers = User::where('updated_at', '>', now()->subHours(24))
                              ->where(function($query) {
                                  $query->where('premuim', 1)
                                        ->orWhere('tv_access', 1);
                              })
                              ->select('id', 'name', 'email', 'premuim', 'tv_access', 'expired_in', 'expired_in_tv', 'pack_name', 'transaction_id', 'updated_at')
                              ->orderByDesc('updated_at')
                              ->limit(20)
                              ->get();

            $output['recent_users'] = $recentUsers;

            return response()->json($output);
        } else {
            // Verificar dados de uma referência específica
            $paymentInfo = session()->get('payment_info_' . $reference);
            $output['payment_info'] = $paymentInfo;

            // Tentar buscar metadados
            $metadata = $this->getMetadataFromRequestNumber($reference);
            $output['metadata'] = $metadata;

            // Se tiver transaction_id, verificar na API
            if ($paymentInfo && isset($paymentInfo['transaction_id'])) {
                try {
                    $statusResponse = $this->checkTransactionStatus($paymentInfo['transaction_id']);
                    $output['api_status'] = $statusResponse;
                } catch (\Exception $e) {
                    $output['api_error'] = $e->getMessage();
                }
            }

            return response()->json($output);
        }
    }

    /**
     * Cria uma página de diagnóstico para webhooks
     */
    public function webhookStatus()
    {
        // Verifica se o usuário é administrador
        if (!auth()->check() || !auth()->user()->admin) {
            abort(403, 'Acesso não autorizado');
        }

        // Informações sobre webhooks recebidos
        $data = [
            'webhooks_today' => $this->getWebhooksCount(now()->format('Y-m-d')),
            'last_30_days' => $this->getWebhooksLastDays(30),
            'last_webhook' => $this->getLastWebhookInfo(),
            'webhook_route' => url('/suitpay/callback'),
            'webhook_test_route' => url('/suitpay/test-webhook'),
            'webhooks_directory' => storage_path('logs/suitpay_webhooks'),
        ];

        return view('admin.suitpay_webhook_status', $data);
    }

    /**
     * Retorna o número de webhooks recebidos em um dia específico
     */
    private function getWebhooksCount($date)
    {
        $counterFile = storage_path('logs/suitpay_webhooks/counter.json');
        if (!file_exists($counterFile)) {
            return 0;
        }

        $counter = json_decode(file_get_contents($counterFile), true) ?: [];
        return $counter[$date] ?? 0;
    }

    /**
     * Retorna estatísticas dos últimos X dias
     */
    private function getWebhooksLastDays($days)
    {
        $counterFile = storage_path('logs/suitpay_webhooks/counter.json');
        if (!file_exists($counterFile)) {
            return [];
        }

        $counter = json_decode(file_get_contents($counterFile), true) ?: [];

        // Preencher com zeros os dias sem webhooks
        $result = [];
        for ($i = $days - 1; $i >= 0; $i--) {
            $date = now()->subDays($i)->format('Y-m-d');
            $result[$date] = $counter[$date] ?? 0;
        }

        return $result;
    }

    /**
     * Retorna informações do último webhook recebido
     */
    private function getLastWebhookInfo()
    {
        $dir = storage_path('logs/suitpay_webhooks');
        if (!file_exists($dir)) {
            return null;
        }

        // Procurar pelo arquivo de webhook mais recente
        $files = glob($dir . '/webhook_*.json');
        if (empty($files)) {
            return null;
        }

        // Ordenar por data de modificação (mais recente primeiro)
        usort($files, function($a, $b) {
            return filemtime($b) - filemtime($a);
        });

        // Retornar informações do webhook mais recente
        $latestFile = $files[0];
        if (file_exists($latestFile)) {
            $content = file_get_contents($latestFile);
            $webhook = json_decode($content, true);
            return [
                'time' => $webhook['timestamp'] ?? 'Desconhecido',
                'ip' => $webhook['ip'] ?? 'Desconhecido',
                'data' => $webhook['data'] ?? [],
                'file' => basename($latestFile)
            ];
        }

        return null;
    }

    /**
     * Endpoint para testar se o webhook está funcionando
     */
    public function testWebhook(Request $request)
    {
        // Esta rota permite gerar um webhook de teste
        // Útil para verificar se o sistema está configurado corretamente

        // Simular uma requisição do SuitPay
        $testData = [
            'requestNumber' => 'test_' . uniqid(),
            'statusTransaction' => 'TEST',
            'idTransaction' => 'test_' . time(),
            'test' => true,
            'timestamp' => now()->format('Y-m-d H:i:s'),
            'message' => 'Este é um webhook de teste para verificar a configuração.'
        ];

        // Log de teste
        $this->logWebhook(new Request($testData));

        return response()->json([
            'success' => true,
            'message' => 'Webhook de teste processado com sucesso',
            'data' => $testData
        ]);
    }

    /**
     * Verifica se a API do SuitPay está acessível
     */
    public function checkApiConnection()
    {
        try {
            // Informações a serem coletadas
            $info = [
                'server_info' => [
                    'php_version' => PHP_VERSION,
                    'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                    'server_ip' => $_SERVER['SERVER_ADDR'] ?? gethostbyname(gethostname()),
                    'client_ip' => $_SERVER['REMOTE_ADDR'] ?? 'Unknown',
                ],
                'suitpay_config' => [
                    'api_url' => $this->baseUrl,
                    'sandbox_mode' => $this->isSandbox ? 'Ativo' : 'Desativado',
                    'client_id_set' => !empty($this->clientId) ? 'Sim' : 'Não',
                    'client_secret_set' => !empty($this->clientSecret) ? 'Sim' : 'Não',
                ],
                'connectivity' => [
                    'status' => 'Verificando...',
                    'message' => '',
                    'response' => null,
                ],
                'webhook_url' => [
                    'local' => url('/suitpay/callback'),
                    'public' => $this->detectPublicUrl('/suitpay/callback'),
                ],
                'recomendations' => [],
            ];

            // Testa a conexão com a API da SuitPay
            if (!empty($this->clientId) && !empty($this->clientSecret)) {
                try {
                    // Tenta fazer uma requisição simples para a API
                    $response = $this->makeApiRequest('/payment/status/test-connection', []);
                    $info['connectivity']['status'] = 'Sucesso';
                    $info['connectivity']['message'] = 'Conexão com a API SuitPay estabelecida com sucesso.';
                    $info['connectivity']['response'] = $response;
                } catch (\Exception $e) {
                    $info['connectivity']['status'] = 'Falha';
                    $info['connectivity']['message'] = 'Erro ao conectar com a API: ' . $e->getMessage();

                    // Adicionar recomendações baseadas no erro
                    if (strpos($e->getMessage(), 'cURL error') !== false) {
                        $info['recomendations'][] = 'Verifique se a URL da API está correta e acessível.';
                        $info['recomendations'][] = 'Verifique se o servidor tem acesso à internet.';
                        $info['recomendations'][] = 'Verifique se há algum firewall ou proxy bloqueando a conexão.';
                    } else if (strpos($e->getMessage(), '401') !== false || strpos($e->getMessage(), 'Unauthorized') !== false) {
                        $info['recomendations'][] = 'Verifique se as credenciais (clientId e clientSecret) estão corretas.';
                    }
                }
            } else {
                $info['connectivity']['status'] = 'Não configurado';
                $info['connectivity']['message'] = 'Credenciais da API não configuradas.';
                $info['recomendations'][] = 'Configure as variáveis de ambiente SUITPAY_CLIENT_ID e SUITPAY_CLIENT_SECRET.';
            }

            // Adicionar mais recomendações gerais
            if (strpos($info['webhook_url']['public'], 'localhost') !== false ||
                strpos($info['webhook_url']['public'], '127.0.0.1') !== false) {
                $info['recomendations'][] = 'Seu servidor parece estar em um ambiente local (localhost). A SuitPay não pode enviar webhooks para servidores locais.';
                $info['recomendations'][] = 'Use um serviço como ngrok para expor temporariamente seu servidor local à internet.';
            }

            return view('admin.suitpay_connectivity', ['info' => $info]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => true,
                'message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Tenta detectar a URL pública do servidor
     */
    private function detectPublicUrl($path)
    {
        $url = url($path);

        // Verificar se é uma URL localhost
        if (strpos($url, 'localhost') !== false || strpos($url, '127.0.0.1') !== false) {
            // Tentar detectar URL pública usando serviços como ngrok
            $headers = getallheaders();
            if (isset($headers['X-Forwarded-Host'])) {
                $protocol = isset($headers['X-Forwarded-Proto']) ? $headers['X-Forwarded-Proto'] : 'http';
                return $protocol . '://' . $headers['X-Forwarded-Host'] . $path;
            }

            // Caso contrário, retornar a URL local com um aviso
            return $url . ' (Aviso: URL local não acessível pela internet)';
        }

        return $url;
    }

    /**
     * Mostra a página de sucesso do pagamento
     */
    public function showSuccess()
    {
        return view('pagamento_sucesso');
    }

    /**
     * Mostra a página de falha do pagamento
     */
    public function showFailure()
    {
        return view('pagamento_falha');
    }
}
