<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\UsuarioSorteio;
use Illuminate\Support\Facades\Auth;
use App\Sorteio;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;




class SorteioAdminController extends Controller {

    public function gerarDatasSorteio()
{
    // Chamar a função do modelo para gerar as datas
    Sorteio::generateNextSorteios();

    // Redirecionar de volta para o dashboard com uma mensagem
    return redirect()->route('sorteioadmin.dashboard')->with('success', 'Datas de sorteio geradas com sucesso!');
}

public function create()
{
    return view('sorteioadmin.create');
}

public function mostrarProximosSorteios() {
    $sorteios = Sorteio::where('data_sorteio', '>', now())->orderBy('data_sorteio', 'asc')->get();
    return view('proximos_sorteios', ['sorteios' => $sorteios]);
}

public function edit($id) {
    $sorteio = Sorteio::find($id);
    
    if (!$sorteio) {
        return redirect()->route('sorteioadmin.dashboard')->with('error', 'Sorteio não encontrado.');
    }

    return view('sorteioadmin.edit', compact('sorteio'));
}

public function update(Request $request, $id) {
    $sorteio = Sorteio::find($id);
    
    if (!$sorteio) {
        return redirect()->route('sorteioadmin.dashboard')->with('error', 'Sorteio não encontrado.');
    }

    $data = $request->validate([
        'titulo' => 'required|string|max:255',
        'descricao' => 'required|string',
        'data_sorteio' => 'required|date|after:now',
        'imagem' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048' // Validação básica de imagem
    ]);

    $sorteio->titulo = $request->titulo;
    $sorteio->descricao = $request->descricao;
    $sorteio->data_sorteio = $request->data_sorteio;

    if ($request->hasFile('imagem')) {
        // Remove a imagem antiga se existente
        if ($sorteio->imagem) {
            Storage::disk('public')->delete($sorteio->imagem);
        }

        $sorteio->imagem = $request->imagem->store('sorteios', 'public');
    }

    $sorteio->save();

    return redirect()->route('sorteioadmin.dashboard')->with('success', 'Sorteio atualizado com sucesso!');
}

public function destroy($id) {
    $sorteio = Sorteio::find($id);

    if (!$sorteio) {
        return redirect()->route('sorteioadmin.dashboard')->with('error', 'Sorteio não encontrado.');
    }

    // Deleta a imagem associada ao sorteio
    if ($sorteio->imagem) {
        Storage::disk('public')->delete($sorteio->imagem);
    }

    $sorteio->delete();

    return redirect()->route('sorteioadmin.dashboard')->with('success', 'Sorteio excluído com sucesso!');
}


public function store(Request $request)
{
    $data = $request->validate([
        'titulo' => 'required|string|max:255',
        'descricao' => 'required|string',
        'data_sorteio' => 'required|date|after:now',
        'imagem' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048' // Validação básica de imagem
    ]);

    // Lidando com o upload da imagem
    $sorteio = new Sorteio;
    $sorteio->titulo = $request->titulo;
    $sorteio->descricao = $request->descricao;
    $sorteio->data_sorteio = $request->data_sorteio;

    if ($request->hasFile('imagem')) {
        $sorteio->imagem = $request->imagem->store('sorteios', 'public');
    }

    $sorteio->save();

    return redirect()->route('sorteioadmin.dashboard')->with('success', 'Sorteio criado com sucesso!');
}




public function showLogin()
{
    return view('sorteioadmin.login');
}

    public function login(Request $request)
    {
        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials) && Auth::user()->role == 'admin') {
            return redirect()->route('sorteioadmin.dashboard');
        }

        return back()->with('error', 'Credenciais inválidas ou você não é administrador.');
    }

    public function marcarSorteado($id, Request $request)
    {
        $usuario = UsuarioSorteio::findOrFail($id);
        $usuario->sorteado = 1;
        $usuario->save();

        return redirect()->route('sorteioadmin.dashboard')->with('success', 'Usuário marcado como sorteado!');
    }

    public function dashboard(Request $request)
{
    $query = UsuarioSorteio::query();

    if ($request->has('search') && trim($request->search) != '') {
        $query->where('nome', 'LIKE', '%' . $request->search . '%')
            ->orWhere('email', 'LIKE', '%' . $request->search . '%');
    }

    $usuarios = $query->get();
    $sorteios = Sorteio::all();

    return view('sorteioadmin.dashboard', compact('usuarios', 'sorteios'));
}

    public function logout(Request $request) {
        Auth::logout();
        return redirect()->route('sorteioadmin.login');
    }

    public function realizarSorteio($sorteio_id)
{
    $sorteio = Sorteio::find($sorteio_id);
    
    if (!$sorteio) {
        return redirect()->back()->with('error', 'Sorteio não encontrado.');
    }

    // Buscar todos os usuários que não foram sorteados anteriormente neste sorteio
    $usuariosNaoSorteados = UsuarioSorteio::where('sorteado', 0)->whereNotIn('id', function ($query) use ($sorteio_id) {
        $query->select('usuario_id')
            ->from('sorteio_vencedores')
            ->where('sorteio_id', $sorteio_id);
    })->get();

    if (!$usuariosNaoSorteados->count()) {
        return redirect()->back()->with('message', 'Todos os usuários já foram sorteados para este sorteio!');
    }

    $vencedorSorteado = $usuariosNaoSorteados->random();

    // Registrar o vencedor na tabela de junção sorteio_vencedores
    DB::table('sorteio_vencedores')->insert([
        'sorteio_id' => $sorteio->id,
        'usuario_id' => $vencedorSorteado->id,
    ]);

    // Atualizar o status do usuário como sorteado
    $vencedorSorteado->sorteado = 1;
    $vencedorSorteado->save();

    return response()->json(['success' => true, 'winner' => $vencedorSorteado]);


}


public function resetVencedores()
{
    // Deleta todos os registros na tabela sorteio_vencedores
    DB::table('sorteio_vencedores')->delete();

    // Mude o status da coluna sorteado da tabela usuarios_sorteio de 1 para 0
    DB::table('usuarios_sorteio')->where('sorteado', 1)->update(['sorteado' => 0]);

    // Após as operações, redirecione o administrador de volta com uma mensagem de sucesso.
    return redirect()->back()->with('message', 'Vencedores resetados e status de sorteio atualizado com sucesso!');
}


public function showWinners()
{
    $sorteiosComVencedores = Sorteio::with('vencedores')->get();

    return view('winners', ['sorteios' => $sorteiosComVencedores]);
}


}