<?php

namespace App\Http\Controllers;

use App\Embed;
use App\Episode;
use App\Genre;
use App\Network;
use App\Cast;
use App\Http\Requests\SerieStoreRequest;
use App\Http\Requests\SerieUpdateRequest;
use App\Http\Requests\StoreImageRequest;
use App\Jobs\SendNotification;
use App\Season;
use App\Serie;
use App\Anime;
use App\SerieGenre;
use App\SerieVideo;
use App\SerieCast;
use App\SerieNetwork;
use App\SerieSubstitle;
use App\SerieDownload;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Profile;
use ZipArchive;


class SerieController extends Controller
{


    public function __construct()
    {
        $this->middleware('doNotCacheResponse', ['only' => ['moviecomment','addcomment']]);
    }


    // returns all Series except children Series, for api.
    public function index()
    {
        $serie = Serie::query()->whereHas('genres', function ($genre) {
            $genre->where('genre_id', '=', 10762);
        })->orderByDesc('id')->paginate(12);

        return response()->json($serie, 200);

    }



    public function moviecomment($movie)
    {

        $movie = Serie::where('id', '=', $movie)->first();

        $comments = $movie->comments;

        return response()
            ->json(['comments' => $comments], 200);

    }



    public function addcomment(Request $request)
    {


        $user = Auth::user();


        $this->validate($request, [
            'comments_message' => 'required',
            'movie_id' => 'required'
        ]);

        $movie = Serie::where('id', '=', $request->movie_id)->first();

        $comment = $movie->commentAsUser($user, $request->comments_message);

        return response()->json($comment, 200);

    }


    public function addtofav($serieId, Request $request)
{
    $profileId = session('selected_profile')->id;
    $profile = Profile::find($profileId);

    if (!$profile) {
        return response()->json(['message' => 'Perfil não encontrado', 'isFavorite' => false], 404);
    }

    $serie = Serie::find($serieId);
    if (!$serie) {
        return response()->json(['message' => 'Série não encontrada', 'isFavorite' => false], 404);
    }

    try {
        // Verificar se a série já está nos favoritos usando SQL direto para evitar problemas com cache
        $exists = DB::table('favorites')
                    ->where('profile_id', $profileId)
                    ->where('favoriteable_id', $serieId)
                    ->where('favoriteable_type', Serie::class)
                    ->exists();

        if (!$exists) {
            DB::beginTransaction();

            DB::table('favorites')->insert([
                'profile_id' => $profileId,
                'favoriteable_id' => $serieId,
                'favoriteable_type' => Serie::class,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Adicionado aos favoritos',
                'isFavorite' => true
            ]);
        }

        return response()->json([
            'message' => 'Série já está nos favoritos',
            'isFavorite' => true
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Erro ao adicionar favorito: ' . $e->getMessage());
        return response()->json([
            'message' => 'Erro ao adicionar aos favoritos: ' . $e->getMessage(),
            'isFavorite' => false,
            'error' => $e->getMessage()
        ], 500);
    }
}

public function removefromfav($serieId, Request $request)
{
    $profileId = session('selected_profile')->id;
    $profile = Profile::find($profileId);

    if (!$profile) {
        return response()->json(['message' => 'Perfil não encontrado', 'isFavorite' => false], 404);
    }

    $serie = Serie::find($serieId);
    if (!$serie) {
        return response()->json(['message' => 'Série não encontrada', 'isFavorite' => false], 404);
    }

    try {
        DB::beginTransaction();

        // Remover diretamente utilizando query builder para garantir consistência
        $deleted = DB::table('favorites')
                    ->where('profile_id', $profileId)
                    ->where('favoriteable_id', $serieId)
                    ->where('favoriteable_type', Serie::class)
                    ->delete();

        DB::commit();

        return response()->json([
            'message' => 'Removido dos favoritos',
            'isFavorite' => false
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Erro ao remover favorito: ' . $e->getMessage());
        return response()->json([
            'message' => 'Erro ao remover dos favoritos: ' . $e->getMessage(),
            'isFavorite' => true,
            'error' => $e->getMessage()
        ], 500);
    }
}

public function isMovieFavorite($serieId, Request $request)
{
    $profileId = session('selected_profile')->id;

    if (!$profileId) {
        return response()->json(['status' => 404, 'error' => 'Perfil não encontrado', 'isFavorite' => 0], 404);
    }

    try {
        // Verificar se a série está nos favoritos usando SQL direto para evitar problemas com cache
        $isFavorite = DB::table('favorites')
                        ->where('profile_id', $profileId)
                        ->where('favoriteable_id', $serieId)
                        ->where('favoriteable_type', Serie::class)
                        ->exists();

        return response()->json([
            'status' => 200,
            'isFavorite' => $isFavorite ? 1 : 0
        ]);
    } catch (\Exception $e) {
        \Log::error('Erro ao verificar favorito: ' . $e->getMessage());
        return response()->json([
            'status' => 500,
            'error' => 'Erro ao verificar favorito',
            'message' => $e->getMessage(),
            'isFavorite' => 0
        ], 500);
    }
}

    public function userfav()
    {


        $user = Auth::user();
        $user->favorite(Serie::class);

        return response()->json($user, 200);


    }



    // returns all Series for admin panel
    public function data()
    {


        return response()->json(Serie::with('seasons.episodes.videos','genres','casters','networks')->orderByDesc('created_at')
        ->paginate(12), 200);



    }


    public function videos($id)
    {




    }


    // returns a specific Serie
    public function show($id)
    {

        $serie = Serie::with(['casters.cast' => function ($query) {
            $query->select('id', 'name','original_name','profile_path');
        },'seasons.episodes.videos'])->where('id', '=', $id)->first()->makeHidden(['casters','networks']);

        $serie->increment('views',1);

        return response()->json($serie, 200);
    }


    public function detail($id)
    {
        // Carrega a série com suas relações
        $serie = Serie::with([
            'casters.cast' => function ($query) {
                $query->select('id', 'name', 'original_name', 'profile_path');
            },
            'seasons.episodes.videos', // Garante que os vídeos sejam carregados
            'genres'
        ])->where('id', '=', $id)->first();

        // Verifica se a série foi encontrada
        if (!$serie) {
            abort(404, 'Série não encontrada');
        }

        // Incrementa visualizações
        $serie->increment('views', 1);

        // Debug: verifica contagem de temporadas e episódios
        $seasonsCount = $serie->seasons->count();
        $episodesCount = $serie->seasons->flatMap->episodes->count();
        $videosCount = $serie->seasons->flatMap->episodes->flatMap->videos->count();

        \Log::info("Serie loaded: {$serie->name}, Seasons: {$seasonsCount}, Episodes: {$episodesCount}, Videos: {$videosCount}");

        // Extrai os IDs dos gêneros da série atual
        $genreIds = $serie->genres->pluck('genre_id')->toArray();

        // Verifica se temos gêneros antes de fazer a consulta
        if (!empty($genreIds)) {
            // Corrige a consulta para usar 'genre_id' em vez de 'genres.id'
            $seriesRelacionadas = Serie::whereHas('genres', function ($query) use ($genreIds) {
                $query->whereIn('genre_id', $genreIds); // Referência correta à coluna
            })
            ->where('id', '!=', $id)
            ->where('active', 1)
            ->inRandomOrder()
            ->limit(12)
            ->get();
        } else {
            // Se não houver gêneros, busque séries aleatórias
            $seriesRelacionadas = Serie::where('id', '!=', $id)
                ->where('active', 1)
                ->inRandomOrder()
                ->limit(12)
                ->get();
        }

        // Retorna a view com os dados
        return view('serie', [
            'serie' => $serie,
            'seriesRelacionadas' => $seriesRelacionadas,
            'debug_info' => [
                'seasons_count' => $seasonsCount,
                'episodes_count' => $episodesCount,
                'videos_count' => $videosCount
            ]
        ]);
    }

    public function getSeasonEpisodes($seasonId)
    {
        $season = Serie::with(['seasons.episodes.videos'])->whereHas('seasons', function ($query) use ($seasonId) {
            $query->where('id', $seasonId);
        })->first();

        if (!$season) {
            abort(404); // Retorna um erro 404 se a temporada não for encontrada
        }

        return response()->json($season->seasons[0]->episodes);
    }

    public function episodesBySeason($seasonId)
    {
        $season = Season::with('episodes')->findOrFail($seasonId);
        $episodes = $season->episodes;

        return response()->json($episodes);
    }

    public function episodeDetail($videoId)
    {
        // Busque o episódio pelo ID do vídeo
        $episode = Episode::with('season.serie')->findOrFail($videoId);

        // Retorne a view com os dados do episódio
        return view('episode', compact('episode'));

        // return response()->json($episode);
    }



    // create a new Serie in the database
    public function store(SerieStoreRequest $request)
    {
        $serie = new Serie();
        $serie->fill($request->serie);
        $serie->name = str_replace(':', '', $serie->name);
        $serie->save();

        $this->onSaveSerieGenre($request,$serie);
        $this->onSaveSerieSeasons($request,$serie);
        $this->onSaveSerieCasters($request,$serie);
        $this->onSaveSerieNetworks($request,$serie);


        if ($request->notification) {
            $this->dispatch(new SendNotification($serie));
        }

        $data = [
            'status' => 200,
            'message' => 'successfully created',
            'body' => $serie->load('seasons.episodes.videos')
        ];

        return response()->json($data, $data['status']);
    }

    public function onSaveSerieCasters($request,$serie) {

        if ($request->serie['casterslist']) {
            foreach ($request->serie['casterslist'] as $cast) {
                $find = Cast::find($cast['id']);
                if ($find == null) {
                    $find = new Cast();
                    $find->fill($cast);
                    $find->save();
                }
                $movieGenre = new SerieCast();
                $movieGenre->cast_id = $cast['id'];
                $movieGenre->serie_id = $serie->id;
                $movieGenre->save();
            }
        }

    }



    public function onSaveSerieNetworks($request,$serie) {

        if ($request->serie['networks']) {
            foreach ($request->serie['networks'] as $network) {
                $find = Network::find($network['id']);
                if ($find == null) {
                    $find = new Network();
                    $find->fill($network);
                    $find->save();
                }
                $serieNetwork = new SerieNetwork();
                $serieNetwork->network_id = $network['id'];
                $serieNetwork->serie_id = $serie->id;
                $serieNetwork->save();
            }
        }

    }

    public function onSaveSerieGenre($request,$serie) {

        if ($request->serie['genres']) {
            foreach ($request->serie['genres'] as $genre) {
                $find = Genre::find($genre['id']);
                if ($find == null) {
                    $find = new Genre();
                    $find->fill($genre);
                    $find->save();
                }
                $serieGenre = new SerieGenre();
                $serieGenre->genre_id = $genre['id'];
                $serieGenre->serie_id = $serie->id;
                $serieGenre->save();
            }
        }

    }


    public function onSaveSerieSeasons($request , $serie){

        if ($request->serie['seasons']) {
            foreach ($request->serie['seasons'] as $reqSeason) {
                $season = new Season();
                $season->fill($reqSeason);
                $season->serie_id = $serie->id;
                $season->save();

                $this->onSaveEpisodes($request,$reqSeason,$season);


            }
        }

    }


    public function onSaveEpisodes($request, $reqSeason,$season) {

        if ($reqSeason['episodes']) {
            foreach ($reqSeason['episodes'] as $reqEpisode) {
                $episode = new Episode();
                $episode->fill($reqEpisode);
                $episode->season_id = $season->id;
                $episode->save();


                if (isset($reqEpisode['videos'])) {
                    foreach ($reqEpisode['videos'] as $reqVideo) {

                        $video = SerieVideo::query()->find($reqVideo['id'] ?? 0) ?? new SerieVideo();
                        $video->fill($reqVideo);
                        $video->episode_id = $episode->id;
                        $video->save();
                    }
                }


               $this->onSaveEpisodeSubstitle($request,$reqEpisode,$episode);
               $this->onSaveEpisodeDownload($request,$reqEpisode,$episode);
            }
        }


    }



    public function onSaveEpisodeDownload($request,$reqEpisode,$episode) {

        if (isset($reqEpisode['downloads'])) {
            foreach ($reqEpisode['downloads'] as $reqVideo) {
                $video = SerieDownload::find($reqVideo['id'] ?? 0) ?? new SerieDownload();
                $video->fill($reqVideo);
                $video->episode_id = $episode->id;
                $video->save();
            }
        }

    }


    public function onSaveEpisodeSubstitle($request,$reqEpisode,$episode) {


         if (isset($reqEpisode['substitles'])) {
                    foreach ($reqEpisode['substitles'] as $reqVideo) {
                        $video = new SerieSubstitle();
                        $video->fill($reqVideo);
                        $video->episode_id = $episode->id;
                        $video->save();
                    }
                }
    }

    // update a Serie in the database
    public function update(SerieUpdateRequest $request, Serie $serie)
    {

        $serie->fill($request->serie);
        $serie->name = str_replace(':', '', $serie->name);
        $serie->save();

        $this->onUpdateSerieGenre($request,$serie);
        $this->onUpdateSerieSeasons($request,$serie);
        $this->onUpdateSerieCasts($request,$serie);
        $this->onUpdateSerieNetwork($request,$serie);

        $data = [
            'status' => 200,
            'message' => 'successfully updated',
            'body' => "Success"
        ];

        return response()->json($data, $data['status']);
    }





    public function onUpdateSerieCasts ($request,$serie) {


        if ($request->serie['casterslist']) {
            foreach ($request->serie['casterslist'] as $genre) {

                    $find = Cast::find($genre['id'] ?? 0) ?? new Cast();
                    $find->fill($genre);
                    $find->save();
                    $movieGenre = SerieCast::where('serie_id', $serie->id)
                        ->where('cast_id', $genre['id'])->get();

                    if (count($movieGenre) < 1) {
                        $movieGenre = new SerieCast();
                        $movieGenre->cast_id = $genre['id'];
                        $movieGenre->serie_id = $serie->id;
                        $movieGenre->save();

                    }

            }
        }

    }



    public function onUpdateSerieNetwork ($request,$serie) {

        if ($request->serie['networks']) {
            foreach ($request->serie['networks'] as $netwok) {
                if (!isset($netwok['network_id'])) {
                    $find = Network::find($netwok['id']) ?? new Network();
                    $find->fill($netwok);
                    $find->save();
                    $serieNetwork = SerieNetwork::where('serie_id', $serie->id)->where('network_id', $netwok['id'])->get();
                    if (count($serieNetwork) < 1) {
                        $serieNetwork = new SerieNetwork();
                        $serieNetwork->network_id = $netwok['id'];
                        $serieNetwork->serie_id = $serie->id;
                        $serieNetwork->save();
                    }
                }
            }
        }

    }

    public function onUpdateSerieGenre ($request,$serie) {

        if ($request->serie['genres']) {
            foreach ($request->serie['genres'] as $genre) {
                if (!isset($genre['genre_id'])) {
                    $find = Genre::find($genre['id']) ?? new Genre();
                    $find->fill($genre);
                    $find->save();
                    $serieGenre = SerieGenre::where('serie_id', $serie->id)->where('genre_id', $genre['id'])->get();
                    if (count($serieGenre) < 1) {
                        $serieGenre = new SerieGenre();
                        $serieGenre->genre_id = $genre['id'];
                        $serieGenre->serie_id = $serie->id;
                        $serieGenre->save();
                    }
                }
            }
        }

    }


    public function onUpdateSerieSeasons($request,$serie) {


        if ($request->serie['seasons']) {
            foreach ($request->serie['seasons'] as $reqSeason) {
                $season = Season::find($reqSeason['id'] ?? 0) ?? new Season();
                $season->fill($reqSeason);
                $season->serie_id = $serie->id;
                $season->save();


                $this->onUpdateSerieEpisodes($request,$reqSeason,$season);
            }
        }


    }




    public function onUpdateSerieEpisodes ($request,$reqSeason,$season) {

        if ($reqSeason['episodes']) {
                    foreach ($reqSeason['episodes'] as $reqEpisode) {
                        $episode = Episode::find($reqEpisode['id'] ?? 0) ?? new Episode();
                        $episode->fill($reqEpisode);
                        $episode->season_id = $season->id;
                        $episode->save();
                        if (isset($reqEpisode['videos'])) {
                            foreach ($reqEpisode['videos'] as $reqVideo) {

                                $video = SerieVideo::find($reqVideo['id'] ?? 0) ?? new SerieVideo();
                                $video->fill($reqVideo);
                                $video->episode_id = $episode->id;
                                $video->save();
                            }
                        }


                        $this->onUpdateSerieSubstitle($request,$reqEpisode,$episode);
                        $this->onUpdateSerieDownload($request,$reqEpisode,$episode);
                    }
                }

    }



    public function onUpdateSerieDownload ($request,$reqEpisode,$episode) {

        if (isset($reqEpisode['downloads'])) {
            foreach ($reqEpisode['downloads'] as $reqVideo) {

                $substitle = SerieDownload::find($reqVideo['id'] ?? 0) ?? new SerieDownload();
                $substitle->fill($reqVideo);
                $substitle->episode_id = $episode->id;
                $substitle->save();
            }

    }

    }


    public function onUpdateSerieSubstitle ($request,$reqEpisode,$episode) {

        if (isset($reqEpisode['substitles'])) {
            foreach ($reqEpisode['substitles'] as $reqVideo) {

                $substitle = SerieSubstitle::find($reqVideo['id'] ?? 0) ?? new SerieSubstitle();
                $substitle->fill($reqVideo);
                $substitle->episode_id = $episode->id;
                $substitle->save();
            }


    }

}


    // delete a Serie from the database
    public function destroy(Serie $serie)
    {
        if ($serie != null) {
            $serie->delete();

            $data = [
                'status' => 200,
                'message' => 'successfully deleted',
            ];
        } else {
            $data = [
                'status' => 400,
                'message' => 'could not be deleted',
            ];
        }


        return response()->json($data, $data['status']);
    }

    // remove a genre from a Series from the database
    public function destroyGenre($genre)
    {
        if ($genre != null) {

            //SerieGenre::find($genre)->delete();
            $movie = SerieGenre::where('id', '=', $genre)->first();
            $movie->delete();

            $data = ['status' => 200, 'message' => 'successfully deleted',];
        } else {
            $data = ['status' => 400, 'message' => 'could not be deleted',];
        }

        return response()->json($data, 200);
    }


     // remove the cast of a movie from the database
     public function destroyCast($id)
     {

         if ($id != null) {

             $movie = SerieCast::where('cast_id', '=', $id)->first();
             $movie->delete();
             $data = ['status' => 200, 'message' => 'successfully deleted',];
         } else {
             $data = [
                 'status' => 400,
                 'message' => 'could not be deleted',
             ];
         }

         return response()->json($data, $data['status']);

     }




// remove Network from  a movie
public function destroyNetworks($id)
{

    if ($id != null) {

        SerieNetwork::find($id)->delete();
        $data = ['status' => 200, 'message' => 'successfully deleted',];
    } else {
        $data = [
            'status' => 400,
            'message' => 'could not be deleted',
        ];
    }

    return response()->json($data, $data['status']);

}



    // save a new image in the Series folder of the storage
    public function storeImg(StoreImageRequest $request)
    {

        if ($request->hasFile('image')) {
            $filename = Storage::disk('series')->put('', $request->image);
            $data = [
                'status' => 200,
                'image_path' => $request->root() . '/api/series/image/' . $filename,
                'message' => 'image uploaded successfully'
            ];
        } else {
            $data = [
                'status' => 400,
                'message' => 'there was an error uploading the image'
            ];
        }

        return response()->json($data, $data['status']);
    }

    // return an image from the Series folder of the storage
    public function getImg($filename)
    {

        $image = Storage::disk('series')->get($filename);

        $mime = Storage::disk('series')->mimeType($filename);

        return (new Response($image, 200))
            ->header('Content-Type', $mime);
    }


    // returns a specific Serie
    public function showbyimdb($serie)
    {

        $movie = Serie::where('tmdb_id', '=', $movie)->orWhere('id', '=', $movie)->first();

        $movie->increment('views',1);

        return response()->json($movie, 200);

    }




    // return the 10 Series with the highest average votes
    public function recommended()
    {


        $movies = Serie::select('series.id','series.name','series.poster_path','series.vote_average','series.subtitle')
        ->orderByDesc('vote_average')->where('active', '=', 1)->limit(10)->get();

    return response()->json(['recommended' =>
    $movies->makeHidden(['casterslist','casters','seasons','overview','backdrop_path','preview_path','videos'
    ,'substitles','vote_count','popularity','runtime','release_date','imdb_external_id','hd','pinned','preview'])], 200);


    }

    // return the 10 movies with the most popularity
    public function popular()
    {


        $movies = Serie::select('series.id','series.name','series.poster_path','series.vote_average'
        ,'series.subtitle')->where('active', '=', 1)
        ->orderByDesc('popularity')
        ->limit(10)->get();


    return response()->json(['popularSeries' => $movies->makeHidden(['casterslist','casters','seasons','overview','backdrop_path','preview_path','videos'
    ,'substitles','vote_count','popularity','runtime','release_date','imdb_external_id','hd','pinned','preview'])], 200);


    }

    // returns the last 10 Series added in the month
    public function recents()
    {
        $movies = Serie::select('series.id','series.name','series.poster_path',
        'series.vote_average','series.newEpisodes','series.subtitle')->where('created_at', '>', Carbon::now()->subMonth(3))
        ->where('active', '=', 1)
        ->orderByDesc('created_at')
        ->limit(10)->get();


    return response()->json(['recents' => $movies->makeHidden(['casterslist','casters','seasons','overview','backdrop_path','preview_path','videos'
    ,'substitles','vote_count','popularity','runtime','release_date','imdb_external_id','hd','pinned','preview'])], 200);

    }


     public function relateds($serie)
    {
        $moviesGenre = Serie::withOnly('genres.genre')
         ->where('id', '=', $serie)
         ->select(['series.id','series.name','series.poster_path',
         'series.vote_average','series.newEpisodes','series.subtitle'])
         ->where('active', '=', 1)
         ->limit(1)
         ->orderByDesc('series.created_at')
         ->get();



         foreach( $moviesGenre as $movie){


            $genre = $movie->genres[0]->genre_id;

        }



        $movies = DB::table('series')->join('serie_genres', 'series.id', '=', 'serie_genres.serie_id')
         ->where('serie_genres.genre_id', '=', $genre)
         ->where('serie_genres.serie_id', '!=', $serie)
         ->select(['series.id','series.name','series.poster_path',
         'series.vote_average','series.newEpisodes','series.subtitle'])
          ->where('active', '=', 1)
         ->limit(10)
         ->orderByDesc('series.created_at')
         ->get();


       return response()->json(['relateds' => $movies], 200);
    }





    public function showEpisodeFromNotifcation($id)
    {



        $order = 'desc';
        $series = Serie::where('active', '=', 1)->join('seasons', 'seasons.serie_id', '=', 'series.id')
        ->join('episodes', 'episodes.season_id', '=', 'seasons.id')
        ->join('serie_videos', 'serie_videos.episode_id', '=', 'episodes.id')
        ->orderBy('serie_videos.updated_at', $order)->select('serie_videos.episode_id','series.id','series.tmdb_id as serieTmdb'
        ,'series.name as serieName','episodes.still_path','episodes.season_id','seasons.season_number','seasons.name as seasons_name','seasons.name','episodes.name as episode_name','serie_videos.link','serie_videos.server','serie_videos.lang'
        ,'serie_videos.embed','serie_videos.youtubelink','serie_videos.hls','seasons.name as seasons_name','seasons.season_number','episodes.vote_average','episodes.overview as epoverview'
        ,'series.premuim','episodes.episode_number','series.poster_path','episodes.hasrecap','episodes.skiprecap_start_in'
        ,'serie_videos.supported_hosts','serie_videos.header','serie_videos.useragent','series.imdb_external_id' ,
        'serie_videos.drmuuid','serie_videos.drmlicenceuri','serie_videos.drm'
        )->addSelect(DB::raw("'serie' as type"))->where('episodes.id', '=', $id)
        ->limit(1)->get()->makeHidden(['seasons','episodes','casterslist']);

        $newEpisodes = [];
        foreach ($series as $item) {
            array_push($newEpisodes, $item);
        }


        return response()->json(['latest_episodes' => $newEpisodes], 200);

    }

    public function newEpisodes()
    {
        $order = 'desc';
        $series = Serie::where('active', '=', 1)->join('seasons', 'seasons.serie_id', '=', 'series.id')
        ->join('episodes', 'episodes.season_id', '=', 'seasons.id')
        ->join('serie_videos', 'serie_videos.episode_id', '=', 'episodes.id')
        ->orderBy('serie_videos.updated_at', $order)->select('serie_videos.episode_id','series.id','series.tmdb_id as serieTmdb'
        ,'series.name','episodes.still_path','episodes.season_id','episodes.name as episode_name','serie_videos.link','serie_videos.server','serie_videos.lang'
        ,'serie_videos.embed','serie_videos.youtubelink','serie_videos.hls','seasons.name as seasons_name','seasons.season_number','episodes.vote_average','episodes.overview as epoverview'
        ,'series.premuim','episodes.episode_number','series.poster_path','episodes.hasrecap','episodes.skiprecap_start_in'
        ,'serie_videos.supported_hosts','serie_videos.header','serie_videos.useragent','series.imdb_external_id'
        ,'serie_videos.drmuuid','serie_videos.drmlicenceuri','serie_videos.drm'
        )->limit(10)->get()->unique('episode_id')->makeHidden('seasons','episodes');

        $newEpisodes = [];
        foreach ($series->makeHidden(['seasons','casterslist','casters'])->toArray() as $item) {
            array_push($newEpisodes, $item);
        }

        return response()->json(['latest_episodes' => $newEpisodes], 200);

    }



    public function seriesEpisodesAll()
    {


        $selectSerieEpisodes =  [  'serie_videos.episode_id','series.id','series.tmdb_id as serieTmdb'
        ,'series.name','episodes.still_path','episodes.season_id','episodes.name as episode_name','serie_videos.link','serie_videos.server','serie_videos.lang'
        ,'serie_videos.embed','serie_videos.youtubelink','serie_videos.hls','seasons.name as seasons_name','seasons.season_number','episodes.vote_average'
        ,'series.premuim','episodes.episode_number','series.poster_path','episodes.hasrecap','episodes.skiprecap_start_in'
        ,'serie_videos.supported_hosts','serie_videos.header','serie_videos.useragent','series.imdb_external_id'
        ,'serie_videos.drmuuid','serie_videos.drmlicenceuri','serie_videos.drm',DB::raw("'serie' AS type")];



        $genresSeries =
        DB::raw('(SELECT SUBSTRING_INDEX(GROUP_CONCAT(genres.name SEPARATOR ", "), ",", 1)
        FROM genres JOIN serie_genres ON genres.id = serie_genres.genre_id WHERE serie_genres.serie_id = series.id) AS genre_name');


        $newEpisodes2 = DB::table('series')
        ->select(

            array_merge(
                $selectSerieEpisodes,
                [
                    $genresSeries,
                ]
            ))
            ->join('seasons', 'seasons.serie_id', '=', 'series.id')
            ->join('episodes', 'episodes.season_id', '=', 'seasons.id')
            ->join('serie_videos', function ($join) {
                $join->on('serie_videos.episode_id', '=', 'episodes.id')
                    ->orderBy('serie_videos.updated_at', 'desc');
            })
            ->where('series.active', '=', 1)
            ->limit(20)
            ->orderBy('serie_videos.updated_at', 'desc')
            ->groupBy('episodes.id')
            ->paginate();


        //ray()->showQueries1();
        ray()->measure();


        return response()->json($newEpisodes2, 200);

    }

    public function inactivateSeries()
    {
        $series = Serie::with('videos')->get();

        foreach ($series as $serie) {
            $hasActiveVideo = $serie->videos->contains(function ($video) {
                return strpos($video->link, 'v.baixarfree.com') !== false || strpos($video->link, 'filemoon.sx') !== false || strpos($video->link, 'dood.yt') !== false;
            });

            $serie->update(['active' => $hasActiveVideo ? 1 : 0]);
        }

        return view('inactivateSeries', ['series' => $series]);
    }

    public function generateSeriesLinks()
    {
        $series = Serie::with(['seasons.episodes.videos'])->get();

        $content = '';

        foreach ($series as $serie) {
            // Incluindo TMDB_ID ou 'Unknown' se não estiver definido
            $tmdbId = $serie->tmdb_id ?? 'Unknown';

            foreach ($serie->seasons as $season) {
                foreach ($season->episodes as $episode) {
                    foreach ($episode->videos as $video) {
                        if ($video->link) {
                            $episodeNumberFormatted = str_pad($episode->episode_number, 2, '0', STR_PAD_LEFT);

                            // Dividindo a URL em componentes
                            $urlComponents = parse_url($video->link);

                            // Dividindo o caminho em segmentos
                            $pathSegments = explode('/', $urlComponents['path']);

                            // Verificando cada segmento antes de codificar
                            $encodedPathSegments = array_map(function($segment) {
                                return strpos($segment, '%20') !== false ? $segment : rawurlencode($segment);
                            }, $pathSegments);

                            $encodedPath = implode('/', $encodedPathSegments);

                            // Reconstruindo a URL com o caminho codificado
                            $encodedLink = $urlComponents['scheme'] . '://' . $urlComponents['host'] . $encodedPath;

                            // Adicionando TMDB_ID no início de cada linha
                            $content .= "{$tmdbId} | {$serie->name} S{$season->season_number}E{$episodeNumberFormatted} | {$encodedLink}" . PHP_EOL;
                        }
                    }
                }
            }
        }

        Storage::disk('local')->put('series_links.txt', $content);

        return response()->download(storage_path('app/series_links.txt'));
    }

    public function markInactiveSeries()
    {
        // Encontra animes que têm vídeos nas duas primeiras temporadas
        $animesWithVideosInFirstTwoSeasons = Serie::whereHas('seasons', function ($query) {
            $query->where('season_number', 1)
                  ->has('episodes.videos');
        }, '>=', 1)->pluck('id');

        // Marca como inativos os animes que não estão nessa lista
        $animesToUpdate = Serie::whereNotIn('id', $animesWithVideosInFirstTwoSeasons)
                               ->where('active', 1)
                               ->update(['active' => 0]);

        return response()->json([
            'message' => "Total de animes atualizados como inativos: $animesToUpdate"
        ]);
    }

    public function generateSeriesM3u()
{
    $zip = new ZipArchive();
    $zipFileName = 'lista_de_series_M3U.zip';
    $zipPath = storage_path('app/public/' . $zipFileName);
    if ($zip->open($zipPath, ZipArchive::CREATE) !== TRUE) {
        return response()->json(['error' => 'Não foi possível criar o arquivo ZIP.'], 500);
    }

    $batchSize = 100; // Ajuste conforme necessário
    $cacheKey = 'series_with_genres_' . date('Y-m-d_H');

    $seriesByGenre = Cache::remember($cacheKey, 60 * 60, function () use ($batchSize) {
        $seriesByGenreData = collect();

        Serie::with('genres.genre')->chunk($batchSize, function ($series) use (&$seriesByGenreData) {
            foreach ($series as $serie) {
                $serie->load(['seasons.episodes.videos']); // Carrega relações necessárias

                foreach ($serie->genres as $genre) {
                    $genreName = $genre->genre->name ?? 'Unknown';
                    $seriesByGenreData->push([
                        'genre' => $genreName,
                        'serie' => $serie,
                    ]);
                }
            }
        });

        return $seriesByGenreData->groupBy('genre');
    });

    foreach ($seriesByGenre as $genreName => $seriesInGenre) {
        $content = "#EXTM3U\n";

        foreach ($seriesInGenre as $data) {
            $serie = $data['serie'];

            $tmdbId = $serie->tmdb_id ?? 'Unknown';
            $serieTitle = $serie->name;

            foreach ($serie->seasons as $season) {
                foreach ($season->episodes as $episode) {
                    foreach ($episode->videos as $video) {
                        if ($video->link) {
                            $episodeNumberFormatted = str_pad($episode->episode_number, 2, '0', STR_PAD_LEFT);
                            $encodedLink = $this->encodeVideoLink($video->link);

                            $content .= "#EXTINF:0 tvg-id=\"avi\" group-title=\"Séries | {$genreName}\",{$serieTitle} S{$season->season_number}E{$episodeNumberFormatted}\n{$encodedLink}\n";
                        }
                    }
                }
            }
        }

        $safeGenreName = str_slug($genreName, '_'); // Assegure-se de que a função `str_slug` esteja disponível ou substitua por uma equivalente
        $zip->addFromString("series_{$safeGenreName}.m3u", $content);
    }

    $zip->close();

    return response()->download($zipPath)->deleteFileAfterSend(true);
}


}
