<?php

namespace App\Http\Controllers;

use App\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use App\Subscription;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PlanController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $plans = Plan::orderBy('created_at', 'desc')->get();
        return view('admin.plans.index', compact('plans'));
    }

    /**
     * Retorna dados para o painel administrativo
     */
    public function data()
    {
        return response()->json(Plan::orderBy('priority', 'desc')->get());
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.plans.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'price' => 'required|numeric',
                'device_limit' => 'required|integer|min:1',
                'description' => 'nullable|string',
                'features' => 'nullable|array',
                'priority' => 'nullable|integer',
                'color' => 'nullable|string',
                'is_active' => 'boolean'
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            $data = $request->all();

            // Gerar slug se não for fornecido
            if (!isset($data['slug']) || empty($data['slug'])) {
                $data['slug'] = Str::slug($data['name']);
            }

            // Definir valores padrão
            $data['is_active'] = $request->has('is_active') ? true : false;
            $data['features'] = $request->features ?? [];

            $plan = Plan::create($data);

            Log::info('Plano criado com sucesso', ['plan_id' => $plan->id]);

            return response()->json([
                'status' => 200,
                'message' => 'Plano criado com sucesso',
                'plan' => $plan
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao criar plano: ' . $e->getMessage(), [
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Erro ao criar plano',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $plan = Plan::findOrFail($id);
        return view('admin.plans.show', compact('plan'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $plan = Plan::findOrFail($id);
        return view('admin.plans.edit', compact('plan'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $plan = Plan::findOrFail($id);

            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:255',
                'price' => 'required|numeric',
                'device_limit' => 'required|integer|min:1',
                'description' => 'nullable|string',
                'features' => 'nullable|array',
                'priority' => 'nullable|integer',
                'color' => 'nullable|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 422,
                    'message' => 'Validation error',
                    'errors' => $validator->errors()
                ], 422);
            }

            // Atualizar apenas campos permitidos
            $updateData = $request->only([
                'name', 'price', 'device_limit', 'description',
                'features', 'priority', 'color'
            ]);

            $updateData['is_active'] = $request->has('is_active');

            $plan->update($updateData);

            Log::info('Plano atualizado com sucesso', ['plan_id' => $plan->id]);

            // Limpar cache relacionado aos limites de dispositivos
            $this->clearDeviceLimitCache();

            return response()->json([
                'status' => 200,
                'message' => 'Plano atualizado com sucesso',
                'plan' => $plan
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao atualizar plano: ' . $e->getMessage(), [
                'plan_id' => $id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Erro ao atualizar plano',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $plan = Plan::findOrFail($id);

            // Verificar se há assinaturas associadas ao plano
            $subscriptionsCount = $plan->subscriptions()->count();

            if ($subscriptionsCount > 0) {
                return response()->json([
                    'status' => 422,
                    'message' => "Este plano possui {$subscriptionsCount} assinaturas ativas. Desative o plano em vez de excluí-lo."
                ], 422);
            }

            $plan->delete();

            Log::info('Plano excluído com sucesso', ['plan_id' => $id]);

            return response()->json([
                'status' => 200,
                'message' => 'Plano excluído com sucesso'
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao excluir plano: ' . $e->getMessage(), [
                'plan_id' => $id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Erro ao excluir plano',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Retorna todas as assinaturas para o painel administrativo
     */
    public function all()
    {
        $subscriptions = Subscription::with('user', 'plan')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return response()->json($subscriptions);
    }

    /**
     * Retorna estatísticas de planos para o dashboard
     */
    public function getStats()
    {
        try {
            $stats = [
                'total_plans' => Plan::count(),
                'active_plans' => Plan::where('is_active', true)->count(),
                'total_subscriptions' => Subscription::count(),
                'active_subscriptions' => Subscription::where('is_active', true)->count(),
                'plans_by_subscribers' => $this->getPlansBySubscribers(),
                'revenue_by_plan' => $this->getRevenueByPlan()
            ];

            return response()->json([
                'status' => 200,
                'stats' => $stats
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Erro ao obter estatísticas',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtém a contagem de assinantes por plano
     */
    private function getPlansBySubscribers()
    {
        return DB::table('plans')
            ->leftJoin('subscriptions', 'plans.id', '=', 'subscriptions.plan_id')
            ->select('plans.name', DB::raw('count(subscriptions.id) as subscriber_count'))
            ->groupBy('plans.id', 'plans.name')
            ->orderBy('subscriber_count', 'desc')
            ->get();
    }

    /**
     * Obtém a receita por plano
     */
    private function getRevenueByPlan()
    {
        return DB::table('plans')
            ->leftJoin('subscriptions', 'plans.id', '=', 'subscriptions.plan_id')
            ->select('plans.name', DB::raw('sum(plans.price) as revenue'))
            ->where('subscriptions.is_active', true)
            ->groupBy('plans.id', 'plans.name')
            ->orderBy('revenue', 'desc')
            ->get();
    }

    /**
     * Limpa o cache relacionado aos limites de dispositivos
     */
    private function clearDeviceLimitCache()
    {
        $users = \App\User::whereNotNull('device_limit')->get();

        foreach ($users as $user) {
            $cacheKey = "user_{$user->id}_device_limit";
            cache()->forget($cacheKey);
        }
    }
}
