<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Plan;

class PaymentController extends Controller
{
    protected $isPaggueEnabled;

    public function __construct()
    {
        // Usando env com a função filter_var para garantir o tipo booleano correto
        $this->isPaggueEnabled = filter_var(env('ENABLE_PAGGUE', 'false'), FILTER_VALIDATE_BOOLEAN);

        // Log detalhado da inicialização
        Log::info('PaymentController inicializado', [
            'ENABLE_PAGGUE' => env('ENABLE_PAGGUE'),
            'isPaggueEnabled' => $this->isPaggueEnabled,
            'tipo' => gettype($this->isPaggueEnabled)
        ]);
    }

    /**
     * Processa o pagamento e redireciona para o gateway apropriado
     */
    public function realizarPagamento(Request $request)
    {
        // Log de entrada da requisição
        Log::info('Iniciando processamento de pagamento', [
            'isPaggueEnabled' => $this->isPaggueEnabled,
            'requestData' => $request->all()
        ]);

        // Verificação básica de autenticação
        if (!Auth::check()) {
            return redirect()->route('login')->withErrors(['Faça login para continuar']);
        }

        // Validação básica dos campos
        $validated = $request->validate([
            'plano' => 'required',
            'name' => 'required|string',
            'email' => 'required|email',
            'numero' => 'required|string',
        ]);

        try {
            // Verificar qual gateway está configurado
            if ($this->isPaggueEnabled) {
                Log::info('Redirecionando para Paggue');
                $controller = app()->make(\App\Http\Controllers\PaggueController::class);
                return $controller->processPayment($request);
            } else {
                Log::info('Redirecionando para SuitPay');
                $controller = app()->make(\App\Http\Controllers\SuitPayController::class);
                return $controller->processPayment($request);
            }
        } catch (\Exception $e) {
            Log::error('Erro ao processar pagamento', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()->withErrors(['system' => 'Erro ao processar o pagamento: ' . $e->getMessage()]);
        }
    }

    /**
     * Mostra a página de sucesso do pagamento
     */
    public function showSuccess()
    {
        return view('pagamento_sucesso');
    }

    /**
     * Mostra a página de falha do pagamento
     */
    public function showFailure()
    {
        return view('pagamento_falha');
    }
}
