<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\User;
use App\Plan;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

class PaggueController extends Controller
{
    protected $baseUrl;
    protected $authUrl;
    protected $clientKey;
    protected $clientSecret;
    protected $companyId;
    protected $webhookToken;
    protected $accessToken;
    protected $tokenExpiration;

    public function __construct()
    {
        // Configurações da API Paggue
        $this->baseUrl = 'https://ms.paggue.io/cashin/api';
        $this->authUrl = 'https://ms.paggue.io/auth/v1/token';
        $this->clientKey = env('PAGGUE_CLIENT_KEY', '');
        $this->clientSecret = env('PAGGUE_CLIENT_SECRET', '');
        $this->companyId = env('PAGGUE_COMPANY_ID', '');
        $this->webhookToken = env('PAGGUE_WEBHOOK_TOKEN', '');

        // Tenta carregar token da sessão se existir
        $this->loadTokenFromSession();
    }

    /**
     * Carrega o token de acesso da sessão, se disponível
     */
    private function loadTokenFromSession()
    {
        if (session()->has('paggue_token')) {
            $tokenData = session()->get('paggue_token');
            $this->accessToken = $tokenData['access_token'];
            $this->tokenExpiration = $tokenData['expires_at'];

            // Verifica se o token expirou (com margem de segurança de 1 hora)
            $expirationDate = Carbon::parse($this->tokenExpiration);
            if ($expirationDate->subHour()->isPast()) {
                $this->accessToken = null;
                $this->tokenExpiration = null;
            }
        }
    }

    /**
     * Obtém o token de autenticação da API Paggue
     */
    private function getAuthToken()
    {
        // Se já temos um token válido, retorna-o
        if ($this->accessToken) {
            return $this->accessToken;
        }

        try {
            $client = new Client();
            $response = $client->post($this->authUrl, [
                'json' => [
                    'client_key' => $this->clientKey,
                    'client_secret' => $this->clientSecret
                ]
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['access_token'])) {
                $this->accessToken = $data['access_token'];
                $this->tokenExpiration = $data['expires_at'];

                // Salva na sessão para próximas requisições
                session()->put('paggue_token', [
                    'access_token' => $this->accessToken,
                    'expires_at' => $this->tokenExpiration
                ]);

                $this->logToFile('Token Paggue obtido com sucesso. Expira em: ' . $this->tokenExpiration);
                return $this->accessToken;
            } else {
                $this->logToFile('Erro ao obter token Paggue: resposta inválida');
                throw new \Exception('Resposta inválida da API de autenticação');
            }
        } catch (\Exception $e) {
            $this->logToFile('Erro ao obter token Paggue: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Gera o QR Code PIX para pagamento via Paggue
     */
    public function processPayment(Request $request)
    {
        // Log para debug
        $this->logToFile('Iniciando processamento de pagamento via Paggue');

        $validated = $request->validate([
            'plano' => 'required',
            'name' => 'required|string',
            'email' => 'required|email',
            'numero' => 'required|string',
        ]);

        try {
            // Buscar o plano selecionado
            $plano = Plan::findOrFail($request->plano);
            $userId = Auth::id();
            $user = Auth::user();

            // Verificar se é plano de TV
            $isTvPlan = stripos($plano->name, 'tv') !== false ||
                      stripos($plano->name, 'canais') !== false ||
                      (isset($plano->is_tv_plan) && $plano->is_tv_plan);

            // Gerar ID externo único
            $externalId = 'ultraflix_' . $userId . '_' . time() . '_' . Str::random(8);

            // Criar metadados para o webhook
            $metadata = [
                'user_id' => $userId,
                'email' => $request->email,
                'plan_id' => $plano->id,
                'plan_name' => $plano->name,
                'plan_duration' => (int)$plano->pack_duration,
                'is_tv_plan' => $isTvPlan,
                'webhook_url' => url('/paggue/callback')
            ];

            // Obter token de autenticação
            $token = $this->getAuthToken();

            // Preparar dados para a API
            $paymentData = [
                'payer_name' => $user->name,
                'amount' => (int)($plano->price * 100), // Converter para centavos
                'external_id' => $externalId,
                'description' => $isTvPlan ? 'Plano TV: ' . $plano->name : 'Plano: ' . $plano->name,
                'meta' => $metadata
            ];

            $this->logToFile('Criando cobrança Paggue: ' . json_encode($paymentData));

            // Fazer requisição para criar a cobrança PIX
            $client = new Client();
            $response = $client->post($this->baseUrl . '/billing_order', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'X-Company-ID' => $this->companyId,
                    'Content-Type' => 'application/json'
                ],
                'json' => $paymentData
            ]);

            $responseData = json_decode($response->getBody(), true);
            $this->logToFile('Resposta da API Paggue: ' . json_encode($responseData));

            if (!isset($responseData['payment'])) {
                $this->logToFile('Erro na resposta da API Paggue: QR Code não encontrado');
                return redirect()->back()->withErrors(['api' => 'Erro ao gerar o QR Code PIX']);
            }

            // Salvar dados do pagamento na sessão e em arquivo local
            $paymentInfo = [
                'hash' => $responseData['hash'],
                'reference' => $responseData['reference'] ?? $externalId,
                'external_id' => $externalId,
                'user_id' => $userId,
                'email' => $request->email,
                'plan_id' => $plano->id,
                'plan_duration' => (int)$plano->pack_duration,
                'plan_name' => $plano->name,
                'is_tv_plan' => $isTvPlan,
                'amount' => (float)$plano->price,
                'created_at' => now()->toDateTimeString(),
                'gateway' => 'paggue'
            ];

            session()->put('payment_info_' . $externalId, $paymentInfo);
            $this->savePaymentInfoToFile($externalId, $paymentInfo);

            // Gerar QR Code a partir do texto (string) de pagamento
            $qrcode = $this->generateQRCode($responseData['payment']);

            // Retorna view com QR Code
            return view('pagamento_pix', [
                'qrcode' => $qrcode,
                'qrcode_text' => $responseData['payment'],
                'plano' => $plano->name,
                'valor' => (float)$plano->price,
                'reference' => $externalId,
                'transaction_id' => $responseData['hash'],
                'gateway' => 'paggue'
            ]);

        } catch (\Exception $e) {
            // Log detalhado do erro
            $this->logToFile('Erro ao processar pagamento Paggue: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            Log::error('Paggue processamento falhou: ' . $e->getMessage());

            return redirect()->back()->withErrors(['system' => 'Erro ao processar o pagamento. Por favor, tente novamente.']);
        }
    }

    /**
     * Gera uma imagem QR Code a partir de uma string
     */
    private function generateQRCode($qrCodeText)
    {
        // Usando api.qrserver.com para gerar QR Code
        $url = 'https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=' . urlencode($qrCodeText);

        try {
            $imageData = file_get_contents($url);
            return base64_encode($imageData);
        } catch (\Exception $e) {
            $this->logToFile('Erro ao gerar QR Code: ' . $e->getMessage());

            // Alternativa: usar biblioteca QR Code local (requer instalação via composer)
            // Aqui estamos apenas retornando um erro, mas você poderia implementar uma alternativa
            throw new \Exception('Falha ao gerar QR Code');
        }
    }

    /**
     * Webhook para receber notificações Paggue
     */
    public function callback(Request $request)
    {
        try {
            // Log completo do webhook recebido
            $this->logWebhook($request);

            // Verificar a assinatura do webhook para garantir que é legítimo
            if (!$this->verifyWebhookSignature($request)) {
                $this->logToFile('Webhook com assinatura inválida');
                return response()->json(['status' => 'error', 'message' => 'Assinatura inválida'], 401);
            }

            // Verificar campos necessários
            if (!$request->has('external_id') || !$request->has('status')) {
                $this->logToFile('Webhook com dados incompletos');
                return response()->json(['status' => 'error', 'message' => 'Dados incompletos'], 400);
            }

            $externalId = $request->input('external_id');
            $status = $request->input('status');
            $hash = $request->input('hash');

            // Processar apenas se o pagamento foi confirmado (status 1 = pago)
            if ($status != 1) {
                $this->logToFile("Webhook com status diferente de pago: {$status}");
                return response()->json(['status' => 'success', 'message' => 'Notificação recebida']);
            }

            // Processar o pagamento
            $success = $this->processarPagamentoConfirmado($externalId, $hash);
            $this->logToFile('Resultado do processamento: ' . ($success ? 'Sucesso' : 'Falha'));

            // Responder com sucesso
            return response()->json(['status' => 'success', 'message' => 'Pagamento processado']);

        } catch (\Exception $e) {
            // Log do erro
            $this->logToFile('Erro no webhook Paggue: ' . $e->getMessage() . "\n" . $e->getTraceAsString());

            // Responder com erro, permitindo que a Paggue reenve o webhook
            return response()->json(['status' => 'error', 'message' => 'Erro interno'], 500);
        }
    }

    /**
     * Verifica a assinatura do webhook Paggue
     */
    private function verifyWebhookSignature(Request $request)
    {
        try {
            $signature = $request->header('Signature');
            if (!$signature) {
                $this->logToFile('Assinatura do webhook ausente');
                return false;
            }

            // Obter o corpo da requisição como string JSON
            $payload = $request->getContent();

            // Calcular a assinatura esperada usando o webhook_token
            // A documentação da Paggue diz que usa HMAC-SHA256, mas não especifica exatamente como
            // Vamos testar algumas abordagens comuns:

            // Método 1: HMAC SHA256
            $expectedSignature1 = hash_hmac('sha256', $payload, $this->webhookToken);

            // Método 2: SHA256 direto
            $expectedSignature2 = hash('sha256', $payload . $this->webhookToken);

            // Log para verificação
            $this->logToFile("Signature recebida: {$signature}");
            $this->logToFile("Signature calculada (método 1): {$expectedSignature1}");
            $this->logToFile("Signature calculada (método 2): {$expectedSignature2}");

            // Verificar se alguma das assinaturas calculadas corresponde
            if ($signature == $expectedSignature1 || $signature == $expectedSignature2) {
                return true;
            }

            // Em ambiente de desenvolvimento, podemos aceitar qualquer assinatura para testar
            if (app()->environment('local') && env('APP_DEBUG', false)) {
                $this->logToFile("Ambiente de desenvolvimento - aceitando webhook sem verificação de assinatura");
                return true;
            }

            return false;
        } catch (\Exception $e) {
            $this->logToFile('Erro ao verificar assinatura: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Processa o pagamento confirmado e ativa a conta do usuário
     */
    private function processarPagamentoConfirmado($externalId, $transactionHash)
    {
        $this->logToFile('Processando pagamento Paggue: ' . $externalId);

        try {
            // 1. Recuperar informações do pagamento
            $paymentInfo = session()->get('payment_info_' . $externalId);

            // Se não encontrado na sessão, tentar recuperar do arquivo
            if (!$paymentInfo) {
                $paymentInfo = $this->getPaymentInfoFromFile($externalId);
            }

            if (!$paymentInfo) {
                $this->logToFile('Informações de pagamento não encontradas para: ' . $externalId);
                return false;
            }

            // 2. Buscar o usuário
            $userId = $paymentInfo['user_id'] ?? null;
            $email = $paymentInfo['email'] ?? null;

            $user = null;
            if ($userId) {
                $user = User::find($userId);
            }

            if (!$user && $email) {
                $user = User::where('email', $email)->first();
            }

            if (!$user) {
                $this->logToFile('Usuário não encontrado para: ' . json_encode($paymentInfo));
                return false;
            }

            // 3. Ativar a assinatura adequada
            $isTvPlan = $paymentInfo['is_tv_plan'] ?? false;
            $planDuration = $paymentInfo['plan_duration'] ?? 30;
            $planName = $paymentInfo['plan_name'] ?? 'Plano Premium';
            $planId = $paymentInfo['plan_id'] ?? null;

            if ($isTvPlan) {
                $user->tv_access = 1;
                $user->expired_in_tv = Carbon::now()->addDays($planDuration);
                $this->logToFile('Ativando plano TV para usuário ' . $user->id . ' por ' . $planDuration . ' dias');
            } else {
                $user->premuim = 1;
                // Se já tem uma assinatura ativa, adiciona dias
                if ($user->expired_in && Carbon::parse($user->expired_in)->isFuture()) {
                    $user->expired_in = Carbon::parse($user->expired_in)->addDays($planDuration);
                } else {
                    $user->expired_in = Carbon::now()->addDays($planDuration);
                }
                $this->logToFile('Ativando plano Premium para usuário ' . $user->id . ' por ' . $planDuration . ' dias');
            }

            // 4. Atualizar informações adicionais
            $user->transaction_id = $transactionHash;
            $user->manual_premuim = 'PIX-Paggue';
            $user->pack_name = $planName;
            $user->pack_id = $planId;
            $user->start_at = Carbon::now();

            // 5. Salvar as alterações
            $result = $user->save();

            $this->logToFile('Usuário atualizado: ' . json_encode([
                'user_id' => $user->id,
                'email' => $user->email,
                'result' => $result,
                'premuim' => $user->premuim,
                'tv_access' => $user->tv_access,
                'expired_in' => $user->expired_in,
                'expired_in_tv' => $user->expired_in_tv,
            ]));

            // 6. Atualizar status na sessão
            if ($paymentInfo) {
                $paymentInfo['confirmed'] = true;
                $paymentInfo['confirmed_at'] = now()->toDateTimeString();
                session()->put('payment_info_' . $externalId, $paymentInfo);
            }

            return $result;

        } catch (\Exception $e) {
            $this->logToFile('Erro ao ativar assinatura: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * Salva informações do pagamento em arquivo
     */
    private function savePaymentInfoToFile($reference, $paymentInfo)
    {
        try {
            $dir = storage_path('pix_payments');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            $filePath = $dir . '/paggue_' . $reference . '.json';
            file_put_contents($filePath, json_encode($paymentInfo, JSON_PRETTY_PRINT));
        } catch (\Exception $e) {
            Log::error('Erro ao salvar info de pagamento Paggue: ' . $e->getMessage());
        }
    }

    /**
     * Recupera informações do pagamento do arquivo
     */
    private function getPaymentInfoFromFile($reference)
    {
        try {
            $filePath = storage_path('pix_payments/paggue_' . $reference . '.json');
            if (!file_exists($filePath)) {
                return null;
            }

            return json_decode(file_get_contents($filePath), true);
        } catch (\Exception $e) {
            Log::error('Erro ao ler info de pagamento Paggue: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Registra informações detalhadas do webhook em arquivo separado
     */
    private function logWebhook(Request $request)
    {
        try {
            // Diretório para webhooks
            $dir = storage_path('logs/paggue_webhooks');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            // Nome do arquivo baseado na data
            $date = now()->format('Y-m-d');
            $time = now()->format('H:i:s');
            $filePath = $dir . '/webhook_' . $date . '.log';

            // Coletar todas as informações possíveis
            $data = [
                'timestamp' => $time,
                'ip' => $request->ip(),
                'method' => $request->method(),
                'headers' => $request->headers->all(),
                'data' => $request->all(),
                'server' => $_SERVER,
            ];

            // Salvar webhook completo em formato JSON
            $webhookFilePath = $dir . '/webhook_' . date('Y-m-d_H-i-s') . '_' . uniqid() . '.json';
            file_put_contents($webhookFilePath, json_encode($data, JSON_PRETTY_PRINT));

            // Adicionar entrada ao log principal de webhooks
            $logEntry = "[{$time}] Webhook recebido de {$request->ip()}: " . json_encode($request->all()) . "\n";
            file_put_contents($filePath, $logEntry, FILE_APPEND);

            // Também registre no log do Laravel
            Log::channel('daily')->info('Paggue Webhook recebido', $data);

        } catch (\Exception $e) {
            Log::error('Erro ao registrar webhook Paggue: ' . $e->getMessage());
        }
    }

    /**
     * Escreve logs em arquivo específico para o Paggue
     */
    private function logToFile($message)
    {
        try {
            $dir = storage_path('logs/paggue');
            if (!file_exists($dir)) {
                mkdir($dir, 0755, true);
            }

            $date = now()->format('Y-m-d');
            $time = now()->format('H:i:s');
            $filePath = $dir . '/paggue_' . $date . '.log';

            $content = "[{$time}] {$message}\n";
            file_put_contents($filePath, $content, FILE_APPEND);

            // Também registra no log do Laravel
            Log::info('Paggue: ' . $message);

        } catch (\Exception $e) {
            Log::error('Erro ao escrever log Paggue: ' . $e->getMessage());
        }
    }

    /**
     * Verifica o status do pagamento
     */
    public function checkStatus(Request $request)
    {
        $externalId = $request->transaction_id;
        $this->logToFile('Verificação de status solicitada para: ' . $externalId);

        try {
            // 1. Verificar informações na sessão
            $paymentInfo = session()->get('payment_info_' . $externalId);
            if ($paymentInfo && isset($paymentInfo['confirmed']) && $paymentInfo['confirmed'] === true) {
                return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
            }

            // 2. Verificar se o usuário já está com plano ativo
            $userId = Auth::id();
            $user = Auth::user();

            if ($user) {
                $isTvPlan = $request->input('isTvPlan', false);
                if (($isTvPlan && $user->tv_access == 1 && $user->expired_in_tv && Carbon::parse($user->expired_in_tv)->isFuture()) ||
                    (!$isTvPlan && $user->premuim == 1 && $user->expired_in && Carbon::parse($user->expired_in)->isFuture())) {
                    return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
                }
            }

            // 3. Verificação forçada: verificar status da transação na API
            if ($request->input('force_check', false)) {
                $this->logToFile('Iniciando verificação forçada via API para: ' . $externalId);

                // Obter o hash do pagamento
                $transactionHash = null;
                if ($paymentInfo && isset($paymentInfo['hash'])) {
                    $transactionHash = $paymentInfo['hash'];
                    $this->logToFile('Hash da transação encontrado: ' . $transactionHash);
                }

                // Se encontrarmos o hash, consultar status
                if ($transactionHash) {
                    try {
                        // Obter token de autenticação
                        $token = $this->getAuthToken();

                        // Consultar status do pagamento na API
                        $client = new Client();
                        $response = $client->get($this->baseUrl . '/billing_order/' . $transactionHash, [
                            'headers' => [
                                'Authorization' => 'Bearer ' . $token,
                                'X-Company-ID' => $this->companyId,
                                'Content-Type' => 'application/json'
                            ]
                        ]);

                        $statusData = json_decode($response->getBody(), true);
                        $this->logToFile('Resposta da verificação na API: ' . json_encode($statusData));

                        // Se o status for 1 (pago), ativar manualmente
                        if (isset($statusData['status']) && $statusData['status'] == 1) {
                            $this->logToFile('Pagamento confirmado na API. Ativando assinatura manualmente.');
                            $this->processarPagamentoConfirmado($externalId, $transactionHash);
                            return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
                        } else {
                            $this->logToFile('Status na API ainda não confirmado: ' . ($statusData['status'] ?? 'N/A'));
                        }
                    } catch (\Exception $apiEx) {
                        $this->logToFile('Erro ao consultar API de status: ' . $apiEx->getMessage());
                    }
                }

                // 4. Se passou muito tempo desde o início do pagamento
                if (isset($paymentInfo['created_at'])) {
                    $createdAt = Carbon::parse($paymentInfo['created_at']);
                    $minutesPassed = $createdAt->diffInMinutes(now());

                    // Se passar de 5 minutos após verificação manual, ativamos por timeout
                    if ($minutesPassed > 5) {
                        $this->logToFile('Pagamento sendo ativado por timeout após verificação manual');
                        $this->processarPagamentoConfirmado($externalId, $transactionHash ?? 'manual_activation');
                        return response()->json(['status' => 'success', 'redirect' => route('pagamento.sucesso')]);
                    }
                }
            }

            // Se chegou aqui, ainda está pendente
            return response()->json(['status' => 'pending']);

        } catch (\Exception $e) {
            $this->logToFile('Erro ao verificar status: ' . $e->getMessage());
            return response()->json(['status' => 'pending']);
        }
    }

    /**
     * Mostra a página de sucesso do pagamento
     */
    public function showSuccess()
    {
        return view('pagamento_sucesso');
    }
}
