<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Movie;
use App\Serie;
use App\Anime;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use App\NotificationRead;

class NotificationsController extends Controller
{
    /**
     * Exibe a página de notificações com conteúdo recentemente adicionado ou atualizado
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        // Verificar se o usuário está autenticado
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        // Obter filmes recentes (adicionados ou atualizados)
        $recentMovies = Movie::where('active', 1)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($movie) {
                $movie->content_type = 'movie';
                $movie->is_new = $movie->created_at->diffInDays() <= 7;
                $movie->display_name = $movie->title;
                $movie->route = route('filme.detail', ['id' => $movie->id]);
                return $movie;
            });

        // Obter séries recentes (adicionadas ou atualizadas)
        $recentSeries = Serie::where('active', 1)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($serie) {
                $serie->content_type = 'serie';
                $serie->is_new = $serie->created_at->diffInDays() <= 7;
                $serie->display_name = $serie->name;
                $serie->route = route('serie.detail', ['id' => $serie->id]);
                return $serie;
            });

        // Obter animes recentes (adicionados ou atualizados)
        $recentAnimes = Anime::where('active', 1)
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($anime) {
                $anime->content_type = 'anime';
                $anime->is_new = $anime->created_at->diffInDays() <= 7;
                $anime->display_name = $anime->name;
                $anime->route = route('anime.detail', ['id' => $anime->id]);
                return $anime;
            });

        // Obter conteúdo atualizado recentemente (últimos 14 dias)
        $updatedMovies = Movie::where('active', 1)
            ->where('updated_at', '>', Carbon::now()->subDays(14))
            ->where('created_at', '<', Carbon::now()->subDays(7)) // Excluir novos para evitar duplicação
            ->orderBy('updated_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($movie) {
                $movie->content_type = 'movie';
                $movie->is_new = false;
                $movie->display_name = $movie->title;
                $movie->route = route('filme.detail', ['id' => $movie->id]);
                return $movie;
            });

        $updatedSeries = Serie::where('active', 1)
            ->where('updated_at', '>', Carbon::now()->subDays(14))
            ->where('created_at', '<', Carbon::now()->subDays(7))
            ->orderBy('updated_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($serie) {
                $serie->content_type = 'serie';
                $serie->is_new = false;
                $serie->display_name = $serie->name;
                $serie->route = route('serie.detail', ['id' => $serie->id]);
                return $serie;
            });

        $updatedAnimes = Anime::where('active', 1)
            ->where('updated_at', '>', Carbon::now()->subDays(14))
            ->where('created_at', '<', Carbon::now()->subDays(7))
            ->orderBy('updated_at', 'desc')
            ->take(10)
            ->get()
            ->map(function ($anime) {
                $anime->content_type = 'anime';
                $anime->is_new = false;
                $anime->display_name = $anime->name;
                $anime->route = route('anime.detail', ['id' => $anime->id]);
                return $anime;
            });

        // Marcar notificações como vistas
        $this->markNotificationsAsSeen();

        // Configuração para a view
        $settings = \App\Setting::first();

        return view('notifications', [
            'recentMovies' => $recentMovies,
            'recentSeries' => $recentSeries,
            'recentAnimes' => $recentAnimes,
            'updatedMovies' => $updatedMovies,
            'updatedSeries' => $updatedSeries,
            'updatedAnimes' => $updatedAnimes,
            'settings' => $settings
        ]);
    }

    /**
     * Retorna a contagem de notificações não visualizadas
     *
     * @return int
     */
    public function getUnreadCount()
    {
        if (!Auth::check()) {
            return 0;
        }

        $userId = Auth::id();

        // Obter a última vez que o usuário visualizou as notificações
        $notificationRead = NotificationRead::where('user_id', $userId)->first();
        $lastSeen = $notificationRead ? $notificationRead->last_read_at : Carbon::now()->subDays(7);

        // Conta novos filmes
        $newMoviesCount = Movie::where('active', 1)
            ->where('created_at', '>', $lastSeen)
            ->count();

        // Conta novas séries
        $newSeriesCount = Serie::where('active', 1)
            ->where('created_at', '>', $lastSeen)
            ->count();

        // Conta novos animes
        $newAnimesCount = Anime::where('active', 1)
            ->where('created_at', '>', $lastSeen)
            ->count();

        // Também verifica atualizações recentes
        $updatedMoviesCount = Movie::where('active', 1)
            ->where('updated_at', '>', $lastSeen)
            ->where('created_at', '<', $lastSeen) // Só conta atualizações, não criações
            ->count();

        $updatedSeriesCount = Serie::where('active', 1)
            ->where('updated_at', '>', $lastSeen)
            ->where('created_at', '<', $lastSeen)
            ->count();

        $updatedAnimesCount = Anime::where('active', 1)
            ->where('updated_at', '>', $lastSeen)
            ->where('created_at', '<', $lastSeen)
            ->count();

        return $newMoviesCount + $newSeriesCount + $newAnimesCount +
               $updatedMoviesCount + $updatedSeriesCount + $updatedAnimesCount;
    }

    /**
     * Marca todas as notificações atuais como visualizadas
     */
    private function markNotificationsAsSeen()
    {
        $userId = Auth::id();

        // Atualizar ou criar registro de visualização
        NotificationRead::updateOrCreate(
            ['user_id' => $userId],
            ['last_read_at' => Carbon::now()]
        );
    }

    /**
     * API para obter a contagem de notificações não lidas
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getUnreadCountApi()
    {
        return response()->json([
            'count' => $this->getUnreadCount()
        ]);
    }
}
