<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Episode;
use App\Network;
use App\MovieNetwork;
use App\Setting;
use App\Upcoming;
use Illuminate\Support\Facades\Auth;
use App\Genre;
use App\Cast;
use App\Http\Requests\MovieStoreRequest;
use App\Http\Requests\MovieUpdateRequest;
use App\Http\Requests\StoreImageRequest;
use App\Jobs\SendNotification;
use App\Movie;
use App\MovieGenre;
use App\MovieSubstitle;
use App\MovieVideo;
use App\MovieDownload;
use App\MovieCast;
use App\Serie;
use App\Anime;
use App\Livetv;
use App\Setting as SettingAlias;
use App\User;
use App\Featured;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\File;
use App\Http\ClearsResponseCache;
use ChristianKuri\LaravelFavorite\Traits\Favoriteable;
use BeyondCode\Comments\Comment;
use Illuminate\Validation\ValidationException;
use Spatie\Ray\Ray;
use Illuminate\Http\JsonResponse;
use App\ContinueWatching;
use App\Resume;

class InicioController extends Controller
{
    public function index()
    {
        // Campos necessários para episódios de anime - removidos campos desnecessários
        $selectAnimeEpisodes = [
            'anime_videos.anime_episode_id', 'animes.id',
            'animes.name', 'anime_episodes.still_path',
            'anime_episodes.anime_season_id', 'anime_episodes.name as episode_name',
            'anime_seasons.season_number', 'anime_episodes.episode_number'
        ];

        // Campos necessários para episódios de séries - removidos campos desnecessários
        $selectSerieEpisodes = [
            'serie_videos.episode_id', 'series.id',
            'series.name', 'episodes.still_path',
            'episodes.name as episode_name',
            'seasons.season_number', 'episodes.episode_number'
        ];

        // Simplificar os seletores - removendo campos não utilizados na interface
        $selectMovie = [
            'id', 'title AS name', 'poster_path', 'backdrop_path',
            'vote_average', 'release_date', 'pinned', 'views',
            DB::raw("'movie' AS type")
        ];

        $selectSerie = [
            'id', 'name', 'poster_path', 'backdrop_path',
            'vote_average', 'first_air_date AS release_date', 'pinned',
            'views', DB::raw("'serie' AS type")
        ];

        $selectAnime = [
            'id', 'name', 'poster_path', 'backdrop_path',
            'vote_average', 'first_air_date AS release_date', 'pinned',
            'views', DB::raw("'anime' AS type")
        ];

        $settings = Setting::query()->first();
        $hoje = Carbon::now()->format('Y-m-d');

        // Géneros para novelas e marvel (apenas IDs necessários)
        $genre = 10776; // Novelas
        $genremarvel = 10773; // Marvel

        // Consulta para novelas - corrigida para evitar ambiguidade em created_at
        if ($settings->anime) {
            $novelas = DB::table(function ($query) use ($genre) {
                $query->select('movies.id', 'title AS name', 'poster_path', 'backdrop_path',
                    'vote_average', 'release_date', 'pinned', 'views',
                    DB::raw("'movie' AS type"), 'movies.created_at as created_at')
                    ->join('movie_genres', 'movies.id', '=', 'movie_genres.movie_id')
                    ->where('movie_genres.genre_id', '=', $genre)
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('movies.created_at', 'desc');

                $query->unionAll(function ($query) use ($genre) {
                    $query->select('series.id', 'name', 'poster_path', 'backdrop_path',
                        'vote_average', 'first_air_date AS release_date', 'pinned',
                        'views', DB::raw("'serie' AS type"), 'series.created_at as created_at')
                        ->join('serie_genres', 'series.id', '=', 'serie_genres.serie_id')
                        ->where('serie_genres.genre_id', '=', $genre)
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('series.created_at', 'desc');
                });

                $query->unionAll(function ($query) use ($genre) {
                    $query->select('animes.id', 'name', 'poster_path', 'backdrop_path',
                        'vote_average', 'first_air_date AS release_date', 'pinned',
                        'views', DB::raw("'anime' AS type"), 'animes.created_at as created_at')
                        ->join('anime_genres', 'animes.id', '=', 'anime_genres.anime_id')
                        ->where('anime_genres.genre_id', '=', $genre)
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->orderBy('animes.created_at', 'desc')
                        ->limit(10);
                });
            })
            ->orderByDesc('created_at')
            ->get();
        } else {
            $novelas = DB::table(function ($query) use ($genre) {
                $query->select('movies.id', 'title AS name', 'poster_path', 'backdrop_path',
                    'vote_average', 'release_date', 'pinned', 'views',
                    DB::raw("'movie' AS type"), 'movies.created_at as created_at')
                    ->join('movie_genres', 'movies.id', '=', 'movie_genres.movie_id')
                    ->where('movie_genres.genre_id', '=', $genre)
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('movies.created_at', 'desc');

                $query->unionAll(function ($query) use ($genre) {
                    $query->select('series.id', 'name', 'poster_path', 'backdrop_path',
                        'vote_average', 'first_air_date AS release_date', 'pinned',
                        'views', DB::raw("'serie' AS type"), 'series.created_at as created_at')
                        ->join('serie_genres', 'series.id', '=', 'serie_genres.serie_id')
                        ->where('serie_genres.genre_id', '=', $genre)
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('series.created_at', 'desc');
                });
            })
            ->orderByDesc('created_at')
            ->get();
        }

        // Consulta para filmes Marvel - otimizada
        if ($settings->anime) {
            $marvel = DB::table(function ($query) use ($genremarvel) {
                $query->select('movies.id', 'title AS name', 'poster_path', 'backdrop_path',
                    'vote_average', 'release_date', 'pinned', 'views',
                    DB::raw("'movie' AS type"), 'movies.created_at as created_at')
                    ->join('movie_genres', 'movies.id', '=', 'movie_genres.movie_id')
                    ->where('movie_genres.genre_id', '=', $genremarvel)
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->inRandomOrder()
                    ->limit(10);

                $query->unionAll(function ($query) use ($genremarvel) {
                    $query->select('series.id', 'name', 'poster_path', 'backdrop_path',
                        'vote_average', 'first_air_date AS release_date', 'pinned',
                        'views', DB::raw("'serie' AS type"), 'series.created_at as created_at')
                        ->join('serie_genres', 'series.id', '=', 'serie_genres.serie_id')
                        ->where('serie_genres.genre_id', '=', $genremarvel)
                        ->from('series')
                        ->where('active', '=', 1)
                        ->inRandomOrder()
                        ->limit(10);
                });

                $query->unionAll(function ($query) use ($genremarvel) {
                    $query->select('animes.id', 'name', 'poster_path', 'backdrop_path',
                        'vote_average', 'first_air_date AS release_date', 'pinned',
                        'views', DB::raw("'anime' AS type"), 'animes.created_at as created_at')
                        ->join('anime_genres', 'animes.id', '=', 'anime_genres.anime_id')
                        ->where('anime_genres.genre_id', '=', $genremarvel)
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->inRandomOrder()
                        ->limit(10);
                });
            })
            ->orderByDesc('created_at')
            ->get();
        } else {
            $marvel = DB::table(function ($query) use ($genremarvel) {
                $query->select('movies.id', 'title AS name', 'poster_path', 'backdrop_path',
                    'vote_average', 'release_date', 'pinned', 'views',
                    DB::raw("'movie' AS type"), 'movies.created_at as created_at')
                    ->join('movie_genres', 'movies.id', '=', 'movie_genres.movie_id')
                    ->where('movie_genres.genre_id', '=', $genremarvel)
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->inRandomOrder()
                    ->limit(10);

                $query->unionAll(function ($query) use ($genremarvel) {
                    $query->select('series.id', 'name', 'poster_path', 'backdrop_path',
                        'vote_average', 'first_air_date AS release_date', 'pinned',
                        'views', DB::raw("'serie' AS type"), 'series.created_at as created_at')
                        ->join('serie_genres', 'series.id', '=', 'serie_genres.serie_id')
                        ->where('serie_genres.genre_id', '=', $genremarvel)
                        ->from('series')
                        ->where('active', '=', 1)
                        ->inRandomOrder()
                        ->limit(10);
                });
            })
            ->inRandomOrder()
            ->get();
        }

        // Consulta de filmes e séries mais recentes - removida consulta de gêneros
        if ($settings->anime) {
            $latest = DB::table(function ($query) use ($selectMovie, $selectSerie, $selectAnime) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('movies.created_at', 'desc')
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('series.created_at', 'desc')
                        ->limit(10);
                });

                $query->unionAll(function ($query) use ($selectAnime) {
                    $query->select(array_merge(
                        $selectAnime,
                        [DB::raw('animes.created_at as created_at')]
                    ))
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->orderBy('animes.created_at', 'desc')
                        ->limit(10);
                });
            })
            ->orderByDesc('created_at')
            ->get();
        } else {
            $latest = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('movies.created_at', 'desc')
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('series.created_at', 'desc')
                        ->limit(10);
                });
            })
            ->orderByDesc('created_at')
            ->get();
        }

        // Consulta de conteúdo da semana atual - otimizada
        if ($settings->anime) {
            $arraythisweek = DB::table(function ($query) use ($selectMovie, $selectSerie, $selectAnime) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('movies.created_at', 'desc')
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('series.created_at', 'desc')
                        ->limit(10);
                });

                $query->unionAll(function ($query) use ($selectAnime) {
                    $query->select(array_merge(
                        $selectAnime,
                        [DB::raw('animes.created_at as created_at')]
                    ))
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->orderBy('animes.created_at', 'desc')
                        ->limit(10);
                });
            })
            ->orderByDesc('created_at')
            ->where('created_at', '>', Carbon::now()->startOfWeek())
            ->get();
        } else {
            $arraythisweek = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('movies.created_at', 'desc')
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('series.created_at', 'desc')
                        ->limit(10);
                });
            })
            ->orderByDesc('created_at')
            ->where('created_at', '>', Carbon::now()->startOfWeek())
            ->get();
        }

        // Similares otimizações para outras consultas (arraychoosed, arraytrending, arrayrecommended, etc.)
        if ($settings->anime) {
            $arraychoosed = DB::table(function ($query) use ($selectMovie, $selectSerie, $selectAnime) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->limit(10);
                });

                $query->unionAll(function ($query) use ($selectAnime) {
                    $query->select(array_merge(
                        $selectAnime,
                        [DB::raw('animes.created_at as created_at')]
                    ))
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->limit(10);
                });
            })
            ->inRandomOrder()
            ->get();
        } else {
            $arraychoosed = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->limit(10);
                });
            })
            ->inRandomOrder()
            ->get();
        }

        if ($settings->anime) {
            $arraytrending = DB::table(function ($query) use ($selectMovie, $selectSerie, $selectAnime) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->limit(10);
                });

                $query->unionAll(function ($query) use ($selectAnime) {
                    $query->select(array_merge(
                        $selectAnime,
                        [DB::raw('animes.created_at as created_at')]
                    ))
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->limit(10);
                });
            })
            ->where('views', '>', 10)
            ->orderByDesc('views')
            ->get();
        } else {
            $arraytrending = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->limit(10);
                });
            })
            ->where('views', '>', 10)
            ->orderByDesc('views')
            ->get();
        }

        if ($settings->anime) {
            $arrayrecommended = DB::table(function ($query) use ($selectMovie, $selectSerie, $selectAnime) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->limit(20);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->limit(20);
                });

                $query->unionAll(function ($query) use ($selectAnime) {
                    $query->select(array_merge(
                        $selectAnime,
                        [DB::raw('animes.created_at as created_at')]
                    ))
                        ->from('animes')
                        ->where('active', '=', 1)
                        ->limit(10);
                });
            })
            ->where('vote_average', '>', 6.0)
            ->inRandomOrder()
            ->get();
        } else {
            $arrayrecommended = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->limit(10);
                });
            })
            ->where('vote_average', '>', 7.0)
            ->inRandomOrder()
            ->get();
        }

        if ($settings->anime) {
            $arraypinned = DB::table(function ($query) use ($selectMovie, $selectSerie, $selectAnime) {
                $query->select(array_merge(
                    $selectMovie,
                    [
                        DB::raw('movies.created_at as created_at'),
                        DB::raw('movies.updated_at as updated_at') // Adicionando updated_at
                    ]
                ))
                      ->from('movies')
                      ->where('active', '=', 1);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [
                            DB::raw('series.created_at as created_at'),
                            DB::raw('series.updated_at as updated_at') // Adicionando updated_at
                        ]
                    ))
                          ->from('series')
                          ->where('active', '=', 1);
                });

                $query->unionAll(function ($query) use ($selectAnime) {
                    $query->select(array_merge(
                        $selectAnime,
                        [
                            DB::raw('animes.created_at as created_at'),
                            DB::raw('animes.updated_at as updated_at') // Adicionando updated_at
                        ]
                    ))
                          ->from('animes')
                          ->where('active', '=', 1);
                });
            })
            ->where('pinned', 1)
            ->orderByDesc('updated_at')
            ->get();
        } else {
            $arraypinned = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [
                        DB::raw('movies.created_at as created_at'),
                        DB::raw('movies.updated_at as updated_at') // Adicionando updated_at
                    ]
                ))
                      ->from('movies')
                      ->where('active', '=', 1);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [
                            DB::raw('series.created_at as created_at'),
                            DB::raw('series.updated_at as updated_at') // Adicionando updated_at
                        ]
                    ))
                          ->from('series')
                          ->where('active', '=', 1);
                });
            })
            ->where('pinned', 1)
            ->orderByDesc('updated_at')
            ->get();
        }

        if ($settings->anime) {
            $arraytop10series = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('views', 'desc')
                    ->limit(0);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('views', 'desc')
                        ->limit(10);
                });
            })
            ->orderBy('views', 'desc')
            ->get();
        } else {
            $arraytop10series = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('views', 'desc')
                    ->limit(0);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('views', 'desc')
                        ->limit(10);
                });
            })
            ->orderBy('views', 'desc')
            ->get();
        }

        if ($settings->anime) {
            $arraytop10 = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('views', 'desc')
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('views', 'desc')
                        ->limit(0);
                });
            })
            ->orderBy('views', 'desc')
            ->get();
        } else {
            $arraytop10 = DB::table(function ($query) use ($selectMovie, $selectSerie) {
                $query->select(array_merge(
                    $selectMovie,
                    [DB::raw('movies.created_at as created_at')]
                ))
                    ->from('movies')
                    ->where('active', '=', 1)
                    ->orderBy('views', 'desc')
                    ->limit(10);

                $query->unionAll(function ($query) use ($selectSerie) {
                    $query->select(array_merge(
                        $selectSerie,
                        [DB::raw('series.created_at as created_at')]
                    ))
                        ->from('series')
                        ->where('active', '=', 1)
                        ->orderBy('views', 'desc')
                        ->limit(0);
                });
            })
            ->orderBy('views', 'desc')
            ->get();
        }

        if ($settings->anime) {
            $animeslatest = DB::table(function ($query) use ($selectAnime) {
                $query->select(array_merge(
                    $selectAnime,
                    [DB::raw('animes.created_at as created_at')]
                ))
                    ->from('animes')
                    ->where('active', '=', 1)
                    ->orderBy('created_at', 'desc')
                    ->limit(10);
            })
            ->orderBy('created_at', 'desc')
            ->get();
        }

        $popularSeries = DB::table(function ($query) use ($selectSerie) {
            $query->select(array_merge(
                $selectSerie,
                [DB::raw('series.created_at as created_at')]
            ))
                ->from('series')
                ->where('active', '=', 1)
                ->orderBy('created_at', 'desc')
                ->limit(10);
        })
        ->where('views', '>', 500)
        ->orderByDesc('views')
        ->get();

        $recentSeries = DB::table(function ($query) use ($selectSerie) {
            $query->select(array_merge(
                $selectSerie,
                [DB::raw('series.created_at as created_at')]
            ))
                ->from('series')
                ->where('active', '=', 1)
                ->orderBy('created_at', 'desc')
                ->limit(10);
        })
        ->orderBy('created_at', 'desc')
        ->get();

        $featured = Featured::query()
            ->orderBy('position')
            ->orderByDesc('updated_at')
            ->limit($settings->featured_home_numbers)
            ->get();

        $newEpisodes2 = DB::table('series')
            ->select(array_merge($selectSerieEpisodes))
            ->join('seasons', 'seasons.serie_id', '=', 'series.id')
            ->join('episodes', 'episodes.season_id', '=', 'seasons.id')
            ->join('serie_videos', function ($join) {
                $join->on('serie_videos.episode_id', '=', 'episodes.id')
                    ->orderBy('serie_videos.updated_at', 'desc');
            })
            ->select('series.id as serie_id', 'series.backdrop_path as serie_backdrop',
                'series.name as serie_name', 'seasons.season_number',
                'episodes.episode_number', 'episodes.name as episode_name',
                'episodes.still_path', 'episodes.id as episode_id')
            ->where('series.active', '=', 1)
            ->limit(10)
            ->orderBy('serie_videos.updated_at', 'desc')
            ->groupBy('episodes.id')
            ->get();

        if ($settings->anime) {
            $latestEpisodesAnimes = DB::table('animes')
                ->select(array_merge($selectAnimeEpisodes))
                ->join('anime_seasons', 'anime_seasons.anime_id', '=', 'animes.id')
                ->join('anime_episodes', 'anime_episodes.anime_season_id', '=', 'anime_seasons.id')
                ->join('anime_videos', function ($join) {
                    $join->on('anime_videos.anime_episode_id', '=', 'anime_episodes.id')
                        ->orderBy('anime_videos.updated_at', 'desc');
                })
                ->where('animes.active', '=', 1)
                ->limit(10)
                ->orderBy('anime_videos.updated_at', 'desc')
                ->groupBy('anime_episode_id')
                ->get();
        }

        // Buscar continue watching apenas para o perfil atual
        $profile = session('selected_profile');
        $continueWatching = collect(); // Coleção vazia por padrão

        if ($profile) {
            // Consulta otimizada para buscar todos os dados necessários de uma vez
            $continueWatching = ContinueWatching::where('profile_id', $profile->id)
                              ->orderByDesc('updated_at')
                              ->get();

            // Enriquecer os dados com informações de cada tipo de mídia (movie, serie, anime)
            foreach ($continueWatching as $item) {
                // Adicionar campos necessários conforme o tipo de mídia
                switch ($item->type) {
                    case 'movie':
                        $movie = DB::table('movies')->where('id', $item->item_id)->first();
                        if ($movie) {
                            $item->title = $movie->title;
                            $item->poster_path = $movie->poster_path;
                            $item->backdrop_path = $movie->backdrop_path;
                            $item->release_date = $movie->release_date;
                        }
                        break;
                    case 'serie':
                        if ($item->episode_id) {
                            $episode = DB::table('episodes')
                                        ->join('seasons', 'episodes.season_id', '=', 'seasons.id')
                                        ->join('series', 'seasons.serie_id', '=', 'series.id')
                                        ->where('episodes.id', $item->episode_id)
                                        ->select('episodes.*', 'seasons.season_number', 'series.id as serie_id',
                                                'series.name as serie_name', 'episodes.name as episode_name',
                                                'episodes.still_path')
                                        ->first();

                            if ($episode) {
                                $item->episode_name = $episode->episode_name;
                                $item->still_path = $episode->still_path;
                                $item->serie_id = $episode->serie_id;
                                $item->serie_name = $episode->serie_name;

                                // Adicione item_id como backup se serie_id não estiver disponível
                                if (empty($item->serie_id)) {
                                    $item->serie_id = $item->item_id;
                                }
                            }
                        }
                        break;
                    case 'anime':
                        if ($item->episode_id) {
                            $animeEpisode = DB::table('anime_episodes')
                                            ->join('anime_seasons', 'anime_episodes.anime_season_id', '=', 'anime_seasons.id')
                                            ->join('animes', 'anime_seasons.anime_id', '=', 'animes.id')
                                            ->where('anime_episodes.id', $item->episode_id)
                                            ->select('anime_episodes.*', 'anime_seasons.season_number', 'animes.id as anime_id',
                                                    'animes.name as anime_name', 'anime_episodes.name as episode_name',
                                                    'anime_episodes.still_path')
                                            ->first();

                            if ($animeEpisode) {
                                $item->episode_name = $animeEpisode->episode_name;
                                $item->still_path = $animeEpisode->still_path;
                                $item->anime_id = $animeEpisode->anime_id;
                                $item->anime_name = $animeEpisode->anime_name;

                                // Garantir que item_id seja usado como fallback se anime_id não estiver disponível
                                if (empty($item->anime_id)) {
                                    $item->anime_id = $item->item_id;
                                }
                            }
                        } else {
                            // Se não tivermos episode_id, atribui o item_id como anime_id
                            $item->anime_id = $item->item_id;

                            // Busca informações básicas do anime
                            $anime = DB::table('animes')->where('id', $item->item_id)->first();
                            if ($anime) {
                                $item->poster_path = $anime->poster_path;
                                $item->anime_name = $anime->name;
                            }
                        }
                        break;
                }
            }
        }

        // Recuperar Featured de forma otimizada
        $featured = Featured::query()
                   ->orderBy('position')
                   ->orderByDesc('updated_at')
                   ->limit($settings->featured_home_numbers)
                   ->get();

        // Retornar a view com todos os dados necessários
        return view('layouts.pages.inicio', [
            'novelas' => $novelas,
            'marvel' => $marvel,
            'latest' => $latest,
            'latest_episodes' => $newEpisodes2,
            'latest_episodes_animes' => isset($latestEpisodesAnimes) ? $latestEpisodesAnimes : null,
            'thisweek' => $arraythisweek,
            'choosed' => $arraychoosed,
            'recommended' => $arrayrecommended,
            'trending' => $arraytrending,
            'pinned' => $arraypinned,
            'top10' => $arraytop10,
            'continueWatching' => $continueWatching,
            'top10series' => $arraytop10series,
            'featured' => $featured,
            'popularSeries' => $popularSeries,
            'recents' => $recentSeries,
            'settings' => $settings,
            'hoje' => $hoje
        ]);
    }

    public function getRandomMovie(): JsonResponse
    {
        // Seleciona um filme aleatório que tenha vídeos associados
        $randomMovie = Movie::with('genres')->has('videos')->inRandomOrder()->first();

        if (!$randomMovie) {
            return response()->json(['message' => 'Nenhum filme encontrado'], 404);
        }

        // Seleciona um vídeo aleatório deste filme
        $randomVideo = $randomMovie->videos()->inRandomOrder()->first();

        if (!$randomVideo) {
            return response()->json(['message' => 'Nenhum vídeo encontrado para o filme selecionado'], 404);
        }

        // Calculando a duração formatada
        $runtimeHours = floor($randomMovie->runtime / 60);
        $runtimeMinutes = $randomMovie->runtime % 60;
        $formattedDuration = $runtimeHours > 0
            ? $runtimeHours . 'h ' . $runtimeMinutes . 'min'
            : $runtimeMinutes . 'min';

        return response()->json([
            'movieUrl' => $randomVideo->link,
            'movieImage' => $randomMovie->backdrop_path,
            'movieBackdrop' => $randomMovie->backdrop_path_tv ?: $randomMovie->backdrop_path,
            'moviePoster' => $randomMovie->poster_path,
            'movieTitle' => $randomMovie->title,
            'movieOverview' => $randomMovie->overview,
            'movieRelease' => $randomMovie->release_date,
            'movieRating' => $randomMovie->vote_average,
            'movieDuration' => $formattedDuration,
            'movieGenres' => $randomMovie->genres->pluck('name')->toArray(),
            'movieTmdbId' => $randomMovie->tmdb_id
        ]);
    }

    public function getRandomSeries(): JsonResponse
    {
        // Seleciona uma série ativa aleatória que tenha temporadas e episódios
        $randomSeries = Serie::with('genres')->where('active', 1)->has('seasons.episodes')->inRandomOrder()->first();

        if (!$randomSeries) {
            return response()->json(['message' => 'Nenhuma série ativa encontrada'], 404);
        }

        // Obtém a primeira temporada
        $firstSeason = $randomSeries->seasons()->orderBy('season_number', 'asc')->first();

        if (!$firstSeason) {
            return response()->json(['message' => 'Nenhuma temporada encontrada para a série selecionada'], 404);
        }

        // Obtém o primeiro episódio da primeira temporada
        $firstEpisode = $firstSeason->episodes()->orderBy('episode_number', 'asc')->first();

        if (!$firstEpisode) {
            return response()->json(['message' => 'Nenhum episódio encontrado para a série selecionada'], 404);
        }

        // Obtém o link do vídeo do primeiro episódio
        $firstEpisodeVideoLink = $firstEpisode->getFirstVideoLink();

        return response()->json([
            'episodeUrl' => $firstEpisodeVideoLink,
            'episodeImage' => $firstEpisode->still_path,
            'seriesBackdrop' => $randomSeries->backdrop_path_tv ?: $randomSeries->backdrop_path,
            'seriesPoster' => $randomSeries->poster_path,
            'seriesTitle' => $randomSeries->name,
            'seriesOverview' => $randomSeries->overview,
            'seasonNumber' => $firstSeason->season_number,
            'episodeNumber' => $firstEpisode->episode_number,
            'seriesId' => $randomSeries->id,
            'seriesGenres' => $randomSeries->genres->pluck('name')->toArray(),
            'seriesRating' => $randomSeries->vote_average,
            'seriesFirstAirDate' => $randomSeries->first_air_date,
            'episodeOverview' => $firstEpisode->overview
        ]);
    }

    public function getRandomAnimes(): JsonResponse
    {
        // Seleciona uma série ativa aleatória que tenha temporadas e episódios
        $randomSeries = Anime::with('genres')->where('active', 1)->has('seasons.episodes')->inRandomOrder()->first();

        if (!$randomSeries) {
            return response()->json(['message' => 'Nenhuma série ativa encontrada'], 404);
        }

        // Obtém a primeira temporada
        $firstSeason = $randomSeries->seasons()->orderBy('season_number', 'asc')->first();

        if (!$firstSeason) {
            return response()->json(['message' => 'Nenhuma temporada encontrada para a série selecionada'], 404);
        }

        // Obtém o primeiro episódio da primeira temporada
        $firstEpisode = $firstSeason->episodes()->orderBy('episode_number', 'asc')->first();

        if (!$firstEpisode) {
            return response()->json(['message' => 'Nenhum episódio encontrado para a série selecionada'], 404);
        }

        // Obtém o link do vídeo do primeiro episódio
        $firstEpisodeVideoLink = $firstEpisode->getFirstVideoLink();

        return response()->json([
            'episodeUrl' => $firstEpisodeVideoLink,
            'episodeImage' => $firstEpisode->still_path,
            'seriesBackdrop' => $randomSeries->backdrop_path_tv ?: $randomSeries->backdrop_path,
            'seriesPoster' => $randomSeries->poster_path,
            'seriesTitle' => $randomSeries->name,
            'seriesOverview' => $randomSeries->overview,
            'seasonNumber' => $firstSeason->season_number,
            'episodeNumber' => $firstEpisode->episode_number,
            'seriesId' => $randomSeries->id,
            'seriesGenres' => $randomSeries->genres->pluck('name')->toArray(),
            'seriesRating' => $randomSeries->vote_average,
            'seriesFirstAirDate' => $randomSeries->first_air_date,
            'episodeOverview' => $firstEpisode->overview
        ]);
    }
}
