<?php

namespace App\Http\Controllers;

use App\Device;
use App\Setting;
use App\Services\DeviceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Jenssegers\Agent\Agent;

class DeviceController extends Controller
{
    protected $deviceService;

    public function __construct(DeviceService $deviceService)
    {
        $this->deviceService = $deviceService;
    }

    /**
     * Mostrar lista de dispositivos do usuário
     */
    public function index()
    {
        try {
            $user = Auth::user();
            $devices = $user->devices()
                ->orderBy('is_current', 'desc')
                ->orderBy('last_active', 'desc')
                ->get();

            // Garantir que last_active seja tratado como objeto Carbon
            $devices = $devices->map(function ($device) {
                if ($device->last_active && !$device->last_active instanceof \Carbon\Carbon) {
                    $device->last_active = \Carbon\Carbon::parse($device->last_active);
                }
                return $device;
            });

            // Determinar o limite de dispositivos do usuário
            $deviceLimit = $this->deviceService->getUserDeviceLimit($user);

            $settings = Setting::first();

            return view('layouts.pages.devices', [
                'devices' => $devices,
                'deviceLimit' => $deviceLimit,
                'settings' => $settings
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao exibir lista de dispositivos: ' . $e->getMessage());
            return redirect()->route('inicio')
                ->with('error', 'Ocorreu um erro ao carregar seus dispositivos. Por favor, tente novamente.');
        }
    }

    /**
     * Remover um dispositivo específico
     */
    public function destroy($id)
    {
        try {
            $device = Device::findOrFail($id);
            $user = Auth::user();

            // Verificar se o usuário tem permissão para remover este dispositivo
            if ($device->user_id !== $user->id) {
                return redirect()->route('devices.index')
                    ->with('error', 'Você não tem permissão para remover este dispositivo.');
            }

            // Impedir a remoção do dispositivo atual
            if ($device->is_current) {
                return redirect()->route('devices.index')
                    ->with('error', 'Não é possível remover o dispositivo que você está usando atualmente.');
            }

            $deviceName = $device->device_name;
            $device->delete();

            Log::info('Dispositivo removido', [
                'user_id' => $user->id,
                'device_id' => $id,
                'device_name' => $deviceName
            ]);

            return redirect()->route('devices.index')
                ->with('success', 'Dispositivo removido com sucesso.');
        } catch (\Exception $e) {
            Log::error('Erro ao remover dispositivo', [
                'error' => $e->getMessage(),
                'user_id' => Auth::id(),
                'device_id' => $id
            ]);

            return redirect()->route('devices.index')
                ->with('error', 'Erro ao remover dispositivo.');
        }
    }

    /**
     * Remove todos os dispositivos exceto o atual
     */
    public function destroyAll()
    {
        try {
            $user = Auth::user();
            $deletedCount = 0;

            // Excluir todos os dispositivos exceto o atual
            $deletedCount = $user->devices()
                ->where('is_current', false)
                ->delete();

            Log::info('Todos os dispositivos removidos', [
                'user_id' => $user->id,
                'deleted_count' => $deletedCount
            ]);

            return redirect()->route('devices.index')
                ->with('success', 'Todos os outros dispositivos foram removidos com sucesso.');
        } catch (\Exception $e) {
            Log::error('Erro ao remover todos os dispositivos', [
                'error' => $e->getMessage(),
                'user_id' => Auth::id()
            ]);

            return redirect()->route('devices.index')
                ->with('error', 'Erro ao remover os dispositivos.');
        }
    }

    /**
     * Registra um novo dispositivo para o usuário ou atualiza um existente
     *
     * @param Request $request
     * @return Device|null
     */
    public static function registerDevice(Request $request)
    {
        try {
            $user = Auth::user();
            if (!$user) {
                Log::warning('Tentativa de registrar dispositivo sem usuário autenticado');
                return null;
            }

            // Detectar informações do dispositivo
            $agent = new Agent();
            $agent->setUserAgent($request->userAgent());

            if ($agent->isPhone()) {
                $deviceType = 'mobile';
            } elseif ($agent->isTablet()) {
                $deviceType = 'tablet';
            } else {
                $deviceType = 'desktop';
            }

            $deviceName = $agent->platform() . ' - ' . $agent->browser();
            $userAgent = $request->userAgent();
            $ipAddress = $request->ip();
            $serialNumber = md5($userAgent . $ipAddress);

            Log::info('Tentando registrar dispositivo', [
                'user_id' => $user->id,
                'device_type' => $deviceType,
                'device_name' => $deviceName,
                'ip' => $ipAddress
            ]);

            // Verificar se precisamos adicionar as colunas necessárias
            if (!self::hasRequiredColumns()) {
                self::updateDeviceTable();
            }

            // Marcar todos os outros dispositivos como não-atuais
            $user->devices()->update(['is_current' => false]);

            // Verificar se este dispositivo já existe para este usuário
            $existingDevice = $user->devices()
                ->where(function($query) use ($userAgent, $ipAddress, $serialNumber) {
                    $query->where('user_agent', $userAgent)
                        ->where('ip_address', $ipAddress)
                        ->orWhere('serial_number', $serialNumber);
                })
                ->first();

            if ($existingDevice) {
                Log::info('Atualizando dispositivo existente', [
                    'device_id' => $existingDevice->id,
                    'user_id' => $user->id
                ]);

                // Atualizar o dispositivo existente
                $existingDevice->update([
                    'last_active' => now(),
                    'is_current' => true,
                    'name' => $deviceName, // Atualiza o nome caso o navegador ou sistema tenha mudado
                    'device_type' => $deviceType,
                    'user_agent' => $userAgent,
                    'ip_address' => $ipAddress
                ]);

                return $existingDevice;
            }

            // Criar novo dispositivo
            Log::info('Criando novo dispositivo', [
                'user_id' => $user->id,
                'device_type' => $deviceType
            ]);

            return Device::create([
                'user_id' => $user->id,
                'name' => $deviceName,
                'device_type' => $deviceType,
                'ip_address' => $ipAddress,
                'user_agent' => $userAgent,
                'last_active' => now(),
                'is_current' => true,
                'serial_number' => $serialNumber
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao registrar dispositivo: ' . $e->getMessage(), [
                'exception' => $e,
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    public static function checkDeviceLimit(Request $request)
    {
        $user = Auth::user();
        if (!$user) return true;

        // Usar o novo método para determinar o limite de dispositivos
        $controller = new self();
        $deviceLimit = $controller->getUserDeviceLimit($user);

        // Contar dispositivos ativos (com base na estrutura existente)
        $query = $user->devices();

        // Verificar se devemos adicionar filtro de data ativa
        if (self::columnExists('devices', 'last_active')) {
            $query->where('last_active', '>=', now()->subDays(30));
        }

        $activeDevices = $query->count();

        // Log para debug
        Log::info("Verificando limite: User {$user->id}, Limite {$deviceLimit}, Ativos {$activeDevices}");

        // Verificar se o limite foi atingido
        if ($activeDevices >= $deviceLimit) {
            // Verificar se este é um dispositivo já registrado
            $deviceQuery = $user->devices();

            if (self::columnExists('devices', 'user_agent') && self::columnExists('devices', 'ip_address')) {
                $deviceQuery->where('user_agent', $request->userAgent())
                    ->where('ip_address', $request->ip());
            } else {
                // Verificação alternativa usando serial_number
                $deviceQuery->where('serial_number', md5($request->userAgent() . $request->ip()));
            }

            $isExistingDevice = $deviceQuery->exists();

            if ($isExistingDevice) {
                return true; // Permitir dispositivo existente
            }

            return false; // Bloquear novo dispositivo
        }

        return true; // Permitir novo dispositivo
    }

    public function checkDeviceLimitForContent(Request $request)
    {
        $canAccess = self::checkDeviceLimit($request);

        if (!$canAccess) {
            return response()->json([
                'success' => false,
                'message' => 'Limite de dispositivos excedido',
                'limitExceededHtml' => view('layouts.pages.device_limit_exceeded', [
                    'settings' => \App\Setting::first()
                ])->render()
            ]);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Verifica se as colunas necessárias existem na tabela
     */
    private static function hasRequiredColumns()
    {
        return self::columnExists('devices', 'last_active') &&
            self::columnExists('devices', 'is_current') &&
            self::columnExists('devices', 'user_agent') &&
            self::columnExists('devices', 'ip_address') &&
            self::columnExists('devices', 'device_type');
    }

    /**
     * Verifica se uma coluna existe na tabela
     */
    private static function columnExists($table, $column)
    {
        return \Schema::hasColumn($table, $column);
    }

    /**
     * Atualiza a tabela de dispositivos para adicionar as colunas necessárias
     */
    private static function updateDeviceTable()
    {
        try {
            \Schema::table('devices', function ($table) {
                if (!self::columnExists('devices', 'device_type')) {
                    $table->string('device_type')->nullable();
                }

                if (!self::columnExists('devices', 'ip_address')) {
                    $table->string('ip_address')->nullable();
                }

                if (!self::columnExists('devices', 'user_agent')) {
                    $table->text('user_agent')->nullable();
                }

                if (!self::columnExists('devices', 'last_active')) {
                    $table->timestamp('last_active')->nullable();
                }

                if (!self::columnExists('devices', 'is_current')) {
                    $table->boolean('is_current')->default(false);
                }

                if (!self::columnExists('devices', 'serial_number')) {
                    $table->string('serial_number')->nullable();
                }
            });

            return true;
        } catch (\Exception $e) {
            \Log::error('Falha ao atualizar tabela de dispositivos: ' . $e->getMessage());
            return false;
        }
    }
}
