<?php

namespace App\Http\Controllers;

use App\ContinueWatching;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class ContinueWatchingController extends Controller
{
    /**
     * Salva ou atualiza o registro de "Continuar Assistindo"
     */
    public function store(Request $request)
    {
        // Validação mais leniente para garantir que pelo menos os dados essenciais estão presentes
        $request->validate([
            'item_id' => 'required',
            'type' => 'required|string',
        ]);

        $userId = Auth::id();

        // Log completo para debug
        Log::info('Recebido request continue-watching', [
            'user_id' => $userId,
            'data' => $request->all()
        ]);

        // Valores padrão para campos opcionais
        $position = $request->has('position') ? $request->position : 0;
        $duration = $request->has('duration') ? $request->duration : 0;
        $episodeId = $request->has('episode_id') ? $request->episode_id : null;
        $seasonNumber = $request->has('season_number') ? $request->season_number : 1;
        $episodeNumber = $request->has('episode_number') ? $request->episode_number : 1;

        // Correto: Obtém o ID do perfil, não o objeto inteiro
        $profileId = null;

        // Verifica se temos uma sessão de perfil selecionado
        if (session()->has('selected_profile')) {
            // Extrair o ID do perfil da sessão
            $profile = session('selected_profile');

            // Se o perfil é um objeto, pega apenas o ID
            if (is_object($profile)) {
                $profileId = $profile->id;
            } else if (is_array($profile)) {
                $profileId = $profile['id'] ?? null;
            } else if (is_numeric($profile)) {
                $profileId = $profile;
            }
        }

        // Buscar registro existente para evitar duplicatas
        $existingQuery = ContinueWatching::where('user_id', $userId)
            ->where('item_id', $request->item_id)
            ->where('type', $request->type);

        // Para séries e animes, SEMPRE verificar pelo episódio específico se houver
        if (($request->type === 'serie' || $request->type === 'anime') && $episodeId) {
            $existingQuery->where('episode_id', $episodeId);
        }

        // Ver se já existe um registro
        $existing = $existingQuery->first();

        try {
            if ($existing) {
                // Atualizar registro existente
                $data = [
                    'position' => $position,
                    'duration' => $duration,
                    'profile_id' => $profileId,
                ];

                // Para séries e animes, garantir que os campos de episódio são atualizados
                if (($request->type === 'serie' || $request->type === 'anime') && $episodeId) {
                    $data['episode_id'] = $episodeId;
                    $data['season_number'] = $seasonNumber;
                    $data['episode_number'] = $episodeNumber;
                }

                // Log para debug
                Log::info('Atualizando continue watching com dados:', [
                    'existing_id' => $existing->id,
                    'data' => $data
                ]);

                $existing->update($data);

                // Garantir que as atualizações são salvas imediatamente
                $existing->touch();

                return response()->json([
                    'message' => 'Registro atualizado com sucesso',
                    'data' => $existing,
                    'updated' => true
                ]);
            } else {
                // Criar um novo registro com todos os campos
                $continueWatching = new ContinueWatching([
                    'user_id' => $userId,
                    'profile_id' => $profileId,
                    'item_id' => $request->item_id,
                    'type' => $request->type,
                    'position' => $position,
                    'duration' => $duration
                ]);

                // Adicionar campos de episódio se for série ou anime
                if (($request->type === 'serie' || $request->type === 'anime') && $episodeId) {
                    $continueWatching->episode_id = $episodeId;
                    $continueWatching->season_number = $seasonNumber;
                    $continueWatching->episode_number = $episodeNumber;
                }

                // Salvar e garantir que é um INSERT, não um UPDATE
                $continueWatching->save();

                // Log para debug
                Log::info('Novo registro de continue watching criado:', [
                    'id' => $continueWatching->id,
                    'data' => $continueWatching->toArray()
                ]);

                return response()->json([
                    'message' => 'Registro criado com sucesso',
                    'data' => $continueWatching,
                    'created' => true
                ]);
            }
        } catch (\Exception $e) {
            // Log do erro completo para diagnóstico
            Log::error('Erro ao salvar continue watching: ' . $e->getMessage(), [
                'exception' => $e,
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all(),
                'profile_id' => $profileId,
                'user_id' => $userId
            ]);

            return response()->json([
                'message' => 'Erro ao salvar registro',
                'error' => $e->getMessage(),
                'request_data' => $request->all() // Retorna os dados enviados para diagnóstico
            ], 500);
        }
    }

    /**
     * Obtém os registros de "Continuar Assistindo" para o usuário atual
     */
    public function index()
    {
        $userId = Auth::id();
        $profileId = session('selected_profile');

        $continueWatching = ContinueWatching::where('user_id', $userId)
            ->where('profile_id', $profileId)
            ->orderBy('updated_at', 'desc')
            ->get();

        return response()->json($continueWatching);
    }

    /**
     * Remove um registro de "Continuar Assistindo"
     */
    public function destroy($id)
    {
        $userId = Auth::id();
        $profileId = session('selected_profile');

        $continueWatching = ContinueWatching::where('id', $id)
            ->where('user_id', $userId)
            ->where('profile_id', $profileId)
            ->first();

        if (!$continueWatching) {
            return response()->json(['message' => 'Registro não encontrado'], 404);
        }

        $continueWatching->delete();

        return response()->json(['message' => 'Registro removido com sucesso']);
    }

    /**
     * Obtém a posição salva para um item específico
     */
    public function getPosition(Request $request)
    {
        try {
            // Validação essencial
            $request->validate([
                'item_id' => 'required',
                'type' => 'required|string',
            ]);

            $userId = Auth::id();

            // Extrair o ID do perfil corretamente
            $profileId = null;

            if (session()->has('selected_profile')) {
                $profile = session('selected_profile');

                if (is_object($profile)) {
                    $profileId = $profile->id;
                } else if (is_array($profile)) {
                    $profileId = $profile['id'] ?? null;
                } else if (is_numeric($profile)) {
                    $profileId = $profile;
                }
            }

            // Para séries e animes, é ESSENCIAL usar o episode_id na consulta
            $episodeId = $request->input('episode_id');

            // Log breve
            Log::info('Buscando posição', [
                'user_id' => $userId,
                'item_id' => $request->item_id,
                'type' => $request->type,
                'episode_id' => $episodeId,
                'profile_id' => $profileId
            ]);

            // Construir a consulta base
            $query = ContinueWatching::where('user_id', $userId)
                ->where('item_id', $request->item_id)
                ->where('type', $request->type);

            // Adicionar filtro de perfil se disponível
            if ($profileId) {
                $query->where('profile_id', $profileId);
            }

            // Para séries e animes, SEMPRE filtrar pelo episode_id quando fornecido
            if (($request->type === 'serie' || $request->type === 'anime') && $episodeId) {
                $query->where('episode_id', $episodeId);
            }

            // Ordenar por atualização mais recente e pegar o primeiro
            $continueWatching = $query->orderBy('updated_at', 'desc')->first();

            // Resposta padrão se não tiver registro
            if (!$continueWatching) {
                return response()->json([
                    'position' => 0,
                    'duration' => 0,
                    'progress' => 0,
                    'season_number' => $request->input('season_number', 1),
                    'episode_number' => $request->input('episode_number', 1),
                    'is_near_end' => false,
                    'status' => 'not_found'
                ]);
            }

            Log::info('Registro encontrado', [
                'episode_id' => $episodeId,
                'position' => $continueWatching->position,
                'duration' => $continueWatching->duration
            ]);

            // Calcular o progresso como porcentagem
            $progress = 0;
            if ($continueWatching->duration > 0) {
                $progress = min(100, round(($continueWatching->position / $continueWatching->duration) * 100));
            }

            // Verificar se a posição está perto do final (95% ou mais)
            $isNearEnd = false;
            if ($continueWatching->duration > 0 && $continueWatching->position > 0) {
                $isNearEnd = ($continueWatching->position / $continueWatching->duration) >= 0.95;
            }

            // Se estiver perto do final, retornar posição 0 para reiniciar o vídeo
            $position = $isNearEnd ? 0 : $continueWatching->position;

            return response()->json([
                'position' => $position,
                'duration' => $continueWatching->duration,
                'progress' => $progress,
                'season_number' => $continueWatching->season_number,
                'episode_number' => $continueWatching->episode_number,
                'is_near_end' => $isNearEnd,
                'status' => 'found',
                'record_id' => $continueWatching->id
            ]);
        } catch (\Exception $e) {
            // Log simplificado do erro para não sobrecarregar
            Log::error('Erro ao obter posição: ' . $e->getMessage(), [
                'request' => $request->all()
            ]);

            // Retornar resposta padrão de fallback
            return response()->json([
                'position' => 0,
                'duration' => 0,
                'progress' => 0,
                'season_number' => $request->input('season_number', 1),
                'episode_number' => $request->input('episode_number', 1),
                'is_near_end' => false,
                'error' => $e->getMessage(),
                'status' => 'error'
            ], 200);
        }
    }
}
