<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Movie;
use App\Serie;
use App\Anime;
use App\Setting;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ContentController extends Controller
{
    /**
     * Exibir mais conteúdo baseado no tipo
     */
    public function showMore($type, Request $request)
    {
        $settings = Setting::first();
        $title = '';
        $content = collect();

        // Seleção dos campos padrão
        $selectMovie = [
            'id', 'title AS name', 'poster_path', 'backdrop_path', 'backdrop_path_tv',
            'vote_average', 'subtitle', 'overview', 'release_date', 'created_at',
            'views', DB::raw("'movie' AS type")
        ];

        $selectSerie = [
            'id', 'name', 'poster_path', 'backdrop_path', 'backdrop_path_tv',
            'vote_average', 'subtitle', 'overview', 'first_air_date AS release_date',
            'created_at', 'views', DB::raw("'serie' AS type")
        ];

        $selectAnime = [
            'id', 'name', 'poster_path', 'backdrop_path', 'backdrop_path_tv',
            'vote_average', 'subtitle', 'overview', 'first_air_date AS release_date',
            'created_at', 'views', DB::raw("'anime' AS type")
        ];

        // Expressões para gêneros
        $genresMovies = DB::raw('(SELECT SUBSTRING_INDEX(GROUP_CONCAT(genres.name SEPARATOR ", "), ",", 1)
            FROM genres JOIN movie_genres ON genres.id = movie_genres.genre_id
            WHERE movie_genres.movie_id = movies.id) AS genre_name');

        $genresSeries = DB::raw('(SELECT SUBSTRING_INDEX(GROUP_CONCAT(genres.name SEPARATOR ", "), ",", 1)
            FROM genres JOIN serie_genres ON genres.id = serie_genres.genre_id
            WHERE serie_genres.serie_id = series.id) AS genre_name');

        $genresAnimes = DB::raw('(SELECT SUBSTRING_INDEX(GROUP_CONCAT(genres.name SEPARATOR ", "), ",", 1)
            FROM genres JOIN anime_genres ON genres.id = anime_genres.genre_id
            WHERE anime_genres.anime_id = animes.id) AS genre_name');

        // Busca conteúdo baseado no tipo de seção
        switch ($type) {
            case 'continue':
                $title = 'Continue Assistindo';
                // Para Continue Assistindo, precisamos dos itens do usuário atual
                $user_id = auth()->id();
                $content = $this->getContinueWatching($user_id);
                break;

            case 'latest':
                $title = 'Adicionados Recentemente';
                // Buscar filmes e séries mais recentes
                $content = $this->getLatestContent($selectMovie, $selectSerie, $selectAnime,
                                                  $genresMovies, $genresSeries, $genresAnimes, $settings);
                break;

            case 'trending':
                $title = 'Em Alta';
                // Buscar conteúdo mais visualizado
                $content = $this->getTrendingContent($selectMovie, $selectSerie, $selectAnime,
                                                   $genresMovies, $genresSeries, $genresAnimes, $settings);
                break;

            case 'series':
                $title = 'Últimas Séries';
                // Buscar séries recentes
                $content = $this->getRecentSeries($selectSerie, $genresSeries);
                break;

            case 'novelas':
                $title = 'Novelas';
                // Buscar novelas (gênero específico)
                $content = $this->getNovelas($selectMovie, $selectSerie, $selectAnime,
                                           $genresMovies, $genresSeries, $genresAnimes, $settings);
                break;

            case 'recommended':
                $title = 'Recomendados';
                // Buscar conteúdo com alta avaliação
                $content = $this->getRecommendedContent($selectMovie, $selectSerie, $selectAnime,
                                                      $genresMovies, $genresSeries, $genresAnimes, $settings);
                break;

            default:
                // Caso tipo não seja reconhecido, redireciona para home
                return redirect()->route('inicio');
        }

        // Verifica se é uma requisição AJAX
        if ($request->ajax()) {
            // Renderiza apenas os itens para o infinite scroll
            $html = view('layouts.partials._content_items', [
                'content' => $content,
                'type' => $type
            ])->render();

            return response()->json([
                'html' => $html,
                'lastPage' => $content->lastPage(),
                'currentPage' => $content->currentPage(),
                'total' => $content->total()
            ]);
        }

        return view('layouts.pages.content-more', [
            'title' => $title,
            'type' => $type,
            'content' => $content,
            'settings' => $settings
        ]);
    }

    /**
     * Obter conteúdo de Continue Assistindo
     */
    private function getContinueWatching($user_id)
    {
        $continueWatching = DB::table('continue_watchings')
                            ->where('user_id', $user_id)
                            ->orderByDesc('updated_at')
                            ->get();

        $results = collect();
        foreach ($continueWatching as $item) {
            switch ($item->type) {
                case 'movie':
                    $content = DB::table('movies')
                                ->where('id', $item->item_id)
                                ->first(['id', 'title as name', 'backdrop_path', 'poster_path',
                                         DB::raw("'movie' as type")]);
                    break;
                case 'serie':
                    $content = DB::table('episodes')
                                ->join('seasons', 'episodes.season_id', '=', 'seasons.id')
                                ->join('series', 'seasons.serie_id', '=', 'series.id')
                                ->where('episodes.id', $item->item_id)
                                ->first([
                                    'series.id as id',
                                    'series.name as name',
                                    'series.poster_path',
                                    'episodes.still_path as backdrop_path',
                                    DB::raw("'serie' as type"),
                                    'episodes.name as episode_name',
                                    'series.id as serie_id'
                                ]);
                    break;
                case 'anime':
                    $content = DB::table('anime_episodes')
                                ->join('anime_seasons', 'anime_episodes.anime_season_id', '=', 'anime_seasons.id')
                                ->join('animes', 'anime_seasons.anime_id', '=', 'animes.id')
                                ->where('anime_episodes.id', $item->item_id)
                                ->first([
                                    'animes.id as id',
                                    'animes.name as name',
                                    'animes.poster_path',
                                    'anime_episodes.still_path as backdrop_path',
                                    DB::raw("'anime' as type"),
                                    'anime_episodes.name as episode_name'
                                ]);
                    break;
                default:
                    $content = null;
                    break;
            }

            if ($content) {
                $content->progress = $item->progress ?? 0;
                $results->push($content);
            }
        }

        return $results;
    }

    /**
     * Obter conteúdo mais recente
     */
    private function getLatestContent($selectMovie, $selectSerie, $selectAnime,
                                      $genresMovies, $genresSeries, $genresAnimes, $settings)
    {
        // Corrigido: Especificando claramente a qual tabela cada campo pertence
        $query = DB::table('movies')
                ->select([
                    'movies.id', 'movies.title AS name', 'movies.poster_path', 'movies.backdrop_path',
                    'movies.backdrop_path_tv', 'movies.vote_average', 'movies.subtitle', 'movies.overview',
                    'movies.release_date', 'movies.created_at', 'movies.views',
                    DB::raw("'movie' AS type"), $genresMovies
                ])
                ->where('active', 1)
                ->orderByDesc('created_at');

        $query->unionAll(
            DB::table('series')
                ->select([
                    'series.id', 'series.name', 'series.poster_path', 'series.backdrop_path',
                    'series.backdrop_path_tv', 'series.vote_average', 'series.subtitle', 'series.overview',
                    'series.first_air_date AS release_date', 'series.created_at', 'series.views',
                    DB::raw("'serie' AS type"), $genresSeries
                ])
                ->where('active', 1)
                ->orderByDesc('created_at')
        );

        if ($settings->anime) {
            $query->unionAll(
                DB::table('animes')
                    ->select([
                        'animes.id', 'animes.name', 'animes.poster_path', 'animes.backdrop_path',
                        'animes.backdrop_path_tv', 'animes.vote_average', 'animes.subtitle', 'animes.overview',
                        'animes.first_air_date AS release_date', 'animes.created_at', 'animes.views',
                        DB::raw("'anime' AS type"), $genresAnimes
                    ])
                    ->where('active', 1)
                    ->orderByDesc('created_at')
            );
        }

        return $query->orderByDesc('created_at')
                    ->paginate(24);
    }

    /**
     * Obter conteúdo em alta (mais visualizado)
     */
    private function getTrendingContent($selectMovie, $selectSerie, $selectAnime,
                                       $genresMovies, $genresSeries, $genresAnimes, $settings)
    {
        // Corrigido: Especificando claramente a qual tabela cada campo pertence
        $query = DB::table('movies')
                ->select([
                    'movies.id', 'movies.title AS name', 'movies.poster_path', 'movies.backdrop_path',
                    'movies.backdrop_path_tv', 'movies.vote_average', 'movies.subtitle', 'movies.overview',
                    'movies.release_date', 'movies.created_at', 'movies.views',
                    DB::raw("'movie' AS type"), $genresMovies
                ])
                ->where('active', 1)
                ->where('views', '>', 10);

        $query->unionAll(
            DB::table('series')
                ->select([
                    'series.id', 'series.name', 'series.poster_path', 'series.backdrop_path',
                    'series.backdrop_path_tv', 'series.vote_average', 'series.subtitle', 'series.overview',
                    'series.first_air_date AS release_date', 'series.created_at', 'series.views',
                    DB::raw("'serie' AS type"), $genresSeries
                ])
                ->where('active', 1)
                ->where('views', '>', 10)
        );

        if ($settings->anime) {
            $query->unionAll(
                DB::table('animes')
                    ->select([
                        'animes.id', 'animes.name', 'animes.poster_path', 'animes.backdrop_path',
                        'animes.backdrop_path_tv', 'animes.vote_average', 'animes.subtitle', 'animes.overview',
                        'animes.first_air_date AS release_date', 'animes.created_at', 'animes.views',
                        DB::raw("'anime' AS type"), $genresAnimes
                    ])
                    ->where('active', 1)
                    ->where('views', '>', 10)
            );
        }

        return $query->orderByDesc('views')
                    ->paginate(24);
    }

    /**
     * Obter séries recentes
     */
    private function getRecentSeries($selectSerie, $genresSeries)
    {
        return DB::table('series')
                ->select(array_merge($selectSerie, [$genresSeries]))
                ->where('active', 1)
                ->orderByDesc('created_at')
                ->paginate(24);
    }

    /**
     * Obter novelas (gênero específico)
     */
    private function getNovelas($selectMovie, $selectSerie, $selectAnime,
                               $genresMovies, $genresSeries, $genresAnimes, $settings)
    {
        $genre = 10776; // ID do gênero de novelas

        // Corrigido: Especificando claramente a qual tabela cada campo pertence
        $query = DB::table('movies')
                ->select([
                    'movies.id', 'movies.title AS name', 'movies.poster_path', 'movies.backdrop_path',
                    'movies.backdrop_path_tv', 'movies.vote_average', 'movies.subtitle', 'movies.overview',
                    'movies.release_date', 'movies.created_at', 'movies.views',
                    DB::raw("'movie' AS type"), $genresMovies
                ])
                ->join('movie_genres', 'movies.id', '=', 'movie_genres.movie_id')
                ->where('movie_genres.genre_id', '=', $genre)
                ->where('movies.active', '=', 1);

        $query->unionAll(
            DB::table('series')
                ->select([
                    'series.id', 'series.name', 'series.poster_path', 'series.backdrop_path',
                    'series.backdrop_path_tv', 'series.vote_average', 'series.subtitle', 'series.overview',
                    'series.first_air_date AS release_date', 'series.created_at', 'series.views',
                    DB::raw("'serie' AS type"), $genresSeries
                ])
                ->join('serie_genres', 'series.id', '=', 'serie_genres.serie_id')
                ->where('serie_genres.genre_id', '=', $genre)
                ->where('series.active', '=', 1)
        );

        if ($settings->anime) {
            $query->unionAll(
                DB::table('animes')
                    ->select([
                        'animes.id', 'animes.name', 'animes.poster_path', 'animes.backdrop_path',
                        'animes.backdrop_path_tv', 'animes.vote_average', 'animes.subtitle', 'animes.overview',
                        'animes.first_air_date AS release_date', 'animes.created_at', 'animes.views',
                        DB::raw("'anime' AS type"), $genresAnimes
                    ])
                    ->join('anime_genres', 'animes.id', '=', 'anime_genres.anime_id')
                    ->where('anime_genres.genre_id', '=', $genre)
                    ->where('animes.active', '=', 1)
            );
        }

        return $query->orderByDesc('created_at')
                    ->paginate(24);
    }

    /**
     * Obter conteúdo recomendado (bem avaliado)
     */
    private function getRecommendedContent($selectMovie, $selectSerie, $selectAnime,
                                          $genresMovies, $genresSeries, $genresAnimes, $settings)
    {
        // Corrigido: Especificando claramente a qual tabela cada campo pertence
        $query = DB::table('movies')
                ->select([
                    'movies.id', 'movies.title AS name', 'movies.poster_path', 'movies.backdrop_path',
                    'movies.backdrop_path_tv', 'movies.vote_average', 'movies.subtitle', 'movies.overview',
                    'movies.release_date', 'movies.created_at', 'movies.views',
                    DB::raw("'movie' AS type"), $genresMovies
                ])
                ->where('active', 1)
                ->where('vote_average', '>', 7.0);

        $query->unionAll(
            DB::table('series')
                ->select([
                    'series.id', 'series.name', 'series.poster_path', 'series.backdrop_path',
                    'series.backdrop_path_tv', 'series.vote_average', 'series.subtitle', 'series.overview',
                    'series.first_air_date AS release_date', 'series.created_at', 'series.views',
                    DB::raw("'serie' AS type"), $genresSeries
                ])
                ->where('active', 1)
                ->where('vote_average', '>', 7.0)
        );

        if ($settings->anime) {
            $query->unionAll(
                DB::table('animes')
                    ->select([
                        'animes.id', 'animes.name', 'animes.poster_path', 'animes.backdrop_path',
                        'animes.backdrop_path_tv', 'animes.vote_average', 'animes.subtitle', 'animes.overview',
                        'animes.first_air_date AS release_date', 'animes.created_at', 'animes.views',
                        DB::raw("'anime' AS type"), $genresAnimes
                    ])
                    ->where('active', 1)
                    ->where('vote_average', '>', 7.0)
            );
        }

        return $query->inRandomOrder()
                    ->paginate(24);
    }
}
