<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class CaktoWebhookController extends Controller
{
    private $planos = [
        '6jytyrs' => 1,  // 1 mês
        '10' => 1,       // 1 mês
        'D9o3qiL' => 3,  // 3 meses
        '8nLhwq5' => 6   // 6 meses
    ];

    public function handle(Request $request)
    {
        $payload = $request->all();
        Log::info('Webhook recebido:', $payload);

        if ($payload['event'] !== 'purchase_approved') {
            Log::info('Evento não processado: ' . $payload['event']);
            return response()->json(['message' => 'Evento não processado'], 200);
        }

        $offerId = $payload['data']['offer']['id'];
        if (!array_key_exists($offerId, $this->planos)) {
            Log::info('Plano não reconhecido: ' . $offerId);
            return response()->json(['message' => 'Plano não reconhecido'], 200);
        }

        try {
            DB::beginTransaction();
            
            $user = $this->processUser($payload['data']);
            
            DB::commit();

            Log::info('Usuário processado com sucesso:', ['user_id' => $user->id, 'email' => $user->email]);
            return response()->json([
                'message' => 'Assinatura ativada com sucesso',
                'plano' => $this->planos[$offerId] . ' mese(s)',
                'expira_em' => $user->expired_in
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Erro ao processar webhook: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json(['message' => 'Erro ao processar assinatura: ' . $e->getMessage()], 500);
        }
    }

    private function processUser(array $data)
    {
        $email = $data['customer']['email'];
        $nome = $data['customer']['name'];
        $offerId = $data['offer']['id'];
        $meses = $this->planos[$offerId];

        Log::info('Processando usuário:', ['email' => $email, 'nome' => $nome, 'plano' => $offerId]);

        $user = User::firstOrNew(['email' => $email]);

        $isNewUser = !$user->exists;

        $user->name = $nome;
        $user->premuim = true;
        $user->expired_in = $this->calculateExpirationDate($meses, $user);

        if ($isNewUser) {
            $user->password = Hash::make(Str::random(10));
        }

        $user->save();

        Log::info('Resultado do processamento:', [
            'user_id' => $user->id, 
            'created' => $isNewUser,
            'expiration' => $user->expired_in
        ]);

        return $user;
    }

    private function calculateExpirationDate($meses, User $user)
    {
        $now = Carbon::now();

        if ($user->exists && $user->expired_in && $user->expired_in > $now) {
            return Carbon::parse($user->expired_in)->addMonths($meses);
        }

        return $now->addMonths($meses);
    }
}