<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Controllers\DeviceController;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Facades\Auth;
use App\UserSession;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class LoginController extends Controller
{
    use AuthenticatesUsers, HasApiTokens;  // Adicione o trait HasApiTokens

    protected function redirectTo()
    {
        return route('choose-profile');
    }

    public function showLoginForm()
    {
        return view('auth.login');
    }

    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    /**
     * Sobrescrito para adicionar funcionalidade de registro de dispositivo
     * após autenticação bem-sucedida
     */
    protected function authenticated(Request $request, $user)
    {
        try {
            // Registrar o dispositivo atual
            $device = DeviceController::registerDevice($request);

            if ($device) {
                Log::info('Dispositivo registrado após login', [
                    'user_id' => $user->id,
                    'device_id' => $device->id,
                    'device_name' => $device->name
                ]);
            } else {
                Log::warning('Falha ao registrar dispositivo após login', [
                    'user_id' => $user->id
                ]);
            }

            // Gerar token de dispositivo para sessão única
            $deviceToken = Str::uuid();

            // Obter sessões atuais
            $activeSessions = UserSession::where('user_id', $user->id)->orderBy('created_at', 'asc')->get();

            // Verificar se excedeu o limite de sessões
            if ($activeSessions->count() >= UserSession::MAX_ACTIVE_SESSIONS) {
                // Invalidar a sessão mais antiga
                $activeSessions->first()->delete();
            }

            // Armazenar nova sessão
            UserSession::create([
                'user_id' => $user->id,
                'device_token' => $deviceToken,
            ]);

            // Adicionar o token de dispositivo à sessão
            session(['device_token' => $deviceToken]);

            return redirect()->route('choose-profile');
        } catch (\Exception $e) {
            Log::error('Erro no processo de autenticação', [
                'error' => $e->getMessage(),
                'user_id' => $user->id
            ]);

            // Mesmo com erro, permitir login
            return redirect()->route('choose-profile');
        }
    }

    /**
     * Manipula login para usuários premium
     */
    public function premiumLogin(Request $request)
    {
        // Validação dos dados de entrada
        $this->validate($request, [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        // Capturar informações para diagnóstico
        $email = $request->email;

        try {
            // Tenta logar
            if (Auth::attempt(['email' => $email, 'password' => $request->password])) {
                $user = Auth::user();

                // Registrar o dispositivo
                $device = DeviceController::registerDevice($request);

                if ($device) {
                    Log::info('Dispositivo registrado para login premium', [
                        'user_id' => $user->id,
                        'device_id' => $device->id,
                        'device_name' => $device->name
                    ]);
                }

        } else {
            
        }
    } catch (\Exception $e) {
        // Capturar qualquer exceção durante o processo
        Log::error('Erro no login premium', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'email' => $email
        ]);

        return back()->with('error', 'Ocorreu um erro durante o login. Por favor, tente novamente.');
    }
}
}
