<?php

namespace App\Http\Controllers;

use App\Embed;
use App\AnimeEpisode;
use App\AnimeSubstitle;
use App\AnimeCast;
use App\Genre;
use App\Cast;
use App\Network;
use App\Http\Requests\AnimeStoreRequest;
use App\Http\Requests\AnimeUpdateRequest;
use App\Http\Requests\StoreImageRequest;
use App\Jobs\SendNotification;
use App\AnimeSeason;
use App\Anime;
use App\AnimeGenre;
use App\AnimeVideo;
use App\AnimeDownload;
use App\AnimeNetwork;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Profile;

class AnimeController extends Controller
{


    public function __construct()
    {
        $this->middleware('doNotCacheResponse', ['only' => ['moviecomment','addcomment']]);
    }



// returns all animes except children animes, for api.
public function index()
{
    $anime = Anime::whereDoesntHave('genres', function ($genre) {
        $genre->where('genre_id', '=', 10762);
    })->orderByDesc('id')->paginate(12);

    return response()->json($anime, 200);

}



public function moviecomment($movie)
{

    $movie = Anime::where('id', '=', $movie)->first();

    $comments = $movie->comments;

    return response()
        ->json(['comments' => $comments], 200);

}



public function addcomment(Request $request)
{


    $user = Auth::user();


    $this->validate($request, [
        'comments_message' => 'required',
        'movie_id' => 'required'
    ]);

    $movie = Anime::where('id', '=', $request->movie_id)->first();

    $comment = $movie->commentAsUser($user, $request->comments_message);

    return response()->json($comment, 200);

}

public function addtofav($serieId, Request $request)
{
    $profileId = session('selected_profile')->id;
    $profile = Profile::find($profileId);

    if (!$profile) {
        return response()->json(['message' => 'Perfil não encontrado', 'isFavorite' => false], 404);
    }

    $serie = Anime::find($serieId);
    if (!$serie) {
        return response()->json(['message' => 'Série não encontrada', 'isFavorite' => false], 404);
    }

    try {
        // Verificar se a série já está nos favoritos usando SQL direto para evitar problemas com cache
        $exists = DB::table('favorites')
                    ->where('profile_id', $profileId)
                    ->where('favoriteable_id', $serieId)
                    ->where('favoriteable_type', Anime::class)
                    ->exists();

        if (!$exists) {
            DB::beginTransaction();

            DB::table('favorites')->insert([
                'profile_id' => $profileId,
                'favoriteable_id' => $serieId,
                'favoriteable_type' => Anime::class,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Adicionado aos favoritos',
                'isFavorite' => true
            ]);
        }

        return response()->json([
            'message' => 'Série já está nos favoritos',
            'isFavorite' => true
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Erro ao adicionar favorito: ' . $e->getMessage());
        return response()->json([
            'message' => 'Erro ao adicionar aos favoritos: ' . $e->getMessage(),
            'isFavorite' => false,
            'error' => $e->getMessage()
        ], 500);
    }
}

public function removefromfav($serieId, Request $request)
{
    $profileId = session('selected_profile')->id;
    $profile = Profile::find($profileId);

    if (!$profile) {
        return response()->json(['message' => 'Perfil não encontrado', 'isFavorite' => false], 404);
    }

    $serie = Anime::find($serieId);
    if (!$serie) {
        return response()->json(['message' => 'Série não encontrada', 'isFavorite' => false], 404);
    }

    try {
        DB::beginTransaction();

        // Remover diretamente utilizando query builder para garantir consistência
        $deleted = DB::table('favorites')
                    ->where('profile_id', $profileId)
                    ->where('favoriteable_id', $serieId)
                    ->where('favoriteable_type', Anime::class)
                    ->delete();

        DB::commit();

        return response()->json([
            'message' => 'Removido dos favoritos',
            'isFavorite' => false
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        \Log::error('Erro ao remover favorito: ' . $e->getMessage());
        return response()->json([
            'message' => 'Erro ao remover dos favoritos: ' . $e->getMessage(),
            'isFavorite' => true,
            'error' => $e->getMessage()
        ], 500);
    }
}

public function isMovieFavorite($serieId, Request $request)
{
    $profileId = session('selected_profile')->id;

    if (!$profileId) {
        return response()->json(['status' => 404, 'error' => 'Perfil não encontrado', 'isFavorite' => 0], 404);
    }

    try {
        // Verificar se a série está nos favoritos usando SQL direto para evitar problemas com cache
        $isFavorite = DB::table('favorites')
                        ->where('profile_id', $profileId)
                        ->where('favoriteable_id', $serieId)
                        ->where('favoriteable_type', Anime::class)
                        ->exists();

        return response()->json([
            'status' => 200,
            'isFavorite' => $isFavorite ? 1 : 0
        ]);
    } catch (\Exception $e) {
        \Log::error('Erro ao verificar favorito: ' . $e->getMessage());
        return response()->json([
            'status' => 500,
            'error' => 'Erro ao verificar favorito',
            'message' => $e->getMessage(),
            'isFavorite' => 0
        ], 500);
    }
}




// returns all animes for admin panel
public function data()
{
    return response()->json(Anime::with('seasons.episodes.videos','genres','casters','networks')->orderByDesc('created_at')
    ->paginate(6), 200);

}

// returns a specific anime
public function show($id)
{

    $serie = Anime::with(['casters.cast' => function ($query) {
        $query->select('id', 'name','original_name','profile_path');
    },'seasons.episodes.videos'])->where('id', '=', $id)->first()->makeHidden(['casters','networks']);

    $serie->increment('views',1);


    //ray()->showQueries1();
    ray()->measure();

    return response()->json($serie, 200);

}


// create a new anime in the database
public function store(AnimeStoreRequest $request)
{
    $anime = new Anime();
    $anime->fill($request->anime);
    $anime->save();


    $this->onSaveAnimeGenre($request,$anime);
    $this->onSaveAnimeSeasons($request,$anime);
    $this->onSaveAnimeCasters($request,$anime);
    $this->onSaveAnimeNetworks($request,$anime);



    if ($request->notification) {
        $this->dispatch(new SendNotification($anime));
    }

    $data = [
        'status' => 200,
        'message' => 'successfully created',
        'body' => $anime->load('seasons.episodes.videos')
    ];

    return response()->json($data, $data['status']);
}




public function onSaveAnimeNetworks($request,$anime) {

    if ($request->anime['networks']) {
        foreach ($request->anime['networks'] as $network) {
            $find = Network::find($network['id']);
            if ($find == null) {
                $find = new Network();
                $find->fill($network);
                $find->save();
            }
            $serieNetwork = new AnimeNetwork();
            $serieNetwork->network_id = $network['id'];
            $serieNetwork->anime_id = $anime->id;
            $serieNetwork->save();
        }
    }

}

public function onSaveAnimeCasters($request,$anime) {

    if ($request->anime['casterslist']) {
        foreach ($request->anime['casterslist'] as $cast) {
            $find = Cast::find($cast['id']);
            if ($find == null) {
                $find = new Cast();
                $find->fill($cast);
                $find->save();
            }
            $movieGenre = new AnimeCast();
            $movieGenre->cast_id = $cast['id'];
            $movieGenre->anime_id = $anime->id;
            $movieGenre->save();
        }
    }

}


public function onSaveAnimeGenre($request,$anime) {

    if ($request->anime['genres']) {
        foreach ($request->anime['genres'] as $genre) {
            $find = Genre::find($genre['id']);
            if ($find == null) {
                $find = new Genre();
                $find->fill($genre);
                $find->save();
            }
            $animeGenre = new AnimeGenre();
            $animeGenre->genre_id = $genre['id'];
            $animeGenre->anime_id = $anime->id;
            $animeGenre->save();
        }
    }

}


public function onSaveAnimeSeasons($request , $anime){

    if ($request->anime['seasons']) {
        foreach ($request->anime['seasons'] as $reqSeason) {
            $season = new AnimeSeason();
            $season->fill($reqSeason);
            $season->anime_id = $anime->id;
            $season->save();

            $this->onSaveEpisodes($request,$reqSeason,$season);


        }
    }

}


public function onSaveEpisodes($request, $reqSeason,$season) {

    if ($reqSeason['episodes']) {
        foreach ($reqSeason['episodes'] as $reqEpisode) {
            $episode = new AnimeEpisode();
            $episode->fill($reqEpisode);
            $episode->anime_season_id = $season->id;
            $episode->save();


            if (isset($reqEpisode['videos'])) {
                foreach ($reqEpisode['videos'] as $reqVideo) {
                    $video = AnimeVideo::find($reqVideo['id'] ?? 0) ?? new AnimeVideo();
                    $video->fill($reqVideo);
                    $video->anime_episode_id = $episode->id;
                    $video->save();
                }
            }

            $this->onSaveEpisodeSubstitle($request,$reqEpisode,$episode);
            $this->onSaveEpisodeDownload($request,$reqEpisode,$episode);

        }
    }


}


public function onSaveEpisodeDownload($request,$reqEpisode,$episode) {

    if (isset($reqEpisode['downloads'])) {
        foreach ($reqEpisode['downloads'] as $reqVideo) {
            $video = AnimeDownload::find($reqVideo['id'] ?? 0) ?? new AnimeDownload();
            $video->fill($reqVideo);
            $video->anime_episode_id = $episode->id;
            $video->save();
        }
    }

}


public function onSaveEpisodeSubstitle($request,$reqEpisode,$episode) {


    if (isset($reqEpisode['substitles'])) {
               foreach ($reqEpisode['substitles'] as $reqVideo) {
                   $video = new AnimeSubstitle();
                   $video->fill($reqVideo);
                   $video->anime_episode_id = $episode->id;
                   $video->save();
               }
           }
}



// update a anime in the database
public function update(AnimeUpdateRequest $request, Anime $anime)
{

    $anime->fill($request->anime);
    $anime->save();

    $this->onUpdateAnimeGenre($request,$anime);
    $this->onUpdateAnimeSeasons($request,$anime);
    $this->onUpdateAnimeCasts($request,$anime);
    $this->onUpdateAnimeNetwork($request,$anime);

    $data = [
        'status' => 200,
        'message' => 'successfully updated',
        'body' => "Success"
    ];

    return response()->json($data, $data['status']);
}





public function onUpdateAnimeNetwork ($request,$anime) {

    if ($request->anime['networks']) {
        foreach ($request->anime['networks'] as $netwok) {
            if (!isset($netwok['network_id'])) {
                $find = Network::find($netwok['id']) ?? new Network();
                $find->fill($netwok);
                $find->save();
                $serieNetwork = AnimeNetwork::where('anime_id', $anime->id)->where('network_id', $netwok['id'])->get();
                if (count($serieNetwork) < 1) {
                    $serieNetwork = new AnimeNetwork();
                    $serieNetwork->network_id = $netwok['id'];
                    $serieNetwork->anime_id = $anime->id;
                    $serieNetwork->save();
                }
            }
        }
    }

}


public function onUpdateAnimeCasts ($request,$anime) {


    if ($request->anime['casterslist']) {
        foreach ($request->anime['casterslist'] as $genre) {

                $find = Cast::find($genre['id'] ?? 0) ?? new Cast();
                $find->fill($genre);
                $find->save();
                $movieGenre = AnimeCast::where('anime_id', $anime->id)
                    ->where('cast_id', $genre['id'])->get();

                if (count($movieGenre) < 1) {
                    $movieGenre = new AnimeCast();
                    $movieGenre->cast_id = $genre['id'];
                    $movieGenre->anime_id = $anime->id;
                    $movieGenre->save();

                }

        }
    }

}


public function onUpdateAnimeGenre ($request,$anime) {

    if ($request->anime['genres']) {
        foreach ($request->anime['genres'] as $genre) {
            if (!isset($genre['genre_id'])) {
                $find = Genre::find($genre['id']) ?? new Genre();
                $find->fill($genre);
                $find->save();
                $animeGenre = AnimeGenre::where('anime_id', $anime->id)->where('genre_id', $genre['id'])->get();
                if (count($animeGenre) < 1) {
                    $animeGenre = new AnimeGenre();
                    $animeGenre->genre_id = $genre['id'];
                    $animeGenre->anime_id = $anime->id;
                    $animeGenre->save();
                }
            }
        }
    }

}


public function onUpdateAnimeSeasons($request,$anime){


    if ($request->anime['seasons']) {
        foreach ($request->anime['seasons'] as $reqSeason) {
            $season = AnimeSeason::find($reqSeason['id'] ?? 0) ?? new AnimeSeason();
            $season->fill($reqSeason);
            $season->anime_id = $anime->id;
            $season->save();


            $this->onUpdateAnimeEpisodes($request,$reqSeason,$season);

        }
    }
}




public function onUpdateAnimeEpisodes ($request,$reqSeason,$season) {

    if ($reqSeason['episodes']) {
                foreach ($reqSeason['episodes'] as $reqEpisode) {
                    $episode = AnimeEpisode::find($reqEpisode['id'] ?? 0) ?? new AnimeEpisode();
                    $episode->fill($reqEpisode);
                    $episode->anime_season_id = $season->id;
                    $episode->save();
                    if (isset($reqEpisode['videos'])) {
                        foreach ($reqEpisode['videos'] as $reqVideo) {
                            $video = AnimeVideo::find($reqVideo['id'] ?? 0) ?? new AnimeVideo();
                            $video->fill($reqVideo);
                            $video->anime_episode_id = $episode->id;
                            $video->save();
                        }
                    }

                    $this->onUpdateAnimeSubstitle($request,$reqEpisode,$episode);
                    $this->onUpdateAnimeDownload($request,$reqEpisode,$episode);
                }
            }

}


public function onUpdateAnimeDownload ($request,$reqEpisode,$episode) {

    if (isset($reqEpisode['downloads'])) {
        foreach ($reqEpisode['downloads'] as $reqVideo) {

            $substitle = AnimeDownload::find($reqVideo['id'] ?? 0) ?? new AnimeDownload();
            $substitle->fill($reqVideo);
            $substitle->anime_episode_id = $episode->id;
            $substitle->save();
        }

}

}

public function onUpdateAnimeSubstitle ($request,$reqEpisode,$episode) {

    if (isset($reqEpisode['substitles'])) {
        foreach ($reqEpisode['substitles'] as $reqVideo) {

            $substitle = AnimeSubstitle::find($reqVideo['id'] ?? 0) ?? new AnimeSubstitle();
            $substitle->fill($reqVideo);
            $substitle->anime_episode_id = $episode->id;
            $substitle->save();
        }

}

}

// delete a anime from the database

    public function destroy(Anime $anime)
    {
        if ($anime != null) {
            $anime->delete();

            $data = [
                'status' => 200,
                'message' => 'successfully deleted',
            ];
        } else {
            $data = [
                'status' => 400,
                'message' => 'could not be deleted',
            ];
        }


        return response()->json($data, $data['status']);
    }


// remove a genre from a animes from the database
public function destroyGenre($genre)
{
    if ($genre != null) {

        AnimeGenre::find($genre)->delete();

        $data = ['status' => 200, 'message' => 'successfully deleted',];
    } else {
        $data = ['status' => 400, 'message' => 'could not be deleted',];
    }

    return response()->json($data, 200);
}

// save a new image in the animes folder of the storage
public function storeImg(StoreImageRequest $request)
{

    if ($request->hasFile('image')) {
        $filename = Storage::disk('animes')->put('', $request->image);
        $data = [
            'status' => 200,
            'image_path' => $request->root() . '/api/animes/image/' . $filename,
            'message' => 'image uploaded successfully'
        ];
    } else {
        $data = [
            'status' => 400,
            'message' => 'there was an error uploading the image'
        ];
    }

    return response()->json($data, $data['status']);
}

// return an image from the animes folder of the storage
public function getImg($filename)
{

    $image = Storage::disk('animes')->get($filename);

    $mime = Storage::disk('animes')->mimeType($filename);

    return (new Response($image, 200))
        ->header('Content-Type', $mime);
}


// returns a specific anime
public function showbyimdb($anime)
{

    $anime_by_imdbid = Anime::where('tmdb_id', '=', $anime)->first();


    return response()->json($anime_by_imdbid, 200);
}


// returns the last 10 animes added in the month
public function recents()
{



    $movies = Anime::select('animes.id','animes.name','animes.poster_path','animes.vote_average',
    'animes.is_anime','animes.vote_average','animes.newEpisodes','animes.subtitle')
    ->where('created_at', '>', Carbon::now()->subMonth(3))
        ->where('active', '=', 1)
        ->orderByDesc('created_at')
        ->limit(10)->get();



    return response()->json(['anime' => $movies->makeHidden(['casterslist','casters','seasons','overview','backdrop_path','preview_path','videos'
    ,'substitles','vote_count','popularity','runtime','release_date','imdb_external_id','hd','pinned','preview'])], 200);

}




// remove Network from  a movie
public function destroyNetworks($id)
{

    if ($id != null) {

        AnimeNetwork::find($id)->delete();
        $data = ['status' => 200, 'message' => 'successfully deleted',];
    } else {
        $data = [
            'status' => 400,
            'message' => 'could not be deleted',
        ];
    }

    return response()->json($data, $data['status']);

}


  // remove the cast of a movie from the database
  public function destroyCast($id)
  {

      if ($id != null) {

          $movie = AnimeCast::where('cast_id', '=', $id)->first();
          $movie->delete();
          $data = ['status' => 200, 'message' => 'successfully deleted',];
      } else {
          $data = [
              'status' => 400,
              'message' => 'could not be deleted',
          ];
      }

      return response()->json($data, $data['status']);

  }

        public function relateds($anime)
        {


            $moviesGenre = Anime::withOnly('genres.genre')
            ->where('id', '=', $anime)
            ->select(['animes.id','name','poster_path',
            'vote_average','newEpisodes','subtitle'])
            ->where('active', '=', 1)
            ->limit(1)
            ->orderByDesc('animes.created_at')
            ->get();



            foreach( $moviesGenre as $movie){


               $genre = $movie->genres[0]->genre_id;

           }



           $movies = DB::table('animes')->join('anime_genres', 'animes.id', '=', 'anime_genres.anime_id')
            ->where('anime_genres.genre_id', '=', $genre)
            ->where('anime_genres.anime_id', '!=', $anime)
            ->select(['animes.id','name','poster_path',
            'vote_average','newEpisodes','subtitle','is_anime'])
             ->where('active', '=', 1)
            ->limit(10)
            ->orderByDesc('animes.created_at')
            ->get();


          return response()->json(['relateds' => $movies], 200);

        }


public function newEpisodes()
{
    $order = 'desc';

    $animes = Anime::where('active', '=', 1)->join('anime_seasons', 'anime_seasons.anime_id', '=', 'animes.id')
    ->join('anime_episodes', 'anime_episodes.anime_season_id', '=', 'anime_seasons.id')
    ->join('anime_videos', 'anime_videos.anime_episode_id', '=', 'anime_episodes.id')
    ->orderBy('anime_videos.updated_at', $order)->orderBy('anime_videos.anime_episode_id', $order)->select('anime_videos.anime_episode_id','animes.id'
    ,'animes.name','anime_episodes.still_path','anime_episodes.anime_season_id','anime_episodes.name as episode_name','anime_videos.link','anime_videos.server','anime_videos.lang'
    ,'anime_videos.embed','anime_videos.youtubelink','anime_videos.hls','anime_seasons.name as seasons_name','anime_seasons.season_number','anime_episodes.vote_average'
    ,'animes.premuim','animes.tmdb_id','anime_episodes.episode_number','animes.poster_path',
    'anime_episodes.hasrecap',
    'anime_episodes.skiprecap_start_in','anime_videos.supported_hosts'
    ,'anime_videos.drmuuid','anime_videos.drmlicenceuri','anime_videos.drm'
    )->addSelect(DB::raw("'anime' as type"))->limit(10)->get()->unique('anime_episode_id')->makeHidden(['seasons','episodes','casterslist','']);


    $newEpisodes = [];
    foreach ($animes as $item) {
        array_push($newEpisodes, $item);
    }

    return response()->json(['latest_episodes' => $newEpisodes], 200);

}



public function animesEpisodesAll()
{
    $order = 'desc';

    $animes = Anime::where('active', '=', 1)->join('anime_seasons', 'anime_seasons.anime_id', '=', 'animes.id')
    ->join('anime_episodes', 'anime_episodes.anime_season_id', '=', 'anime_seasons.id')
    ->join('anime_videos', 'anime_videos.anime_episode_id', '=', 'anime_episodes.id')
    ->orderBy('anime_videos.updated_at', $order)->orderBy('anime_videos.anime_episode_id', $order)->select('anime_videos.anime_episode_id','animes.id','animes.tmdb_id as serieTmdb'
    ,'animes.name','anime_episodes.still_path','anime_episodes.anime_season_id','anime_episodes.name as episode_name','anime_videos.link','anime_videos.server','anime_videos.lang'
    ,'anime_videos.embed','anime_videos.youtubelink','anime_videos.hls','anime_seasons.name as seasons_name','anime_seasons.season_number','anime_episodes.vote_average'
    ,'animes.premuim','animes.tmdb_id','anime_episodes.episode_number','animes.poster_path',
    'anime_episodes.hasrecap','anime_episodes.skiprecap_start_in','anime_videos.supported_hosts','animes.is_anime'
    ,'anime_videos.drmuuid','anime_videos.drmlicenceuri','anime_videos.drm'
    )->addSelect(DB::raw("'anime' as type"))->groupBy('anime_episode_id')->paginate(12);



    $animes->setCollection($animes->getCollection()->makeHidden(['seasons','episodes','casterslist']));
    return $animes;


    return response()->json(['latest_episodes' => $animes], 200);

}



public function showEpisodeFromNotifcation($id)
{

    $order = 'desc';
    $animes = Anime::where('active', '=', 1)->join('anime_seasons', 'anime_seasons.anime_id', '=', 'animes.id')
    ->join('anime_episodes', 'anime_episodes.anime_season_id', '=', 'anime_seasons.id')
    ->join('anime_videos', 'anime_videos.anime_episode_id', '=', 'anime_episodes.id')
    ->orderBy('anime_videos.updated_at', $order)->orderBy('anime_videos.anime_episode_id', $order)->select('anime_videos.anime_episode_id','animes.id'
    ,'animes.name','anime_episodes.still_path','anime_episodes.anime_season_id','anime_episodes.name as episode_name','anime_videos.link','anime_videos.server','anime_videos.lang'
    ,'anime_videos.embed','anime_videos.hls','anime_seasons.name as seasons_name','anime_seasons.season_number','anime_episodes.vote_average'
    ,'animes.premuim','animes.tmdb_id','anime_episodes.episode_number','animes.poster_path',
    'anime_episodes.hasrecap','anime_episodes.skiprecap_start_in',
    'anime_videos.supported_hosts','animes.imdb_external_id','anime_videos.header'
    ,'anime_videos.useragent','animes.imdb_external_id','anime_videos.drmuuid','anime_videos.drmlicenceuri','anime_videos.drm'
    )->addSelect(DB::raw("'anime' as type"))->where('anime_episodes.id', '=', $id)->limit(1)
    ->get()->makeHidden(['seasons','episodes','casterslist']);


    return response()->json(['latest_episodes' => $animes], 200);

}

public function detail($id)
    {

        $anime = Anime::with(['casters.cast' => function ($query) {
            $query->select('id', 'name','original_name','profile_path');
        },'seasons.episodes.videos'])->where('id', '=', $id)->first()->makeHidden(['casters','networks']);

        $anime->increment('views',1);

        $animesRelacionados = Anime::query()
        ->when(isset($generosDoFilme) && is_array($generosDoFilme), function ($query) use ($generosDoFilme) {
            $query->whereHas('genres.genre', function ($query) use ($generosDoFilme) {
                $query->whereIn('name', $generosDoFilme);
            });
        })
        ->where('id', '!=', $id) // Para garantir que não traga a mesma série atual
        ->limit(12)
        ->where('active', '=', 1)
        ->orderByRaw('RAND()')
        ->get();

        return view('anime', ['anime' => $anime, 'animesRelacionados' => $animesRelacionados]);

        // return response()->json($seriesRelacionadas);
    }

    public function markInactiveAnimes()
    {
        // Encontra animes que têm vídeos nas duas primeiras temporadas
        $animesWithVideosInFirstTwoSeasons = Anime::whereHas('seasons', function ($query) {
            $query->where('season_number', 1)
                  ->has('episodes.videos');
        }, '>=', 1)->pluck('id');

        // Marca como inativos os animes que não estão nessa lista
        $animesToUpdate = Anime::whereNotIn('id', $animesWithVideosInFirstTwoSeasons)
                               ->where('active', 1)
                               ->update(['active' => 0]);

        return response()->json([
            'message' => "Total de animes atualizados como inativos: $animesToUpdate"
        ]);
    }

    public function generateAnimesLinks()
    {
        $series = Anime::with(['seasons.episodes.videos'])->get();

        $content = '';

        foreach ($series as $serie) {
            // Incluindo TMDB_ID ou 'Unknown' se não estiver definido
            $tmdbId = $serie->tmdb_id ?? 'Unknown';

            foreach ($serie->seasons as $season) {
                foreach ($season->episodes as $episode) {
                    foreach ($episode->videos as $video) {
                        if ($video->link) {
                            $episodeNumberFormatted = str_pad($episode->episode_number, 2, '0', STR_PAD_LEFT);

                            // Dividindo a URL em componentes
                            $urlComponents = parse_url($video->link);

                            // Dividindo o caminho em segmentos
                            $pathSegments = explode('/', $urlComponents['path']);

                            // Verificando cada segmento antes de codificar
                            $encodedPathSegments = array_map(function($segment) {
                                return strpos($segment, '%20') !== false ? $segment : rawurlencode($segment);
                            }, $pathSegments);

                            $encodedPath = implode('/', $encodedPathSegments);

                            // Reconstruindo a URL com o caminho codificado
                            $encodedLink = $urlComponents['scheme'] . '://' . $urlComponents['host'] . $encodedPath;

                            // Adicionando TMDB_ID no início de cada linha
                            $content .= "{$tmdbId} | {$serie->name} S{$season->season_number}E{$episodeNumberFormatted} | {$encodedLink}" . PHP_EOL;
                        }
                    }
                }
            }
        }

        Storage::disk('local')->put('animes_links.txt', $content);

        return response()->download(storage_path('app/animes_links.txt'));
    }


}
