<?php

namespace App\Http\Controllers;

use App\Episode;
use App\Livetv;
use App\Movie;
use App\Serie;
use App\Anime;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AdminController extends Controller
{

    const FEATURED = "featured";
    const CREATED_AT = "created_at";
    const VIEWS = "views";




    public function store(Request $request)
    {
        if ($request->hasFile('image')) {
            $filename = Storage::disk('movies')->put('', $request->image);
            $data = ['status' => 200, 'image_path' => $request->root() . '/api/movies/image/' . $filename, 'message' => 'successfully uploaded'];
        } else {
            $data = ['status' => 400, 'message' => 'could not be uploaded'];
        }

        return response()->json($data, $data['status']);

    }




    /**
     * Construct Controller
     */
    public function __construct()
    {
        $this->middleware('admin');
    }

    // navigation routes for the admin panel

    public function updateWebPanel(Request $request)
    {


        $user = Auth::user();

        $this->validate($request, [
            'webpanelversion' => 'required'
        ]);


        DB::table('settings')
            ->update(
                array( "webpanelversion" => $request->webpanelversion)

            );

        return response()->json("Success", 204);

    }



    public function set(Request $request)
    {


        $user = Auth::user();

        $this->validate($request, [
            'startapp_banner' => 'required'
        ]);


        $s = DB::table('settings')
            ->update(
                array("startapp_banner" => $request->startapp_banner)

            );

            if ($s != null) {

                $data = ['status' => 200, 'message' => 'successfully removed',];
            } else {
                $data = ['status' => 400, 'message' => 'could not be deleted',];
            }

             return response()->json($data, 200);

    }


    public function home()
    {
        return view('admin.home');
    }

    public function index()
    {
        return view('admin.index');
    }



    public function comments()
    {
        return view('admin.comments');
    }

    public function users()
    {
        return view('admin.users');
    }

    public function movies()
    {
        return view('admin.movies');
    }

    public function series()
    {
        return view('admin.series');
    }


    public function animes()
    {
        return view('admin.animes');
    }


    public function streaming()
    {
        return view('admin.streaming');
    }

    public function servers()
    {
        return view('admin.servers');
    }



    public function headers()
    {
        return view('admin.headers');
    }

    public function genres()
    {
        return view('admin.genres');
    }


    public function networks()
    {
        return view('admin.networks');
    }

    public function casters()
    {
        return view('admin.casters');
    }



    public function notifications()
    {
        return view('admin.notifications');
    }

    public function settings()
    {
        return view('admin.settings');
    }

    public function account()
    {
        return view('admin.account');
    }


    public function reports()
    {
        return view('admin.reports');
    }



    public function suggestions()
    {
        return view('admin.suggestions');
    }


    public function ads()
    {
        return view('admin.ads');
    }


    public function upcomings()
    {
        return view('admin.upcomings');
    }

    public function plans()
    {
        return view('admin.plans');
    }



    public function categories()
    {
        return view('admin.categories');
    }

    public function previews()
    {
        return view('admin.previews');
    }


    public function featured()

    {


        return view('admin.featured');

    }

    public function certifications()

    {


        return view('admin.certifications');

    }



    public function topcontentmovies()

    {

        $movies = Movie::withOnly(['genres.genre','substitles','networks.network'])
        ->select('movies.id','movies.title','movies.poster_path','movies.views')->where('active', '=', 1)
        ->orderByDesc('views')->limit(5)->get();


        return response()
            ->json($movies->makeHidden(['videos','casterslist','casters','genres','genreslist','substitles'])
            ->toArray(), 200);

    }





    public function topcontentseries()

    {

        $movies = Serie::withOnly([
            'genres.genre','networks.network'])->select('series.id','series.name','series.poster_path','series.views')->where('active', '=', 1)
        ->orderByDesc('views')->limit(5)->get();

        return response()
            ->json($movies->makeHidden(['seasons','casterslist','casters','genres','genreslist','substitles'])->toArray(), 200);

    }


    public function topcontentanimes()

    {

        $movies = Anime::withOnly([
            'genres.genre','networks.network'])->select('animes.id','animes.name','animes.poster_path','animes.views')->where('active', '=', 1)
        ->orderByDesc('views')->take(5)->get();

        return response()
            ->json($movies->makeHidden(['seasons','casterslist','casters','genres','genreslist','substitles'])->toArray(), 200);

    }



    // most viewed metrics

    public function topMovies()
    {
        $movies = Movie::orderBy(self::VIEWS, 'desc')->limit(5)->get();

        return response()->json($movies, 200);
    }

    public function topSeries()
    {
        $series = anime::all()->makeHidden(['seasons', 'genres'])->sortByDesc(self::VIEWS);

        if ($series->count() > 10) {
            $series = $series->take(10);
        }

        return response()->json($animes, 200);
    }

    public function topEpisodes()
    {
        $episodes = Episode::orderBy(self::VIEWS, 'desc')->limit(10)->get();

        return response()->json($episodes, 200);
    }

    public function topLivetv()
    {
        $livetv = Livetv::orderBy(self::VIEWS, 'desc')->limit(10)->get();

        return response()->json($livetv, 200);
    }

    public function topUsers()
    {
        $users = User::orderBy('id', 'desc')->limit(10)->get();

        return response()->json($users, 200);
    }



    public function moviesCountViews()
    {
        return response()->json([
            'status' => 'success',
            'data' => [

                'count' => [
                    'movies' => DB::table('movies')->sum('views'),
                    'series' => DB::table('series')->sum('views'),
                    'animes' => DB::table('animes')->sum('views'),
                    'tvs' => DB::table('livetvs')->sum('views'),

                ],

            ]
        ]);
    }


    public function moviesInactiveCount()
    {
        return response()->json([
            'status' => 'success',
            'data' => [

                'count' => [
                    'movies' => DB::table('movies')->where('active', '=', 1)->count(),
                    'series' => DB::table('series')->where('active', '=', 0)->count(),
                    'animes' => DB::table('animes')->where('active', '=', 0)->where('active', '=', 0)->count(),
                    'tvs' => DB::table('livetvs')->where('active', '=', 0)->count()
                ],

            ]
        ]);
    }

    public function moviesCount()
    {
        return response()->json([
            'status' => 'success',
            'data' => [

                'count' => [
                    'reports' => DB::table('reports')->count(),
                    'movies' => DB::table('movies')->count(),
                    'series' => DB::table('series')->count(),
                    'animes' => DB::table('animes')->count(),
                    'tvs' => DB::table('livetvs')->count(),
                    'users' => DB::table('users')->count()
                ],

            ]
        ]);
    }

    public function updateMailSettings(Request $request){
        // some code
        $env_update = $this->changeEnv([
            'MAIL_DRIVER'   => 'sendmail',
            'MAIL_HOST'   => $request->get('email_smtp_adress'),
            'MAIL_PORT'       => $request->get('email_smtp_port')
        ]);
        if($env_update){
            // Do something
        } else {
            // Do something else
        }
        // more code
    }

    protected function changeEnv($data = array()){
        if(count($data) > 0){

            // Read .env-file
            $env = file_get_contents(base_path() . '/.env');

            // Split string on every " " and write into array
            $env = preg_split('/\s+/', $env);;

            // Loop through given data
            foreach((array)$data as $key => $value){

                // Loop through .env-data
                foreach($env as $env_key => $env_value){

                    // Turn the value into an array and stop after the first split
                    // So it's not possible to split e.g. the App-Key by accident
                    $entry = explode("=", $env_value, 2);

                    // Check, if new key fits the actual .env-key
                    if($entry[0] == $key){
                        // If yes, overwrite it with the new one
                        $env[$env_key] = $key . "=" . $value;
                    } else {
                        // If not, keep the old one
                        $env[$env_key] = $env_value;
                    }
                }
            }

            // Turn the array back to an String
            $env = implode("\n", $env);

            // And overwrite the .env with the new data
            file_put_contents(base_path() . '/.env', $env);

            return true;
        } else {
            return false;
        }
    }

    /**
     * Ferramenta de diagnóstico para limites de dispositivos
     * Adicione este método ao AdminController
     */
    public function deviceLimitDiagnostic(Request $request)
    {
        if (!Auth::check() || !Auth::user()->role === 'admin') {
            abort(403, 'Acesso não autorizado');
        }

        $userId = $request->input('user_id');
        $email = $request->input('email');
        $user = null;

        if ($userId) {
            $user = User::find($userId);
        } elseif ($email) {
            $user = User::where('email', $email)->first();
        }

        if (!$user) {
            $users = User::orderBy('created_at', 'desc')
                ->take(100)
                ->get(['id', 'name', 'email', 'premuim', 'tv_access']);

            return view('admin.device_diagnostic', [
                'users' => $users,
                'diagnosticResult' => null
            ]);
        }

        $userDeviceLimit = $user->device_limit ?? 'não definido';
        $subscription = $user->subscription;
        $subscriptionLimit = $subscription ? ($subscription->device_limit ?? 'não definido') : 'sem assinatura';
        $plan = $subscription ? $subscription->plan : null;
        $planLimit = $plan ? ($plan->device_limit ?? 'não definido') : 'sem plano';
        $activeDevices = $user->devices()->count();

        // Obter o limite efetivo calculado pelo DeviceController
        $deviceController = app()->make(DeviceController::class);
        $effectiveLimit = $deviceController->getUserDeviceLimit($user);

        $diagnosticResult = [
            'user' => $user,
            'user_device_limit' => $userDeviceLimit,
            'subscription_limit' => $subscriptionLimit,
            'plan_limit' => $planLimit,
            'active_devices' => $activeDevices,
            'effective_limit' => $effectiveLimit,
            'config_default' => config('app.default_device_limit', 1)
        ];

        return view('admin.device_diagnostic', [
            'users' => [],
            'diagnosticResult' => $diagnosticResult
        ]);
    }

    /**
     * Atualiza o limite de dispositivos para um usuário
     * Adicione este método ao AdminController
     */
    public function updateUserDeviceLimit(Request $request, $userId)
    {
        // Verificar se é admin
        if (!Auth::check() || !Auth::user()->role === 'admin') {
            abort(403, 'Acesso não autorizado');
        }

        $user = User::findOrFail($userId);

        $request->validate([
            'device_limit' => 'required|integer|min:1|max:10',
            'target' => 'required|in:user,subscription,plan',
        ]);

        $deviceLimit = $request->device_limit;
        $target = $request->target;

        try {
            switch ($target) {
                case 'user':
                    // Atualizar diretamente no usuário
                    $user->device_limit = $deviceLimit;
                    $user->save();
                    break;

                case 'subscription':
                    // Atualizar na assinatura ativa
                    $subscription = $user->subscription;
                    if (!$subscription) {
                        throw new \Exception('Usuário não possui assinatura ativa');
                    }
                    $subscription->device_limit = $deviceLimit;
                    $subscription->save();
                    break;

                case 'plan':
                    // Atualizar no plano (afetará todos os usuários com este plano)
                    $subscription = $user->subscription;
                    if (!$subscription || !$subscription->plan) {
                        throw new \Exception('Usuário não possui plano associado');
                    }
                    $plan = $subscription->plan;
                    $plan->device_limit = $deviceLimit;
                    $plan->save();
                    break;
            }

            // Limpar cache para garantir atualização
            cache()->forget("user_{$user->id}_device_limit");
            return back()->with('success', 'Limite de dispositivos atualizado com sucesso');
        } catch (\Exception $e) {
            return back()->with('error', 'Erro ao atualizar limite: ' . $e->getMessage());
        }
    }

    /**
     * Exibe a página de gerenciamento de dispositivos
     */
    public function devicesManagement()
    {
        $devices = Device::with('user')
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Usuários com mais dispositivos
        $usersWithMostDevices = User::withCount('devices')
            ->orderBy('devices_count', 'desc')
            ->limit(10)
            ->get();

        return view('admin.devices.index', compact('devices', 'usersWithMostDevices'));
    }

    /**
     * Remove um dispositivo específico
     */
    public function removeDevice($id)
    {
        try {
            $device = Device::findOrFail($id);
            $device->delete();

            return response()->json([
                'status' => 200,
                'message' => 'Dispositivo removido com sucesso'
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao remover dispositivo: ' . $e->getMessage());

            return response()->json([
                'status' => 500,
                'message' => 'Erro ao remover dispositivo'
            ], 500);
        }
    }

    /**
     * Define o limite de dispositivos para um usuário específico
     */
    public function setUserDeviceLimit(Request $request, $userId)
    {
        $validator = Validator::make($request->all(), [
            'device_limit' => 'required|integer|min:1'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 422,
                'message' => 'Erro de validação',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = User::findOrFail($userId);
            $user->device_limit = $request->device_limit;
            $user->save();

            // Limpar cache
            $cacheKey = "user_{$user->id}_device_limit";
            cache()->forget($cacheKey);

            return response()->json([
                'status' => 200,
                'message' => 'Limite de dispositivos atualizado com sucesso'
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao definir limite de dispositivos: ' . $e->getMessage());

            return response()->json([
                'status' => 500,
                'message' => 'Erro ao definir limite de dispositivos'
            ], 500);
        }
    }

    /**
     * Obtém os dispositivos de um usuário específico
     */
    public function getUserDevices($userId)
    {
        try {
            $user = User::findOrFail($userId);
            $devices = $user->devices()->orderBy('last_active', 'desc')->get();
            $deviceLimit = app(DeviceService::class)->getUserDeviceLimit($user);

            return response()->json([
                'status' => 200,
                'devices' => $devices,
                'device_limit' => $deviceLimit,
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'custom_device_limit' => $user->device_limit
                ]
            ]);
        } catch (\Exception $e) {
            Log::error('Erro ao obter dispositivos do usuário: ' . $e->getMessage());

            return response()->json([
                'status' => 500,
                'message' => 'Erro ao obter dispositivos do usuário'
            ], 500);
        }
    }

    /**
     * Retorna estatísticas sobre os tipos de dispositivos
     */
    public function getDeviceStats()
    {
        $stats = [
            'deviceTypes' => [
                'mobile' => Device::where('device_type', 'mobile')->count(),
                'tablet' => Device::where('device_type', 'tablet')->count(),
                'desktop' => Device::where('device_type', 'desktop')->count(),
                'other' => Device::whereNull('device_type')
                    ->orWhereNotIn('device_type', ['mobile', 'tablet', 'desktop'])
                    ->count(),
            ],
            'totalDevices' => Device::count(),
            'activeDevices' => Device::where('last_active', '>=', now()->subDays(30))->count(),
            'currentDevices' => Device::where('is_current', true)->count()
        ];

        return response()->json($stats);
    }
}
