<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Setting;

class AdminAuthController extends Controller
{
    /**
     * Construtor para garantir que não use o middleware de sessão única
     */
    public function __construct()
    {
        // Não aplicar nenhum middleware neste controller
    }

    /**
     * Exibe o formulário de login para o admin
     *
     * @return \Illuminate\View\View
     */
    public function showLoginForm()
    {
        // Se já estiver logado e for admin, redireciona para o dashboard
        if (Auth::check() && Auth::user()->role === 'admin') {
            return redirect()->route('admin');
        }

        $settings = Setting::first();
        return view('admin.auth.login', compact('settings'));
    }

    /**
     * Realiza o login do admin
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required'
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput($request->except('password'));
        }

        $credentials = $request->only('email', 'password');

        // Adicionar 'remember me' baseado no checkbox
        $remember = $request->has('remember');

        if (Auth::attempt($credentials, $remember)) {
            if (Auth::user()->role === 'admin') {
                // Regenerar sessão para evitar fixação de sessão
                $request->session()->regenerate();

                return redirect()->intended('/admin');
            } else {
                Auth::logout();
                return redirect()->route('admin.login')
                    ->withErrors(['email' => 'Acesso restrito a administradores.']);
            }
        }

        return redirect()->back()
            ->withErrors(['email' => 'Credenciais inválidas.'])
            ->withInput($request->except('password'));
    }

    /**
     * Realiza o logout do admin
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function logout()
    {
        Auth::logout();
        return redirect()->route('admin.login');
    }
}
