<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class EpgHelper
{
    /**
     * URL base da API EPG
     */
    protected $baseUrl = 'https://iptv-org.github.io/epg';

    /**
     * Tempo de cache para dados EPG em minutos
     */
    protected $cacheTime = 60;

    /**
     * Obtém informações de programação para um canal específico
     *
     * @param string $channelName Nome do canal (ex: 'National Geographic')
     * @param string $channelId ID do canal na API EPG (opcional)
     * @return array Dados da programação
     */
    public function getChannelPrograms($channelName, $channelId = null)
    {
        $cacheKey = 'epg_' . md5($channelName . ($channelId ?? ''));

        Log::info("Buscando programação para canal", [
            'channelName' => $channelName,
            'channelId' => $channelId,
            'cacheKey' => $cacheKey
        ]);

        try {
            // Verificar cache primeiro
            if (Cache::has($cacheKey)) {
                Log::info("Dados EPG encontrados no cache");
                return Cache::get($cacheKey);
            }

            // Se temos um ID de canal específico, usamos ele
            if ($channelId) {
                $data = $this->fetchProgramsByChannelId($channelId);
            } else {
                // Caso contrário, tentamos buscar pelo nome
                $data = $this->fetchProgramsByChannelName($channelName);
            }

            // Armazenar no cache
            Cache::put($cacheKey, $data, now()->addMinutes($this->cacheTime));

            return $data;
        } catch (\Exception $e) {
            Log::error('Erro ao buscar programação: ' . $e->getMessage(), [
                'channelName' => $channelName,
                'channelId' => $channelId,
                'exception' => $e
            ]);

            // Se falhar, tentamos buscar do cache mesmo expirado como fallback
            if (Cache::has($cacheKey)) {
                Log::info("Usando dados EPG expirados do cache como fallback");
                return Cache::get($cacheKey);
            }

            return $this->getDefaultEpgData();
        }
    }

    /**
     * Busca programas pelo ID do canal
     */
    protected function fetchProgramsByChannelId($channelId)
    {
        try {
            Log::info("Buscando programas pelo ID do canal: {$channelId}");

            // Para iptv-org/epg, precisamos construir a URL manualmente
            // Exemplo: https://iptv-org.github.io/epg/guides/br/mi.tv.epg.xml

            // Como estamos usando uma adaptação simplificada, vamos simular uma resposta
            // Em implementação real, aqui faríamos o parsing do XML

            $now = Carbon::now();

            // Simular dados para fins de demonstração
            $response = [
                'data' => [
                    [
                        'title' => 'Programa Atual (' . $channelId . ')',
                        'description' => 'Descrição do programa atual via ID',
                        'start_time' => $now->copy()->subHour()->toIso8601String(),
                        'end_time' => $now->copy()->addHour()->toIso8601String(),
                    ],
                    [
                        'title' => 'Próximo Programa',
                        'description' => 'Descrição do próximo programa',
                        'start_time' => $now->copy()->addHour()->toIso8601String(),
                        'end_time' => $now->copy()->addHours(2)->toIso8601String(),
                    ],
                    [
                        'title' => 'Programa Seguinte',
                        'description' => 'Descrição do programa seguinte',
                        'start_time' => $now->copy()->addHours(2)->toIso8601String(),
                        'end_time' => $now->copy()->addHours(3)->toIso8601String(),
                    ]
                ]
            ];

            Log::info("Dados recuperados por ID", ['count' => count($response['data'])]);

            return $this->formatEpgData($response);
        } catch (\Exception $e) {
            Log::error('Erro ao buscar dados EPG por ID: ' . $e->getMessage(), [
                'channelId' => $channelId,
                'exception' => $e
            ]);
            return $this->getDefaultEpgData();
        }
    }

    /**
     * Busca programas pelo nome do canal
     */
    protected function fetchProgramsByChannelName($channelName)
    {
        try {
            Log::info("Buscando programas pelo nome do canal: {$channelName}");

            // Primeiro, tentamos localizar o canal pelo nome
            // Em implementação real, isso seria feito consultando um índice de canais

            // Vamos simular uma resposta para este exemplo
            $now = Carbon::now();

            $response = [
                'data' => [
                    [
                        'title' => 'Programa Atual no ' . $channelName,
                        'description' => 'Este é o programa que está passando agora no canal ' . $channelName,
                        'start_time' => $now->copy()->subHour()->toIso8601String(),
                        'end_time' => $now->copy()->addHour()->toIso8601String(),
                    ],
                    [
                        'title' => 'Próximo Programa',
                        'description' => 'Este é o próximo programa a ser exibido',
                        'start_time' => $now->copy()->addHour()->toIso8601String(),
                        'end_time' => $now->copy()->addHours(2)->toIso8601String(),
                    ],
                    [
                        'title' => 'Programa da Noite',
                        'description' => 'Programa a ser exibido mais tarde',
                        'start_time' => $now->copy()->addHours(2)->toIso8601String(),
                        'end_time' => $now->copy()->addHours(4)->toIso8601String(),
                    ]
                ]
            ];

            Log::info("Dados recuperados por nome", ['count' => count($response['data'])]);

            return $this->formatEpgData($response);
        } catch (\Exception $e) {
            Log::error('Erro ao buscar dados EPG por nome: ' . $e->getMessage(), [
                'channelName' => $channelName,
                'exception' => $e
            ]);
            return $this->getDefaultEpgData();
        }
    }

    /**
     * Busca canais pelo nome
     */
    public function searchChannels($query)
    {
        try {
            Log::info("Buscando canais com a query: {$query}");

            // Em uma implementação real, faríamos uma busca em um índice de canais
            // Para este exemplo, vamos simular alguns resultados

            // Simular resultados de busca
            $results = [
                [
                    'id' => 'natgeo_br',
                    'name' => 'National Geographic Brasil',
                ],
                [
                    'id' => 'discovery_br',
                    'name' => 'Discovery Channel Brasil',
                ],
                [
                    'id' => 'history_br',
                    'name' => 'History Channel Brasil',
                ],
                [
                    'id' => 'globo_sp',
                    'name' => 'Globo São Paulo',
                ],
                [
                    'id' => 'sbt_sp',
                    'name' => 'SBT São Paulo',
                ]
            ];

            // Filtrar resultados com base na query
            $filteredResults = array_filter($results, function($channel) use ($query) {
                return stripos($channel['name'], $query) !== false;
            });

            Log::info("Canais encontrados", ['count' => count($filteredResults)]);

            return array_values($filteredResults); // Reindexar array
        } catch (\Exception $e) {
            Log::error('Erro ao buscar canais: ' . $e->getMessage(), [
                'query' => $query,
                'exception' => $e
            ]);
            return [];
        }
    }

    /**
     * Formata os dados da API EPG para um formato padrão
     */
    protected function formatEpgData($data)
    {
        $now = Carbon::now();
        $current = null;
        $next = null;
        $upcoming = [];

        // Organiza os programas em atual, próximo e futuros
        foreach ($data['data'] ?? [] as $program) {
            $startTime = Carbon::parse($program['start_time']);
            $endTime = Carbon::parse($program['end_time']);

            // Verifica se o programa está passando agora
            if ($startTime <= $now && $endTime >= $now) {
                $current = $this->formatProgram($program, $startTime, $endTime, $now);
            }
            // Verifica se é o próximo programa
            elseif ($startTime > $now && $next === null) {
                $next = $this->formatProgram($program, $startTime, $endTime);
            }
            // Adiciona aos próximos programas (limitado a 5)
            elseif ($startTime > $now && count($upcoming) < 5) {
                $upcoming[] = $this->formatProgram($program, $startTime, $endTime);
            }
        }

        return [
            'current' => $current ?? $this->getDefaultCurrentProgram(),
            'next' => $next,
            'upcoming' => $upcoming
        ];
    }

    /**
     * Formata um programa individual
     */
    protected function formatProgram($program, $startTime, $endTime, $now = null)
    {
        $formatted = [
            'title' => $program['title'] ?? 'Sem título',
            'description' => $program['description'] ?? 'Sem descrição disponível',
            'start' => $startTime->format('H:i'),
            'end' => $endTime->format('H:i'),
            'duration' => $startTime->diffInMinutes($endTime)
        ];

        // Calcula o progresso se for o programa atual
        if ($now) {
            $totalDuration = $startTime->diffInSeconds($endTime);
            $elapsed = $startTime->diffInSeconds($now);
            $formatted['progress'] = min(100, round(($elapsed / $totalDuration) * 100));
        }

        return $formatted;
    }

    /**
     * Programa padrão quando não há dados disponíveis
     */
    protected function getDefaultCurrentProgram()
    {
        $now = Carbon::now();
        return [
            'title' => 'Programação ao vivo',
            'description' => 'Informações de programação não disponíveis no momento.',
            'start' => $now->subHour()->format('H:i'),
            'end' => $now->addHours(2)->format('H:i'),
            'progress' => 50,
            'duration' => 120
        ];
    }

    /**
     * Retorna dados EPG padrão quando a API falha
     */
    protected function getDefaultEpgData()
    {
        return [
            'current' => $this->getDefaultCurrentProgram(),
            'next' => null,
            'upcoming' => []
        ];
    }
}
