<?php

namespace App;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Device extends Model
{
    use HasFactory;

    /**
     * Os atributos que podem ser preenchidos em massa.
     *
     * @var array
     */
    protected $fillable = [
        'user_id', 'name', 'serial_number', 'device_type',
        'ip_address', 'user_agent', 'last_active', 'is_current'
    ];

    /**
     * Os atributos que devem ser convertidos.
     *
     * @var array
     */
    protected $casts = [
        'last_active' => 'datetime',
        'is_current' => 'boolean',
    ];

    /**
     * Relacionamento com o usuário proprietário do dispositivo
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Retorna o nome do dispositivo ou um valor padrão
     */
    public function getDeviceNameAttribute()
    {
        // Se tiver o campo device_name, retorna ele
        if ($this->attributes['name'] ?? null) {
            return $this->attributes['name'];
        }

        // Determina o tipo de dispositivo para exibição
        $deviceType = $this->attributes['device_type'] ?? 'desconhecido';
        $deviceTypes = [
            'mobile' => 'Celular',
            'tablet' => 'Tablet',
            'desktop' => 'Computador',
        ];

        // Fallback para nome do dispositivo
        return $deviceTypes[$deviceType] ?? 'Dispositivo ' . $deviceType;
    }

    /**
     * Verifica se o dispositivo está ativo (usado nos últimos 30 dias)
     *
     * @return bool
     */
    public function isActive()
    {
        return $this->last_active && $this->last_active->greaterThan(now()->subDays(30));
    }

    /**
     * Marca este dispositivo como atual e todos os outros como inativos
     */
    public function markAsCurrent()
    {
        if ($this->user) {
            // Atualiza todos os dispositivos do usuário para não serem mais o atual
            $this->user->devices()->where('id', '!=', $this->id)->update(['is_current' => false]);

            // Define este como atual
            $this->update([
                'is_current' => true,
                'last_active' => now()
            ]);
        }
    }
}
