<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Carbon\Carbon;

class TestPaggueIntegration extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'paggue:test {action=auth : The action to test (auth|billing)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test the Paggue integration';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $action = $this->argument('action');

        $this->info('Testando integração com Paggue - Ação: ' . $action);

        try {
            if ($action === 'auth') {
                $this->testAuth();
            } elseif ($action === 'billing') {
                $this->testBilling();
            } else {
                $this->error('Ação inválida. Use "auth" ou "billing".');
                return 1;
            }
        } catch (\Exception $e) {
            $this->error('Erro: ' . $e->getMessage());
            $this->line('Stack trace:');
            $this->line($e->getTraceAsString());
            return 1;
        }

        return 0;
    }

    /**
     * Testa a autenticação com a API da Paggue
     */
    private function testAuth()
    {
        $this->info('Testando autenticação com a API da Paggue...');

        $clientKey = env('PAGGUE_CLIENT_KEY', '');
        $clientSecret = env('PAGGUE_CLIENT_SECRET', '');
        $companyId = env('PAGGUE_COMPANY_ID', '');

        if (empty($clientKey) || empty($clientSecret) || empty($companyId)) {
            $this->error('Credenciais da Paggue não configuradas no .env');
            return;
        }

        $this->line('> Client Key: ' . substr($clientKey, 0, 5) . '...');
        $this->line('> Client Secret: ' . substr($clientSecret, 0, 5) . '...');
        $this->line('> Company ID: ' . $companyId);

        $client = new Client();

        try {
            $this->line('Fazendo requisição para obter token...');
            $response = $client->post('https://ms.paggue.io/auth/v1/token', [
                'json' => [
                    'client_key' => $clientKey,
                    'client_secret' => $clientSecret
                ]
            ]);

            $data = json_decode($response->getBody(), true);

            if (isset($data['access_token'])) {
                $this->info('Autenticação bem-sucedida!');
                $this->line('> Access Token: ' . substr($data['access_token'], 0, 20) . '...');

                if (isset($data['expires_at'])) {
                    $expires = Carbon::parse($data['expires_at']);
                    $this->line('> Expira em: ' . $expires->diffForHumans());
                }

                if (isset($data['user']['companies'])) {
                    $companies = $data['user']['companies'];
                    $this->line('> Companies: ' . json_encode($companies));

                    // Verificar se o company_id configurado está entre os disponíveis
                    $companyIds = array_column($companies, 'id');
                    if (in_array((int)$companyId, $companyIds)) {
                        $this->info('Company ID configurado está correto!');
                    } else {
                        $this->warn('Company ID configurado não está na lista de empresas disponíveis!');
                        $this->line('IDs disponíveis: ' . implode(', ', $companyIds));
                    }
                }
            } else {
                $this->error('Autenticação falhou: resposta inválida');
                $this->line(json_encode($data));
            }
        } catch (RequestException $e) {
            $this->error('Erro na requisição: ' . $e->getMessage());

            if ($e->hasResponse()) {
                $response = $e->getResponse();
                $this->line('Status: ' . $response->getStatusCode());
                $this->line('Corpo da resposta: ' . $response->getBody());
            }
        }
    }

    /**
     * Testa a criação de uma cobrança de teste
     */
    private function testBilling()
    {
        $this->info('Testando criação de cobrança PIX...');

        // Primeiro autenticar para obter o token
        $clientKey = env('PAGGUE_CLIENT_KEY', '');
        $clientSecret = env('PAGGUE_CLIENT_SECRET', '');
        $companyId = env('PAGGUE_COMPANY_ID', '');

        if (empty($clientKey) || empty($clientSecret) || empty($companyId)) {
            $this->error('Credenciais da Paggue não configuradas no .env');
            return;
        }

        $client = new Client();

        try {
            // Obter token de autenticação
            $this->line('Obtendo token de autenticação...');
            $authResponse = $client->post('https://ms.paggue.io/auth/v1/token', [
                'json' => [
                    'client_key' => $clientKey,
                    'client_secret' => $clientSecret
                ]
            ]);

            $authData = json_decode($authResponse->getBody(), true);

            if (!isset($authData['access_token'])) {
                $this->error('Falha ao obter token de autenticação');
                return;
            }

            $token = $authData['access_token'];
            $this->info('Token obtido com sucesso!');

            // Criar uma cobrança de teste
            $this->line('Criando cobrança PIX de teste...');

            // ID externo único para evitar duplicação
            $externalId = 'teste_' . time() . '_' . rand(1000, 9999);

            $billingData = [
                'payer_name' => 'Teste de Integração CLI',
                'amount' => 100, // R$ 1,00 em centavos
                'external_id' => $externalId,
                'description' => 'Teste de integração via CLI',
                'meta' => [
                    'test' => true,
                    'command' => 'paggue:test',
                    'timestamp' => time()
                ]
            ];

            $response = $client->post('https://ms.paggue.io/cashin/api/billing_order', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $token,
                    'X-Company-ID' => $companyId,
                    'Content-Type' => 'application/json'
                ],
                'json' => $billingData
            ]);

            $responseData = json_decode($response->getBody(), true);

            if (isset($responseData['payment'])) {
                $this->info('Cobrança criada com sucesso!');
                $this->line('> Hash: ' . $responseData['hash']);
                $this->line('> External ID: ' . $responseData['external_id']);
                $this->line('> Status: ' . $responseData['status']);
                $this->line('> Valor: R$ ' . ($responseData['amount'] / 100));

                $qrCode = $responseData['payment'];

                // Mostrar parte do QR Code PIX
                if (strlen($qrCode) > 50) {
                    $this->line('> QR Code PIX: ' . substr($qrCode, 0, 50) . '...');
                } else {
                    $this->line('> QR Code PIX: ' . $qrCode);
                }

                // Consultar status da cobrança
                $this->line("\nConsultando status da cobrança...");

                try {
                    $statusResponse = $client->get('https://ms.paggue.io/cashin/api/billing_order/' . $responseData['hash'], [
                        'headers' => [
                            'Authorization' => 'Bearer ' . $token,
                            'X-Company-ID' => $companyId,
                            'Content-Type' => 'application/json'
                        ]
                    ]);

                    $statusData = json_decode($statusResponse->getBody(), true);
                    $this->line('Status atual: ' . ($statusData['status'] == 0 ? 'Pendente' : 'Pago'));
                    $this->line('Dados completos: ' . json_encode($statusData));
                } catch (\Exception $e) {
                    $this->error('Erro ao consultar status: ' . $e->getMessage());
                }
            } else {
                $this->error('Falha ao criar cobrança');
                $this->line(json_encode($responseData));
            }

        } catch (RequestException $e) {
            $this->error('Erro na requisição: ' . $e->getMessage());

            if ($e->hasResponse()) {
                $response = $e->getResponse();
                $this->line('Status: ' . $response->getStatusCode());
                $this->line('Corpo da resposta: ' . $response->getBody());
            }
        }
    }
}
