<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Movie;
use App\MovieVideo;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use GuzzleHttp\Client;

class ImportMovieLinks extends Command
{
    protected $signature = 'movies:import-links';
    protected $description = 'Importa ou atualiza links MP4 de filmes do arquivo filmes_links_com_tmdb.txt';

    private $supportedExtensions = ['mp4', 'mkv', 'avi', 'webm'];
    private $oldDomain = 'hubby.run';
    private $tmdb;
    private $apiKey = 'bbd5ddca4e045c351719281ae6a22974';

    public function __construct()
    {
        parent::__construct();
        $this->tmdb = new Client([
            'base_uri' => 'https://api.themoviedb.org/3/',
            'timeout'  => 5.0,
        ]);
    }

    public function handle()
    {
        $this->info('Iniciando importação de links...');

        $filePath = public_path('filmes_links_com_tmdb.txt');

        if (!file_exists($filePath)) {
            $this->error('Arquivo filmes_links_com_tmdb.txt não encontrado em ' . public_path());
            return 1;
        }

        $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $total = count($lines);
        $processed = 0;
        $updated = 0;
        $created = 0;
        $newMovies = 0;
        $errors = 0;

        $this->output->progressStart($total);

        DB::beginTransaction();
        try {
            foreach ($lines as $line) {
                if (str_starts_with($line, 'group,')) {
                    $this->output->progressAdvance();
                    continue;
                }

                try {
                    $result = $this->processLine($line);
                    if ($result === 'updated') {
                        $updated++;
                    } elseif ($result === 'created') {
                        $created++;
                    } elseif ($result === 'new_movie') {
                        $newMovies++;
                        $created++;
                    }
                    $processed++;
                } catch (\Exception $e) {
                    $errors++;
                    Log::error("Erro ao processar linha: {$line}", [
                        'error' => $e->getMessage(),
                        'trace' => $e->getTraceAsString()
                    ]);
                }

                $this->output->progressAdvance();
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Erro durante a transação: " . $e->getMessage());
            return 1;
        }

        $this->output->progressFinish();

        $this->info("\nProcessamento concluído!");
        $this->info("Total de linhas processadas: {$processed}");
        $this->info("Links atualizados: {$updated}");
        $this->info("Novos links criados: {$created}");
        $this->info("Novos filmes importados: {$newMovies}");
        $this->info("Erros encontrados: {$errors}");

        if ($errors > 0) {
            $this->info('Verifique o arquivo de log para mais detalhes sobre os erros.');
        }

        return 0;
    }

    private function fetchMovieFromTMDB($tmdbId)
    {
        try {
            $response = $this->tmdb->get("movie/{$tmdbId}", [
                'query' => [
                    'api_key' => $this->apiKey,
                    'language' => 'pt-BR',
                    'append_to_response' => 'credits,videos'
                ]
            ]);

            $movieData = json_decode($response->getBody(), true);

            return Movie::create([
                'tmdb_id' => $movieData['id'],
                'title' => $movieData['title'],
                'overview' => $movieData['overview'] ?? '',
                'poster_path' => $movieData['poster_path'] ?? '',
                'backdrop_path' => $movieData['backdrop_path'] ?? '',
                'vote_average' => $movieData['vote_average'] ?? 0,
                'vote_count' => $movieData['vote_count'] ?? 0,
                'popularity' => $movieData['popularity'] ?? 0,
                'runtime' => $movieData['runtime'] ?? 0,
                'release_date' => $movieData['release_date'] ?? null,
                'active' => 1,
                'premuim' => 0,
                'featured' => 0,
                'pinned' => 0
            ]);

        } catch (\Exception $e) {
            Log::error("Erro ao buscar filme na TMDB API: ", [
                'tmdb_id' => $tmdbId,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    private function processLine($line)
    {
        $parts = explode(',', $line);
        
        if (count($parts) !== 3) {
            throw new \Exception("Formato de linha inválido: {$line}");
        }

        [$extension, $filename_with_id, $url] = $parts;

        if (!in_array(strtolower($extension), $this->supportedExtensions)) {
            return null;
        }

        $parts = explode('_', $filename_with_id);
        if (count($parts) < 2) {
            throw new \Exception("Formato de nome inválido: {$filename_with_id}");
        }

        $tmdbId = end($parts);
        if (!is_numeric($tmdbId)) {
            throw new \Exception("TMDB ID inválido: {$tmdbId}");
        }

        array_pop($parts);
        $originalTitle = implode('_', $parts);

        // Verifica se é legendado
        $isLegendado = str_contains($originalTitle, '[L]');
        $serverType = $isLegendado ? 'Legendado' : 'Dublado';
        $lang = $isLegendado ? 'Legendado' : 'Dublado';

        // Busca ou cria o filme
        $movie = Movie::where('tmdb_id', $tmdbId)->first();
        $newMovie = false;

        if (!$movie) {
            $movie = $this->fetchMovieFromTMDB($tmdbId);
            $newMovie = true;
        }

        // Procura por um vídeo existente com mesmo servidor e domínio
        $existingVideo = MovieVideo::where('movie_id', $movie->id)
            ->where('server', $serverType)
            ->where('link', 'like', "%{$this->oldDomain}%")
            ->first();

        if ($existingVideo) {
            $existingVideo->update([
                'link' => $url,
                'status' => 1
            ]);
            return 'updated';
        }

        // Verifica se já existe um vídeo com esse link específico
        $duplicateVideo = MovieVideo::where('movie_id', $movie->id)
            ->where('link', $url)
            ->first();

        if ($duplicateVideo) {
            return null;
        }

        // Cria novo vídeo
        MovieVideo::create([
            'movie_id' => $movie->id,
            'server' => $serverType,
            'link' => $url,
            'lang' => $lang,
            'embed' => 0,
            'youtubelink' => 0,
            'supported_hosts' => 0,
            'hls' => 0,
            'status' => 1,
            'drm' => 0
        ]);

        return $newMovie ? 'new_movie' : 'created';
    }
}