<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Device;
use App\User;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class FixDeviceRegistration extends Command
{
    /**
     * Nome e assinatura do comando.
     *
     * @var string
     */
    protected $signature = 'devices:fix {--debug : Apenas mostra informações sem alterar} {--user_id= : ID específico de usuário}';

    /**
     * Descrição do comando.
     *
     * @var string
     */
    protected $description = 'Verifica e corrige problemas de registro de dispositivos';

    /**
     * Executa o comando.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Verificando configuração de dispositivos...');

        // Verificar se a tabela existe
        if (!Schema::hasTable('devices')) {
            $this->error('Tabela de dispositivos não encontrada!');
            return 1;
        }

        // Verificar colunas necessárias
        $requiredColumns = ['user_id', 'name', 'device_type', 'ip_address', 'user_agent', 'last_active', 'is_current', 'serial_number'];
        $missingColumns = [];

        foreach ($requiredColumns as $column) {
            if (!Schema::hasColumn('devices', $column)) {
                $missingColumns[] = $column;
            }
        }

        if (count($missingColumns) > 0) {
            $this->warn('Colunas ausentes na tabela de dispositivos: ' . implode(', ', $missingColumns));

            if (!$this->option('debug') && $this->confirm('Deseja adicionar estas colunas?')) {
                $this->info('Adicionando colunas ausentes...');

                Schema::table('devices', function ($table) use ($missingColumns) {
                    foreach ($missingColumns as $column) {
                        switch ($column) {
                            case 'device_type':
                            case 'ip_address':
                            case 'name':
                            case 'serial_number':
                                $table->string($column)->nullable();
                                break;
                            case 'user_agent':
                                $table->text($column)->nullable();
                                break;
                            case 'last_active':
                                $table->timestamp($column)->nullable();
                                break;
                            case 'is_current':
                                $table->boolean($column)->default(false);
                                break;
                        }
                    }
                });

                $this->info('Colunas adicionadas com sucesso!');
            }
        } else {
            $this->info('Todas as colunas necessárias estão presentes.');
        }

        // Verificar dispositivos duplicados ou inconsistentes
        if ($this->option('user_id')) {
            $users = User::where('id', $this->option('user_id'))->get();
        } else {
            $users = User::has('devices', '>', 1)->get();
        }

        $this->info('Verificando ' . $users->count() . ' usuários com múltiplos dispositivos...');

        foreach ($users as $user) {
            $devices = $user->devices()->get();
            $this->info("Usuário #{$user->id} ({$user->name}) tem {$devices->count()} dispositivos registrados.");

            // Verificar dispositivos duplicados
            $duplicates = $user->devices()
                ->select('ip_address', 'user_agent', DB::raw('COUNT(*) as count'))
                ->whereNotNull('ip_address')
                ->whereNotNull('user_agent')
                ->groupBy('ip_address', 'user_agent')
                ->having('count', '>', 1)
                ->get();

            if ($duplicates->count() > 0) {
                $this->warn("Encontrado(s) {$duplicates->count()} dispositivo(s) duplicado(s).");

                if (!$this->option('debug') && $this->confirm('Deseja remover duplicatas?')) {
                    foreach ($duplicates as $duplicate) {
                        // Manter apenas o dispositivo mais recente
                        $keepDeviceId = $user->devices()
                            ->where('ip_address', $duplicate->ip_address)
                            ->where('user_agent', $duplicate->user_agent)
                            ->latest('last_active')
                            ->value('id');

                        $user->devices()
                            ->where('ip_address', $duplicate->ip_address)
                            ->where('user_agent', $duplicate->user_agent)
                            ->where('id', '!=', $keepDeviceId)
                            ->delete();

                        $this->info("Duplicatas removidas, mantendo dispositivo #{$keepDeviceId}.");
                    }
                }
            }

            // Verificar e corrigir dispositivo atual
            $currentDevices = $user->devices()->where('is_current', true)->get();

            if ($currentDevices->count() == 0) {
                $this->warn("Usuário #{$user->id} não tem dispositivo marcado como atual.");

                if (!$this->option('debug') && $this->confirm('Definir o dispositivo mais recente como atual?')) {
                    $mostRecent = $user->devices()->latest('last_active')->first();
                    if ($mostRecent) {
                        $mostRecent->update(['is_current' => true]);
                        $this->info("Dispositivo #{$mostRecent->id} definido como atual.");
                    }
                }
            } elseif ($currentDevices->count() > 1) {
                $this->warn("Usuário #{$user->id} tem {$currentDevices->count()} dispositivos marcados como atual.");

                if (!$this->option('debug') && $this->confirm('Manter apenas o dispositivo mais recente como atual?')) {
                    $mostRecent = $user->devices()->where('is_current', true)->latest('last_active')->first();

                    $user->devices()
                        ->where('is_current', true)
                        ->where('id', '!=', $mostRecent->id)
                        ->update(['is_current' => false]);

                    $this->info("Dispositivo #{$mostRecent->id} mantido como atual, outros atualizados.");
                }
            } else {
                $this->info("Usuário #{$user->id} tem configuração de dispositivo atual correta.");
            }
        }

        $this->info('Verificação concluída!');
        return 0;
    }
}
