<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;

class ClearAllCache extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cache:clear-all';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Limpa todas as caches do Laravel (config, route, view, cache)';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Iniciando limpeza completa de cache...');

        $this->info('Limpando cache de configuração...');
        Artisan::call('config:clear');
        $this->info('✓ Cache de configuração limpa!');

        $this->info('Limpando cache de rotas...');
        Artisan::call('route:clear');
        $this->info('✓ Cache de rotas limpa!');

        $this->info('Limpando cache de views...');
        Artisan::call('view:clear');
        $this->info('✓ Cache de views limpa!');

        $this->info('Limpando cache de aplicação...');
        Artisan::call('cache:clear');
        $this->info('✓ Cache de aplicação limpa!');

        $this->info('Limpando arquivos compilados...');
        Artisan::call('clear-compiled');
        $this->info('✓ Arquivos compilados limpos!');

        // Verificar se o .env tem a configuração correta
        $this->checkEnvConfiguration();

        $this->info('Todas as caches foram limpas com sucesso!');
        $this->info('Para aplicar todas as alterações, reinicie o servidor web (Apache/Nginx).');

        return 0;
    }

    /**
     * Verifica se a configuração do .env está correta
     */
    private function checkEnvConfiguration()
    {
        $this->info('Verificando configuração do arquivo .env...');

        $envPath = base_path('.env');
        if (!file_exists($envPath)) {
            $this->error('Arquivo .env não encontrado!');
            return;
        }

        $envContent = file_get_contents($envPath);
        $lines = explode("\n", $envContent);

        $foundEnablePaggue = false;

        foreach ($lines as $line) {
            if (strpos($line, 'ENABLE_PAGGUE=') !== false) {
                $foundEnablePaggue = true;
                $value = trim(explode('=', $line)[1]);

                // Verificar se o valor está correto
                if ($value === 'true') {
                    $this->info('✓ Configuração ENABLE_PAGGUE=true encontrada');
                } else {
                    $this->warn('⚠ ENABLE_PAGGUE está definido como "' . $value . '" ao invés de "true"');
                }

                break;
            }
        }

        if (!$foundEnablePaggue) {
            $this->warn('⚠ ENABLE_PAGGUE não encontrado no arquivo .env');
        }

        // Verificar a presença das credenciais do Paggue
        $this->checkCredential('PAGGUE_CLIENT_KEY', $envContent);
        $this->checkCredential('PAGGUE_CLIENT_SECRET', $envContent);
        $this->checkCredential('PAGGUE_COMPANY_ID', $envContent);
    }

    /**
     * Verifica se uma credencial específica existe no .env
     */
    private function checkCredential($name, $envContent)
    {
        if (strpos($envContent, $name . '=') !== false) {
            $this->info('✓ ' . $name . ' encontrado');
        } else {
            $this->warn('⚠ ' . $name . ' não encontrado no arquivo .env');
        }
    }
}
